/* +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   Copyright (c) 2016-2021 The VES code team
   (see the PEOPLE-VES file at the root of this folder for a list of names)

   See http://www.ves-code.org for more information.

   This file is part of VES code module.

   The VES code module is free software: you can redistribute it and/or modify
   it under the terms of the GNU Lesser General Public License as published by
   the Free Software Foundation, either version 3 of the License, or
   (at your option) any later version.

   The VES code module is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU Lesser General Public License for more details.

   You should have received a copy of the GNU Lesser General Public License
   along with the VES code module.  If not, see <http://www.gnu.org/licenses/>.
+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++ */

#include "WaveletGrid.h"
#include "tools/Exception.h"

#include <algorithm>
#include <vector>

namespace PLMD {
namespace ves {

// returns the filter coefficients, at the moment simply a lookup table (calculated with python script)
// the coefficients are normalized such that their sum equals 2
// lowpass coefficients are for the scaling function, highpass for the actual wavelets
std::vector<double> WaveletGrid::getFilterCoefficients(const unsigned order, const bool lowpass, const Type type) {
  std::vector<double> h;
  if (type == Type::db) {
    switch(order) {
    case 4:
      h = { 3.2580342805129858252399799312116e-01,
            1.0109457150918295109676137144561e+00,
            8.9220013824675992175627925462322e-01,
            -3.9575026235645245631378230655173e-02,
            -2.6450716736904000825347793579567e-01,
            4.3616300474177409352005696518972e-02,
            4.6503601070981823162320267783798e-02,
            -1.4986989330361497746446630685568e-02
          };
      break;
    case 5:
      h = { 2.2641898258355833140065271891217e-01,
            8.5394354270502825787758638398373e-01,
            1.0243269442591973472644895082340e+00,
            1.9576696134780988134416190860065e-01,
            -3.4265671538293490661075679781788e-01,
            -4.5601131883547744994000794349631e-02,
            1.0970265864213350759381881971422e-01,
            -8.8268001083581463378502363070766e-03,
            -1.7791870101954165156454834573196e-02,
            4.7174279390678596629582486343679e-03
          };
      break;
    case 6:
      h = { 1.5774243200290161137644417976844e-01,
            6.9950381407523654075930608087219e-01,
            1.0622637598817388226990487964940e+00,
            4.4583132293003513435536433462403e-01,
            -3.1998659889212388574009082731209e-01,
            -1.8351806406029547158453851807280e-01,
            1.3788809297474488579915430364053e-01,
            3.8923209708329416234295905496765e-02,
            -4.4663748330189156909852954413509e-02,
            7.8325115229715134342741711392932e-04,
            6.7560623629278858393032436424619e-03,
            -1.5235338056025088393496957905882e-03
          };
      break;
    case 7:
      h = { 1.1009943074562379694558700293783e-01,
            5.6079128362552554953879280219553e-01,
            1.0311484916361983721344586228952e+00,
            6.6437248221107991774658785288921e-01,
            -2.0351382246269289488793674536282e-01,
            -3.1683501128066671181926494682557e-01,
            1.0084646500938766744415175935501e-01,
            1.1400344515974307479933003151018e-01,
            -5.3782452589690873945293958513503e-02,
            -2.3439941564206586560592526780056e-02,
            1.7749792379361528649184265304939e-02,
            6.0751499540213742267535934615807e-04,
            -2.5479047181873734281842835969201e-03,
            5.0022685312249071977463987792589e-04
          };
      break;
    case 8:
      h = { 7.6955622108152516025469935811998e-02,
            4.4246724715225010937658112197823e-01,
            9.5548615042774787120549717656104e-01,
            8.2781653242239250367617842130130e-01,
            -2.2385735333760647947087818465661e-02,
            -4.0165863278097890809448244908708e-01,
            6.6819409243975983261898399234724e-04,
            1.8207635684731554581716750362830e-01,
            -2.4563901045696636543658186724315e-02,
            -6.2350206650278829079159237380736e-02,
            1.9772159296701483782721098236834e-02,
            1.2368844819631834991424668146465e-02,
            -6.8877192568836120845077175545157e-03,
            -5.5400454895877760883027285387925e-04,
            9.5522971129925185115872032781681e-04,
            -1.6613726137322542978150274883831e-04
          };
      break;
    case 9:
      h = { 5.3850349589325587074117862584899e-02,
            3.4483430381395580033299097522104e-01,
            8.5534906435941493452190798052470e-01,
            9.2954571436629451763877796111046e-01,
            1.8836954950636755623527562875097e-01,
            -4.1475176180187667274168461517547e-01,
            -1.3695354902476589153970110146474e-01,
            2.1006834227901241551883515512600e-01,
            4.3452675461228981723671438430756e-02,
            -9.5647264120193975500683336576913e-02,
            3.5489281323312395211022618468633e-04,
            3.1624165852511730567808712066835e-02,
            -6.6796202262771747770742081229400e-03,
            -6.0549605750901376935435571624566e-03,
            2.6129672804944947123173548675368e-03,
            3.2581467135218489857700907563753e-04,
            -3.5632975902155574095264700673624e-04,
            5.5645514034309739761618301967516e-05
          };
      break;
    case 10:
      h = { 3.7717157592241430663637657971776e-02,
            2.6612218279384214358884719331400e-01,
            7.4557507148646740002817523418344e-01,
            9.7362811073364052827372461251798e-01,
            3.9763774176901656165483700533514e-01,
            -3.5333620179411384665257855886011e-01,
            -2.7710987872096615181050083265291e-01,
            1.8012744853339432427041799655854e-01,
            1.3160298710107026787241579768306e-01,
            -1.0096657119677993064144061463594e-01,
            -4.1659248087601748189445771686223e-02,
            4.6969814097397283303436665846675e-02,
            5.1004369678145037442229536850391e-03,
            -1.5179002335856551370030231851160e-02,
            1.9733253649632047886652852497491e-03,
            2.8176865901946854600645941957282e-03,
            -9.6994783985641268905103506980936e-04,
            -1.6470900609077862271692471729523e-04,
            1.3235436685110702815550143274237e-04,
            -1.8758415627500432871534175038875e-05
          };
      break;
    case 11:
      h = { 2.6437729433313437304997606247525e-02,
            2.0374153520190504962350530604454e-01,
            6.3625434846077988382262446975801e-01,
            9.6970753662635966740879212011350e-01,
            5.8260559778060438951285959774395e-01,
            -2.2949185235529601101234220550396e-01,
            -3.8782098279100135629349210830696e-01,
            9.3399738135533413863065277382702e-02,
            2.1186617983636121809176700026001e-01,
            -6.5732582904507175225461423906381e-02,
            -9.3958631797504951754795854412805e-02,
            4.4314509565959142589619546015456e-02,
            2.9473489598288812874038455902337e-02,
            -2.1729138108983869076551442844902e-02,
            -4.7246879281915600545804778676029e-03,
            6.9698350902376990032016657039549e-03,
            -4.3641620618785245374129933004781e-04,
            -1.2629255926067382322097731872645e-03,
            3.5235487790788363191354926051702e-04,
            7.6988477762888945306503363230632e-05,
            -4.8981264369811758688493669167130e-05,
            6.3558636358923973935959594872003e-06
          };
      break;
    case 12:
      h = { 1.8543533036448515277649917720737e-02,
            1.5495010900239064577377234854794e-01,
            5.3366075005107249662472668205737e-01,
            9.2941934664592906756297452375293e-01,
            7.2957365443751254208137879686547e-01,
            -6.3305694196084630331888831733522e-02,
            -4.4714385742734169548739942001703e-01,
            -3.3628948114692740201103760000478e-02,
            2.5806371934571636916899706193362e-01,
            7.5795761219506355491071225571886e-03,
            -1.3637561208887127861544286133721e-01,
            1.5342987147736677697174023649040e-02,
            5.8755309099664532190843146963743e-02,
            -1.7279779228315594180598324669518e-02,
            -1.8159669147498957303410449526382e-02,
            9.4914929220922782115987814677283e-03,
            3.1800108566658129814752609121342e-03,
            -3.0822835767046288281412191878417e-03,
            9.2562090856036967085468392113512e-06,
            5.4963843249920504071409643742641e-04,
            -1.2516371156799253796382265235820e-04,
            -3.4282722782480915088738904206167e-05,
            1.8069339114439924513354912560814e-05,
            -2.1624340181021667700003892342853e-06
          };
      break;
    case 13:
      h = { 1.3013782053568881749705532513417e-02,
            1.1718349488016345516516736324775e-01,
            4.4122943020981653106105113693047e-01,
            8.6416347207619259584987503330922e-01,
            8.3281561724389663314838117003092e-01,
            1.2301639369613130703395853515758e-01,
            -4.4543895786553527882389857950329e-01,
            -1.7617810218390753118633540452720e-01,
            2.5381750565052108026264932050253e-01,
            1.0316537133807110460104183857766e-01,
            -1.4963456864376814547945571121090e-01,
            -3.7460263683081813923525515974688e-02,
            7.9393209899753916514164586715196e-02,
            3.3657890397617833827537126012430e-03,
            -3.3702718379158690120700470060910e-02,
            5.5492912148471726374587831287499e-03,
            1.0260952934778005268823264373168e-02,
            -3.9059323261223279077913872470162e-03,
            -1.8606438898585472407204433054062e-03,
            1.3185082588273093311126427096269e-03,
            6.9652174801158197692090012775878e-05,
            -2.3352765515756047508440229609761e-04,
            4.3386003918454429126392440929649e-05,
            1.4767119831443566670332347146033e-05,
            -6.6473927339140924229086634933417e-06,
            7.3822444322983881586676402353220e-07
          };
      break;
    case 14:
      h = { 9.1374508518302227644269919437647e-03,
            8.8197087778947372282978278690280e-02,
            3.6041270508674011496808020638127e-01,
            7.8390652259161064385750705696410e-01,
            8.9263441660919018438846705976175e-01,
            3.0924705232209759309114360803505e-01,
            -3.8422563537411602574422886391403e-01,
            -3.0834597516857004118051577279402e-01,
            1.9572038841513531015614546504366e-01,
            1.9797436583701560897097238012066e-01,
            -1.2268078015402955616153235496313e-01,
            -1.0118550324745094470824113841445e-01,
            7.8117093102298987283838016537629e-02,
            3.8157473560976581938852802977635e-02,
            -4.2688533533420329468821563523306e-02,
            -7.9408791992275871984841018047518e-03,
            1.8087074833128147927086004642661e-02,
            -1.0553130151236629532401911646389e-03,
            -5.4442114973955908763247890647108e-03,
            1.5014579323156973956338022802015e-03,
            1.0012931187669377956078742286650e-03,
            -5.4706298624917549557056251074982e-04,
            -5.9081947567055248949751539155884e-05,
            9.7234313623187232453887185457120e-05,
            -1.4619021425907697688093174470669e-05,
            -6.2079802069149134127572231545145e-06,
            2.4395108649265364349071390920320e-06,
            -2.5273975810449469030721054638167e-07
          };
      break;
    case 15:
      h = { 6.4184610900818745565787892815024e-03,
            6.6105143008711259033916007865628e-02,
            2.9136174262291886227771442463563e-01,
            6.9668653280553660689378148163087e-01,
            9.1331770185222738156483046623180e-01,
            4.7942198331894603224867523749708e-01,
            -2.7323191454187872873049514055310e-01,
            -4.0854168599855755772409793280531e-01,
            9.2324037310499082464865239217033e-02,
            2.6890806178955328764601517832489e-01,
            -5.6096444852683710613483469842322e-02,
            -1.5714873480744581679680038632796e-01,
            4.7909516391089444764528337827869e-02,
            7.7471397590875790140074741429999e-02,
            -3.6440051225646634402988155443381e-02,
            -2.9429855183644534899478628631186e-02,
            2.1331881448689315322608806013704e-02,
            7.2139038913583038159504923214627e-03,
            -9.1750422042750877371641848867512e-03,
            -3.4189530802674857664497087839095e-04,
            2.7482751291413886653158815676079e-03,
            -5.2818381052845782188537304691067e-04,
            -5.0850204513539821755269132452781e-04,
            2.2047093037207382386355458692151e-04,
            3.6476384955660787364189329151642e-05,
            -3.9786489133705268018015993591874e-05,
            4.7559820825002058672348846657396e-06,
            2.5615231760346128998698331941108e-06,
            -8.9334206571767810297517772161191e-07,
            8.6738807723125058175416435962690e-08
          };
      break;
    case 16:
      h = { 4.5102394860307721408121395256785e-03,
            4.9366963027983538914167382927189e-02,
            2.3343614837332890554755238099460e-01,
            6.0855408871050020724879914268968e-01,
            9.0135796889724240088526130421087e-01,
            6.2266445266940384506426653388189e-01,
            -1.2692720787074832089480480590282e-01,
            -4.6253736950318169451890071286471e-01,
            -3.9482308578843897106658289430925e-02,
            2.9866874362698586642395071066858e-01,
            3.8664971797927487340285068739831e-02,
            -1.8722533722795392518989388008777e-01,
            -8.8243005420050348031058007336469e-03,
            1.0737308432663163204612288836870e-01,
            -1.0732430476938423918875642470994e-02,
            -5.2168072309854519874594558359604e-02,
            1.4563089924943300318416206096117e-02,
            1.9790177710283740619701120522222e-02,
            -9.8853733967658469899975060002362e-03,
            -5.1537896658028692006836557482075e-03,
            4.4236930891221213843489401540410e-03,
            5.7685344236866427022891734566201e-04,
            -1.3308057204323708866411379858619e-03,
            1.6156190702488963770351582116547e-04,
            2.4718649272750667844275596074510e-04,
            -8.6317891212541529946146445784194e-05,
            -1.9722154219490776696404113477179e-05,
            1.6032385720093409830870612164944e-05,
            -1.4758327456010213390660444285341e-06,
            -1.0413783294645852350716365430916e-06,
            3.2651137682251684123506938785264e-07,
            -2.9830567125395646088604952733681e-08
          };
      break;
    case 17:
      h = { 3.1703938650899862738974377407430e-03,
            3.6748896199244887594836939115339e-02,
            1.8556589584340402176998452432599e-01,
            5.2375501693137660463861493553850e-01,
            8.6407970046528492780169017351000e-01,
            7.3300918312113150232534053429845e-01,
            3.8629201600176063524649805458466e-02,
            -4.6431565514482092682513325598848e-01,
            -1.7903908657682734983929151439952e-01,
            2.7903931176262103397434088947193e-01,
            1.4302718043199780240115615015384e-01,
            -1.7934447123456620509784897876671e-01,
            -8.0739459938227717272773986678658e-02,
            1.1470118631597893410489774623784e-01,
            3.1554408431299391102342610793130e-02,
            -6.6358348749728263138436545887089e-02,
            -4.6258296806782469787200362532076e-03,
            3.2150273747621076203540013693782e-02,
            -4.3034377017964172210118256600708e-03,
            -1.2166368290071909522009896420514e-02,
            4.1973811742346631842304205406435e-03,
            3.2543956632576265426692874171977e-03,
            -2.0320061170844903701560024700257e-03,
            -4.6404945920505854227433584213713e-04,
            6.2149796814451457871442130453943e-04,
            -3.6218164282872656299389763079333e-05,
            -1.1603343965511909259211154044067e-04,
            3.2791106542406162017901788408025e-05,
            9.8862027222343679003311450737002e-06,
            -6.3723649625617885940229850882943e-06,
            4.2660453700255329024138440524583e-07,
            4.1828207733402810550264326779069e-07,
            -1.1913262141068780269119692198643e-07,
            1.0277787120590941886621601220574e-08
          };
      break;
    case 18:
      h = { 2.2292392894261341476647153569957e-03,
            2.7278103162550408622433195660051e-02,
            1.4649621327148704130749479190854e-01,
            4.4502322663201776542507559497608e-01,
            8.0868522687201060961115217651241e-01,
            8.0864965533087529436784279823769e-01,
            2.0820492164265561041425200983213e-01,
            -4.1528952705523225485961802405654e-01,
            -3.0615027286519064242398258102185e-01,
            2.1147297628013594916751571872737e-01,
            2.3628865852884980980341822487389e-01,
            -1.3057700280563036554504208197613e-01,
            -1.5097047504012414687046828021266e-01,
            9.1764381191532157222567889220954e-02,
            8.0682648302152173980239524553326e-02,
            -6.2969473759350580444049683137564e-02,
            -3.3563828020481217828674402881006e-02,
            3.7718074039279007869662763141605e-02,
            8.8560428508370828137863384199591e-03,
            -1.8457581363752618985873965584688e-02,
            1.6776820278716257444327764059011e-04,
            6.9909435703212766469172301242452e-03,
            -1.5821269103307486047355068237152e-03,
            -1.8958894667740898605512445129762e-03,
            8.8878469233540330178555688434017e-04,
            3.0204994152642787499235543791087e-04,
            -2.8093147682505936052477224507129e-04,
            -2.1721148267625707617217104689056e-07,
            5.2909093510065790795602463925107e-05,
            -1.2049951668048177800575404550187e-05,
            -4.7130568784725091659583255665478e-06,
            2.5013378893915945502541674921870e-06,
            -1.0877611266828616855187312041989e-07,
            -1.6632548270216130592390321382701e-07,
            4.3399892982154842698451632180406e-08,
            -3.5467549197311356709477164192218e-09
          };
      break;
    case 19:
      h = { 1.5678958152846272748603961844083e-03,
            2.0196523114657562170126681166948e-02,
            1.1494461010411641255224424185144e-01,
            3.7390170590255294991166579166020e-01,
            7.4166503761237689840157827347866e-01,
            8.5093873391779384363076133013237e-01,
            3.6896118039375752628572513458494e-01,
            -3.2256994316024734104075832874514e-01,
            -4.0423686967938060599081495638529e-01,
            1.0557425228315596432704381868461e-01,
            3.0030788695033916457788336629164e-01,
            -4.7402376549099775604467765788286e-02,
            -2.0192946643669554274147515116056e-01,
            3.9010162763916250705520383235125e-02,
            1.2290471236884119299315187845423e-01,
            -3.7478407724544741952410475960278e-02,
            -6.4593110252131366588734806555294e-02,
            3.0580625140236610687294316335283e-02,
            2.7401165431703535663165993696566e-02,
            -1.9782568984933180528118512597757e-02,
            -8.2970810591973861297665493452769e-03,
            9.9571204158029405495433650230552e-03,
            1.0874656837087327745233755749155e-03,
            -3.8007722061321243950393622412776e-03,
            4.8339043345746983285429743126826e-04,
            1.0405819037270433446984441161476e-03,
            -3.6865172646368375578354581456608e-04,
            -1.7621212955287076360778864003720e-04,
            1.2319596840241719124843522426005e-04,
            7.2209044276267830383855654552416e-06,
            -2.3532762999714585322235879671915e-05,
            4.2581465719275421569255460352288e-06,
            2.1664782709628571646042708570556e-06,
            -9.7054021264692123684527432642133e-07,
            2.0464918981107693371393718402909e-08,
            6.5576202903915930451975407466242e-08,
            -1.5788309442633350191744509658648e-08,
            1.2256775171147230574615288761797e-09
          };
      break;
    case 20:
      h = { 1.1030209784697370036660180048216e-03,
            1.4919096953433321622872753664524e-02,
            8.9694770502224324681961320493429e-02,
            3.1104511992145722487634884600993e-01,
            6.6849335614886884471275152463932e-01,
            8.6336781824427866194326952609117e-01,
            5.1124145370634632712381062447093e-01,
            -1.9687562291233057232275882597605e-01,
            -4.6214632517885934070989151223330e-01,
            -2.3655675146157809085600831622287e-02,
            3.2285230023796102072353164658125e-01,
            5.6356759004892016673426979878059e-02,
            -2.1985187363946029348582555940084e-01,
            -3.4954872441050534315198916601730e-02,
            1.4466233657484778296620220316981e-01,
            7.9651998921310210310942068190343e-03,
            -8.7289361758279004899208075585193e-02,
            8.3080546928725445415819805816682e-03,
            4.5671036383773788713824615115300e-02,
            -1.2429982511459568284006849125944e-02,
            -1.9531033366661638278616308639357e-02,
            9.5058164920688980464857920082977e-03,
            6.2515909968039852853682880606812e-03,
            -5.0649977555009108656780547619292e-03,
            -1.1760065027624948871126653315855e-03,
            1.9693767000596028478309751363895e-03,
            -7.5657029268068814531825183600233e-05,
            -5.4462035854466845589266776883619e-04,
            1.4358918969252962532894002389838e-04,
            9.5802798228155261386213004115575e-05,
            -5.2475613049099814925394641429435e-05,
            -6.1888020007977826320456210540399e-06,
            1.0240671536937550416246876794357e-05,
            -1.4311756540085687366042942539335e-06,
            -9.6832328287254565765814970368375e-07,
            3.7249313630534454078768497216345e-07,
            2.8486815247482631938422053924332e-10,
            -2.5665759353263497063665855347538e-08,
            5.7362298926689411330572237329717e-09,
            -4.2409952349582263523522284550586e-10
          };
      break;
    default:
      plumed_merror("Specified order currently not implemented");
    }
  }

  else if (type == Type::sym) {
    switch(order) {
    case 4:
      h = { 4.55703458959498611258176481442206e-02,
            -1.78247014416803885306084254125381e-02,
            -1.40317624178845662408221528494323e-01,
            4.21234534203536570284853723933338e-01,
            1.13665824340874954678781705297297e+00,
            7.03739068655225841197875524812844e-01,
            -4.19109651250538992051097864077747e-02,
            -1.07148901417882122522051702162571e-01
          };
      break;
    case 5:
      h = { 2.76321529578600101617613660209827e-02,
            -2.98424998687600064040115199759384e-02,
            -2.47951362613500064790983401508129e-01,
            2.34789231361400077546797149352642e-02,
            8.96581648381840179418134084698977e-01,
            1.02305296689210023508564972871682e+00,
            2.81990696854580058516859253359144e-01,
            -5.53441861167200052640247065482981e-02,
            4.17468644215800155361506540430128e-02,
            3.86547959548800140749946763207845e-02
          };
      break;
    case 6:
      h = { -1.10318675093800024761270250905909e-02,
            2.49992209278000017055876291749428e-03,
            6.32505626598800080317985816691362e-02,
            -2.97837512984400025473075146464907e-02,
            -1.02724969861800033421594946503319e-01,
            4.77904371332100064151404694712255e-01,
            1.11389278392792001604050255991751e+00,
            6.94457972956700086974990426824661e-01,
            -6.83231215864800039971882483769150e-02,
            -1.66863215411680032573826792940963e-01,
            4.93661237184000057515165238442023e-03,
            2.17847003265600018540038007586190e-02
          };
      break;
    case 7:
      h = { 1.45213947620201448696075630095947e-02,
            5.67134268576005746986812283694235e-03,
            -1.52463871896281522033689270756440e-01,
            -1.98056706807362009614692510695022e-01,
            4.08183939725744127091644486426958e-01,
            1.08578270981199098699221394781489e+00,
            7.58162601965387539237894998223055e-01,
            2.46656594886402458222107725305250e-02,
            -7.00782912221206977232412782541360e-02,
            9.60147679356809641282310963106283e-02,
            4.31554525820804293911869820021820e-02,
            -1.78704316511401790912305642677893e-02,
            -1.48122591460001472307705938646905e-03,
            3.79265853420003790433256973813059e-03
          };
      break;
    case 8:
      h = { 2.67279339279997366768393263214421e-03,
            -4.28394300239995752879740287255572e-04,
            -2.11456865283597902904766385745461e-02,
            5.38638875375994745364760518668845e-03,
            6.94904659111793138182022744331334e-02,
            -3.84935212633396237058569511191308e-02,
            -7.34625087607592708316417429159628e-02,
            5.15398670373074963002579806925496e-01,
            1.09910663053800905508694540912984e+00,
            6.80745347188773330948663442541147e-01,
            -8.66536154056391505484668869030429e-02,
            -2.02648655285637996348313549788145e-01,
            1.07586117504798947641653228401992e-02,
            4.48236230436195592918835473028594e-02,
            -7.66690896219992417716382337999903e-04,
            -4.78345851149995348466914535379146e-03
          };
      break;
    case 9:
      h = { 1.51248730936219739386106031275858e-03,
            -6.69141509131216294554989065801465e-04,
            -1.45155785529446045012402777274474e-02,
            1.25288962418580607921514413760633e-02,
            8.77912515542808719049006072054908e-02,
            -2.57864459294240427889022271301656e-02,
            -2.70893783504461360678305936744437e-01,
            4.98828309581242426706815251691296e-02,
            8.73048407349484634210057265590876e-01,
            1.01525979083351192144846208975650e+00,
            3.37658923602252358797670694912085e-01,
            -7.71721610974591964993152259921771e-02,
            8.25140928710566164128170640879034e-04,
            4.27444336024572976717728067796997e-02,
            -1.63033512255879239893907595160272e-02,
            -1.87693968364245344915275381936226e-02,
            8.76502538903070420837904119792938e-04,
            1.98119373648757883879523156167579e-03
          };
      break;
    case 10:
      h = { -6.49589896781769724452837433403829e-04,
            8.06612029979205351478896424133325e-05,
            6.49572837868184205234456385369413e-03,
            -1.13753531067373615109916684673408e-03,
            -2.87862319438586915232303198308728e-02,
            8.15281678377939578172828305469011e-03,
            7.07035675622903064807545092662622e-02,
            -4.52407723042161716286990724711359e-02,
            -5.02565403406056709911808866309002e-02,
            5.42813011102809772623345452302601e-01,
            1.08825153073747227239209678373300e+00,
            6.67071338429151339255440689157695e-01,
            -1.00240215012934938410005258901947e-01,
            -2.25558972319967215858227405078651e-01,
            1.64188694163413528848138867033413e-02,
            6.49509246029642689501670815843681e-02,
            -2.07236392054689850042437093691206e-03,
            -1.22206426340899319132882183680522e-02,
            1.35245019942278199434962382419201e-04,
            1.08917044724438603908733824709998e-03
          };
      break;
    case 11:
      h = { 6.91923233208036969968390828000793e-04,
            1.56320234204571122593394050248605e-04,
            -9.03626416328936338251409665645042e-03,
            -2.83333713296809227427663913090328e-03,
            6.08114528464804224294759649183106e-02,
            4.98747296770196690829379804199561e-02,
            -2.04498595618993084244152669270989e-01,
            -2.89425585980639887839771517974441e-01,
            3.36144293016675366203571684309281e-01,
            1.03286175231250965111939876805991e+00,
            8.08962653490911898224169362947578e-01,
            1.37459287684679687391309244048898e-01,
            -3.22902447410400722516143900975294e-02,
            9.89621390609645434910390804361668e-02,
            5.23788159925569485331209307332756e-02,
            -3.40554527782290436532797173185827e-02,
            -1.39412251318646231851294814418907e-02,
            9.21005970815790518668730868512284e-03,
            8.32056417857734271180392138234083e-04,
            -2.45276429733504728117332582826293e-03,
            -5.48653425032330071273514726648557e-05,
            2.42851511636182730569100129969229e-04
          };
      break;
    case 12:
      h = { -2.53238395868029360011308703803934e-04,
            -2.56794013941539134206898387091655e-05,
            3.32382276159138428939243681270455e-03,
            4.35079685976638231084678842819358e-04,
            -2.06331445793122500442784428287268e-02,
            -3.68316511823289565938366330044573e-03,
            8.17474545336821928342985188464809e-02,
            2.16399291162249034214948295584691e-02,
            -2.40940550666163183501922162577102e-01,
            -1.10779056856021962396141589124454e-01,
            6.54414642216806785590677009167848e-01,
            1.07972249467402248157554822682869e+00,
            5.64109951994629099658595805522054e-01,
            -3.13422339595566742942445159769704e-02,
            -5.06979026234183824395351791736175e-02,
            6.95500589276436959051963526690088e-02,
            1.06826589882294663985673466299886e-02,
            -3.42532744974815631100462098856951e-02,
            -1.99249856148396879348960730737872e-03,
            1.04863447995954799790707667739298e-02,
            2.54861210329473779345743622570808e-04,
            -1.90884289542041446843811947076119e-03,
            -1.60568790225561159679470990546690e-05,
            1.58345524644348421491879541633807e-04
          };
      break;
    case 13:
      h = { 9.96028729759322644298977222909741e-05,
            5.21920796195277043654989213017359e-05,
            -1.02016329687258286949835817125631e-03,
            5.84439592204865189864482211135055e-04,
            8.02545515204910846207564389942490e-03,
            -2.11063917638504251403364087025238e-03,
            -2.93444878165318237539516132983408e-02,
            2.49160343945351858696568569939700e-02,
            1.31417253195061006776356293812569e-01,
            1.24730209143528725235228193923831e-02,
            -1.98682995060360217953743244834186e-01,
            1.55889076380235497287785051412357e-01,
            9.11551693535715079619308198743965e-01,
            9.83923742598004968940017533896025e-01,
            2.79596836052714847742350912085385e-01,
            -1.75875078644905619151828091162315e-01,
            -8.45001480790217585070678296688129e-02,
            1.96045318821402025921685208231793e-02,
            -2.43409385742523465268050131271593e-02,
            -2.85908276815921014168964120472083e-02,
            7.49018377371175360368571460867315e-03,
            1.06436900199696239666113939392744e-02,
            -2.41749708924265434154335374650202e-04,
            -1.60663632304836349537491280159429e-03,
            -5.05420462645238263775342157924086e-05,
            9.64539648683696427926281824483112e-05
          };
      break;
    case 14:
      h = { 6.31007638147709694776549271821864e-05,
            2.73353579399835730078405920950857e-05,
            -8.56674265589630994088832238730902e-04,
            -1.03540533784965597756920485661425e-04,
            6.41017395468853039969259199892804e-03,
            1.43279913888553589278174182908288e-03,
            -2.74913418748536537983273575491694e-02,
            -3.34468414059296489937445073792333e-03,
            9.87511620870536871441558446349518e-02,
            3.66261337327129302132711075046245e-02,
            -2.26270508787995705546336466795765e-01,
            -8.21825286701456242388630357709189e-02,
            6.72226282198253310973257157456828e-01,
            1.07476870846271177484254621958826e+00,
            5.56070925104528490656718986429041e-01,
            -4.99473531504178586137854267690273e-02,
            -8.15074892290164332298019189693150e-02,
            5.29383812442578288726124924323813e-02,
            6.05357014372925363032695855736165e-03,
            -4.12896871319010858036868683029752e-02,
            -3.89442565747022016384160103541490e-03,
            1.41954425904793363893974245115714e-02,
            5.18276140762161932271090414303671e-04,
            -3.64788147216562339339440690366700e-03,
            -8.89051363774028501323759554608728e-05,
            5.63473132455830365457327690847933e-04,
            1.58545584728775700955431815497576e-05,
            -3.65985604352032339978313679740296e-05
          };
      break;
    case 15:
      h = { 4.05323627037270219162362616671658e-05,
            3.07137347973606144061961553681783e-05,
            -5.68752200237824536398212327270585e-04,
            -1.52953421693918361354627699633113e-04,
            4.92291805096766160038113113728286e-03,
            2.15828545106170558570690687361093e-03,
            -2.42838185667302274006740248069036e-02,
            -1.23669990433466353563218831368431e-02,
            9.61238540674488606274650237537571e-02,
            9.67227466631194188195763672410976e-02,
            -1.89584235398369660430262229056098e-01,
            -2.78071663287250692153662612327025e-01,
            3.45015366735401951547856924662483e-01,
            1.02084020241597150935319859854644e+00,
            8.18321122934300926310413615283323e-01,
            1.57732464332277610985499904927565e-01,
            -5.80996643342105573148970165675564e-02,
            5.76086678570035190527498514256877e-02,
            3.10245118607714344405312800745378e-02,
            -5.49799802677569254827361078241665e-02,
            -2.74428303436324676212176854050995e-02,
            1.42552403061262534383901723344934e-02,
            4.84149046146408433555219374966327e-03,
            -5.07726066683766071724326351954915e-03,
            -3.78042544045809091653870259719383e-04,
            1.51401068119577392884211874957145e-03,
            7.79550547699306238926333922911738e-05,
            -2.27210190383633081493211514079178e-04,
            -1.04081406015788967609316320594992e-05,
            1.37354357168879101134487813462570e-05
          };
      break;
    case 16:
      h = { -1.52706527381910275756142503222890e-05,
            -7.63177970130589524008421498102450e-06,
            2.33991244594547811736645614999475e-04,
            5.17120268202257300760825431407142e-05,
            -1.89323683821244256181293152963008e-03,
            -3.14120133085196543993450335463535e-04,
            9.81147588368609310793821975948958e-03,
            1.92311087754286520745683475297483e-03,
            -3.52885288476778719024551378424803e-02,
            -4.96427860935466733899312785638358e-03,
            1.10362189954493067345353551900189e-01,
            4.34462515148709496615175851275126e-02,
            -2.25697442637171785673189106091741e-01,
            -7.64249514011508951361051344974840e-02,
            6.72236243037978686842848219384905e-01,
            1.06988789812795226374930734891677e+00,
            5.61616638655705324723044213897083e-01,
            -4.88953427358677847336387856103102e-02,
            -9.47283364442045167086448032023327e-02,
            4.57258966692524679298337275668018e-02,
            6.88619390233227712805330611445243e-03,
            -4.39130321893507619135377240127127e-02,
            -4.42156298801803758491413276487947e-03,
            1.79134637043109300191812138791647e-02,
            1.01570512112273466515111053354303e-03,
            -5.48843874201232484466572714154609e-03,
            -1.53365269849333965166901516674614e-04,
            1.20541159197261760899211324726821e-03,
            3.97091116582507492704760820867449e-05,
            -1.54759482169960371376535301912725e-04,
            -4.40323369887264181395027801135811e-06,
            8.81055997054166680230915392746383e-06
          };
      break;
    case 17:
      h = { 5.36164168581481346588513137163545e-06,
            -3.46866471633543758739618573239127e-06,
            -1.07580985050661424558618439739632e-04,
            3.56494009271542727150354490373019e-05,
            1.01798919677591626770096766563256e-03,
            8.25906783075320302753979939680562e-05,
            -5.56114774235276932562443974461530e-03,
            -2.69465339683962841130226983921148e-03,
            1.75319890806961142803999109673896e-02,
            1.40756428708096667512528199495137e-02,
            -2.55108531297768073575760183757666e-02,
            -1.02695023497632355324871156199151e-02,
            2.28520064519878510211903233084740e-02,
            -1.21722598355376729339916153094237e-01,
            -2.19310589964256619222027211435488e-01,
            2.55321529059814311679588172410149e-01,
            9.63770979824808082803144770878134e-01,
            9.20252282282772893218236731627258e-01,
            2.01381678415994358477547621077974e-01,
            -1.67678964148070125395051377381606e-01,
            2.44251344634779481157504932298252e-02,
            1.48145397031054587566600844183995e-01,
            2.53200120416010171497944725160778e-02,
            -4.70811260450583399239654625034746e-02,
            -6.81539610621377002497300878758324e-03,
            1.48243054824644532319055656444107e-02,
            1.21165585310027278283240281808730e-03,
            -3.87731534844140950796242961473581e-03,
            -1.96069824769335180822404929301683e-04,
            6.73160536609661666086756870441832e-04,
            -1.91009105827510641166869231888370e-05,
            -8.90063957100391663296648503411745e-05,
            3.93169287554600132558258254733552e-06,
            6.07736121570621208446468658470607e-06
          };
      break;
    case 18:
      h = { -2.13992159246166473816484938919036e-06,
            1.10977553385413026034323619173350e-06,
            4.18005291525171048536493878522435e-05,
            -1.39424713387653467029560910850350e-05,
            -3.75940547191951806357129717284238e-04,
            6.70565555942921202913289957514564e-05,
            2.01961905200075380467650276727909e-03,
            -2.66969918023956154402853702123366e-04,
            -7.41018163371885089235613364166966e-03,
            1.53836000238939262126092177851433e-03,
            2.12306779452164293142679696302366e-02,
            -4.61138867117545040580361970228296e-03,
            -4.48485088469943032896125600927917e-02,
            8.87835433253391509422769445336598e-03,
            4.03469430616465660199843057398539e-02,
            -1.04367839842760060098392216332286e-01,
            -4.59344672605674214649340569849301e-02,
            5.67783920692876864499964995047776e-01,
            1.06579255093152891475938304211013e+00,
            6.70293472647235155648104409920052e-01,
            -7.35803422739719908962641170546704e-02,
            -2.26186698989002510362666953369626e-01,
            4.80771334803232336385647727183823e-02,
            1.19104967186232965636882852322742e-01,
            -7.18008269166581052089792791548462e-03,
            -4.28861550987859954653380611944158e-02,
            2.32353364582536326202188980971641e-03,
            1.34380897535976585183137288481703e-02,
            -5.81978733884339397529339432679762e-04,
            -3.27230890166736818752690751921364e-03,
            9.92958015089934567000418130788830e-05,
            5.60266733289934543670030020479089e-04,
            -1.98286778613536816145718894022210e-05,
            -6.39885786401657284281838111716922e-05,
            1.91614024625604985099780984414419e-06,
            3.69479211060541101930730298330552e-06
          };
      break;
    case 19:
      h = { 2.47619856876035267576182871684942e-06,
            2.91655676074685013668191455038503e-06,
            -3.98117220914099361156295031616281e-05,
            -2.37890336748671808665862859033524e-05,
            3.90632340413758891075474144471968e-04,
            1.82868669835172258141334222791841e-04,
            -2.41117792464671338789616861220111e-03,
            -8.73874136465524497564827388629283e-04,
            1.16845675580482498412404623877592e-02,
            6.10848600213812192166740189236407e-03,
            -3.91877120522209840269844960403134e-02,
            -2.39118550567039567233962316095131e-02,
            1.18896719019111141157907240994973e-01,
            1.32414008615193368179063782008598e-01,
            -1.64390631231593420391590143481153e-01,
            -2.49745683326209749752777611320198e-01,
            3.65243519237511582442579083362943e-01,
            1.01760518334862726241851760278223e+00,
            8.17620422715306993666217749705538e-01,
            1.54185800478859824647059895141865e-01,
            -9.54948528806143087654945134090667e-02,
            1.26637042824246960626144442585428e-02,
            9.92151969664582453201084888405603e-03,
            -6.59532404097190189107280389180232e-02,
            -3.20347562503009650192176138716604e-02,
            2.23409529027087383479255322527024e-02,
            1.12690735439448276589136099801181e-02,
            -7.24389178390821321384951403388186e-03,
            -1.64148228823170379275797614582189e-03,
            3.00014804641317691172308634861565e-03,
            2.25083469591080945247998923441912e-04,
            -8.99106799594682343235163557437772e-04,
            -6.52240895281959979666239202167333e-05,
            1.63397150769660883320999467116508e-04,
            1.25487584192582468661661745934488e-05,
            -1.68015900652064169079266675588258e-05,
            -9.14098333564239404867173867386132e-07,
            7.76082610749008778884821136562211e-07
          };
      break;
    case 20:
      h = { -8.95074013313225034112988621332052e-07,
            -4.60567304495735576428397861289965e-07,
            1.73767995228992186179835244264069e-05,
            6.39991346364665771576834066292072e-06,
            -1.66016418292479542555967197969835e-04,
            -3.76400722545106348484143798938106e-05,
            1.05728141728119568419630436295620e-03,
            1.77400246424138069000625561599804e-04,
            -4.90965140656484527736935774555604e-03,
            -8.64263223121293565169576122997341e-04,
            1.71926521880978698531006187977255e-02,
            2.74159026446779826788335832077337e-03,
            -5.00254525875804267887048126794980e-02,
            -9.67845613584030449449979727205573e-03,
            1.25751400487193104593686143743980e-01,
            5.12665874779235433700286250768841e-02,
            -2.27092007441644233578514899818401e-01,
            -7.22498274381410260369662523771694e-02,
            6.67496745413005965197328350768657e-01,
            1.06230451808460291118763052509166e+00,
            5.73932332635040842738760602514958e-01,
            -4.21709558919741780980139367329684e-02,
            -1.11714873948519005653068347783119e-01,
            3.61746629928947835419528189504490e-02,
            1.14879797113225934346303347410867e-02,
            -4.47307789806269048837883417490957e-02,
            -4.68650205569037543035237902699919e-03,
            2.40473567441355992602147750858421e-02,
            2.01254944419308181777217292562909e-03,
            -9.34312323805295934231462950947389e-03,
            -4.31706836813935599089092631786002e-04,
            2.95428464804820467212720380700830e-03,
            1.02049526041390444994726161009879e-04,
            -6.99655419422072467026951780866284e-04,
            -2.72718682941954995583092036248729e-05,
            1.13037635169613691495625606986408e-04,
            4.27893798113462174641198274005482e-06,
            -1.11996683141520158823655969881905e-05,
            -2.68922267421127013513505862526975e-07,
            5.22627921716999034327578478598131e-07
          };
      break;
    default:
      plumed_merror("Wavelets: Specified order currently not implemented");
    }
  }

  // to get highpass: reverse order and inverse sign of every second
  if (!lowpass) {
    std::reverse(h.begin(), h.end());
    for (unsigned i=1; i < h.size(); i += 2) {
      h[i] = -h[i];
    }
  }

  return h;
}


}
}
