/**
 * Skill Generation Utilities
 *
 * Shared utilities for generating skill and command files.
 */
import { getExploreSkillTemplate, getNewChangeSkillTemplate, getContinueChangeSkillTemplate, getApplyChangeSkillTemplate, getFfChangeSkillTemplate, getSyncSpecsSkillTemplate, getArchiveChangeSkillTemplate, getBulkArchiveChangeSkillTemplate, getVerifyChangeSkillTemplate, getOnboardSkillTemplate, getOpsxProposeSkillTemplate, getOpsxExploreCommandTemplate, getOpsxNewCommandTemplate, getOpsxContinueCommandTemplate, getOpsxApplyCommandTemplate, getOpsxFfCommandTemplate, getOpsxSyncCommandTemplate, getOpsxArchiveCommandTemplate, getOpsxBulkArchiveCommandTemplate, getOpsxVerifyCommandTemplate, getOpsxOnboardCommandTemplate, getOpsxProposeCommandTemplate, } from '../templates/skill-templates.js';
/**
 * Gets skill templates with their directory names, optionally filtered by workflow IDs.
 *
 * @param workflowFilter - If provided, only return templates whose workflowId is in this array
 */
export function getSkillTemplates(workflowFilter) {
    const all = [
        { template: getExploreSkillTemplate(), dirName: 'openspec-explore', workflowId: 'explore' },
        { template: getNewChangeSkillTemplate(), dirName: 'openspec-new-change', workflowId: 'new' },
        { template: getContinueChangeSkillTemplate(), dirName: 'openspec-continue-change', workflowId: 'continue' },
        { template: getApplyChangeSkillTemplate(), dirName: 'openspec-apply-change', workflowId: 'apply' },
        { template: getFfChangeSkillTemplate(), dirName: 'openspec-ff-change', workflowId: 'ff' },
        { template: getSyncSpecsSkillTemplate(), dirName: 'openspec-sync-specs', workflowId: 'sync' },
        { template: getArchiveChangeSkillTemplate(), dirName: 'openspec-archive-change', workflowId: 'archive' },
        { template: getBulkArchiveChangeSkillTemplate(), dirName: 'openspec-bulk-archive-change', workflowId: 'bulk-archive' },
        { template: getVerifyChangeSkillTemplate(), dirName: 'openspec-verify-change', workflowId: 'verify' },
        { template: getOnboardSkillTemplate(), dirName: 'openspec-onboard', workflowId: 'onboard' },
        { template: getOpsxProposeSkillTemplate(), dirName: 'openspec-propose', workflowId: 'propose' },
    ];
    if (!workflowFilter)
        return all;
    const filterSet = new Set(workflowFilter);
    return all.filter(entry => filterSet.has(entry.workflowId));
}
/**
 * Gets command templates with their IDs, optionally filtered by workflow IDs.
 *
 * @param workflowFilter - If provided, only return templates whose id is in this array
 */
export function getCommandTemplates(workflowFilter) {
    const all = [
        { template: getOpsxExploreCommandTemplate(), id: 'explore' },
        { template: getOpsxNewCommandTemplate(), id: 'new' },
        { template: getOpsxContinueCommandTemplate(), id: 'continue' },
        { template: getOpsxApplyCommandTemplate(), id: 'apply' },
        { template: getOpsxFfCommandTemplate(), id: 'ff' },
        { template: getOpsxSyncCommandTemplate(), id: 'sync' },
        { template: getOpsxArchiveCommandTemplate(), id: 'archive' },
        { template: getOpsxBulkArchiveCommandTemplate(), id: 'bulk-archive' },
        { template: getOpsxVerifyCommandTemplate(), id: 'verify' },
        { template: getOpsxOnboardCommandTemplate(), id: 'onboard' },
        { template: getOpsxProposeCommandTemplate(), id: 'propose' },
    ];
    if (!workflowFilter)
        return all;
    const filterSet = new Set(workflowFilter);
    return all.filter(entry => filterSet.has(entry.id));
}
/**
 * Converts command templates to CommandContent array, optionally filtered by workflow IDs.
 *
 * @param workflowFilter - If provided, only return contents whose id is in this array
 */
export function getCommandContents(workflowFilter) {
    const commandTemplates = getCommandTemplates(workflowFilter);
    return commandTemplates.map(({ template, id }) => ({
        id,
        name: template.name,
        description: template.description,
        category: template.category,
        tags: template.tags,
        body: template.content,
    }));
}
/**
 * Generates skill file content with YAML frontmatter.
 *
 * @param template - The skill template
 * @param generatedByVersion - The OpenSpec version to embed in the file
 * @param transformInstructions - Optional callback to transform the instructions content
 */
export function generateSkillContent(template, generatedByVersion, transformInstructions) {
    const instructions = transformInstructions
        ? transformInstructions(template.instructions)
        : template.instructions;
    return `---
name: ${template.name}
description: ${template.description}
license: ${template.license || 'MIT'}
compatibility: ${template.compatibility || 'Requires openspec CLI.'}
metadata:
  author: ${template.metadata?.author || 'openspec'}
  version: "${template.metadata?.version || '1.0'}"
  generatedBy: "${generatedByVersion}"
---

${instructions}
`;
}
//# sourceMappingURL=skill-generation.js.map