import re
from typing import TYPE_CHECKING, Any, Optional, cast

from tmt._compat.pathlib import Path
from tmt.package_managers import (
    FileSystemPath,
    Installable,
    Options,
    Package,
    PackageManager,
    PackageManagerEngine,
    escape_installables,
    provides_package_manager,
)

if TYPE_CHECKING:
    from tmt._compat.typing import TypeAlias

    # TODO: Move Repository abstraction to tmt.package_manager subpackage
    # This class will be added in a future PR.
    # For now, just type it as Any to satisfy pyright.
    Repository: TypeAlias = Any
else:
    Repository: Any = None  # type: ignore[assignment]

from tmt.utils import Command, GeneralError, RunError, ShellScript


class DnfEngine(PackageManagerEngine):
    _base_command = Command('dnf')
    _base_debuginfo_command = Command('debuginfo-install')

    skip_missing_packages_option = '--skip-broken'
    skip_missing_debuginfo_option = skip_missing_packages_option

    def prepare_command(self) -> tuple[Command, Command]:
        options = Command('-y')

        command = self._base_command

        if self.guest.facts.sudo_prefix:
            command = Command(self.guest.facts.sudo_prefix) + self._base_command

        return (command, options)

    def _extra_dnf_options(self, options: Options, command: Optional[Command] = None) -> Command:
        """
        Collect additional options for ``yum``/``dnf`` based on given options
        """

        command = command or self._base_command

        extra_options = Command()

        for package in options.excluded_packages:
            extra_options += Command('--exclude', package)

        if options.skip_missing:
            if str(command) == str(self._base_command):
                extra_options += Command(self.skip_missing_packages_option)

            elif str(command) == str(self._base_debuginfo_command):
                extra_options += Command(self.skip_missing_debuginfo_option)

            else:
                raise GeneralError(f"Unhandled package manager command '{command}'.")

        return extra_options

    def _construct_presence_script(
        self, *installables: Installable, what_provides: bool = True
    ) -> ShellScript:
        if what_provides:
            return ShellScript(
                f'rpm -q --whatprovides {" ".join(escape_installables(*installables))}'
            )

        return ShellScript(f'rpm -q {" ".join(escape_installables(*installables))}')

    def check_presence(self, *installables: Installable) -> ShellScript:
        return self._construct_presence_script(*installables)

    def _construct_install_script(
        self, *installables: Installable, options: Optional[Options] = None
    ) -> ShellScript:
        options = options or Options()

        extra_options = self._extra_dnf_options(options)

        script = ShellScript(
            f'{self.command.to_script()} install '
            f'{self.options.to_script()} {extra_options} '
            f'{" ".join(escape_installables(*installables))}'
        )

        if options.check_first:
            script = self._construct_presence_script(*installables) | script

        return script

    def _construct_reinstall_script(
        self, *installables: Installable, options: Optional[Options] = None
    ) -> ShellScript:
        options = options or Options()

        extra_options = self._extra_dnf_options(options)

        script = ShellScript(
            f'{self.command.to_script()} reinstall '
            f'{self.options.to_script()} {extra_options} '
            f'{" ".join(escape_installables(*installables))}'
        )

        if options.check_first:
            script = self._construct_presence_script(*installables) & script

        return script

    def _construct_install_debuginfo_script(
        self, *installables: Installable, options: Optional[Options] = None
    ) -> ShellScript:
        options = options or Options()

        extra_options = self._extra_dnf_options(options, command=self._base_debuginfo_command)

        return ShellScript(
            f'{self._base_debuginfo_command.to_script()} '
            f'{self.options.to_script()} {extra_options} '
            f'{" ".join(escape_installables(*installables))}'
        )

    def refresh_metadata(self) -> ShellScript:
        return ShellScript(
            f'{self.command.to_script()} makecache {self.options.to_script()} --refresh'
        )

    def install(
        self,
        *installables: Installable,
        options: Optional[Options] = None,
    ) -> ShellScript:
        return self._construct_install_script(*installables, options=options)

    def reinstall(
        self,
        *installables: Installable,
        options: Optional[Options] = None,
    ) -> ShellScript:
        return self._construct_reinstall_script(*installables, options=options)

    def install_debuginfo(
        self,
        *installables: Installable,
        options: Optional[Options] = None,
    ) -> ShellScript:
        # Make sure debuginfo-install is present on the target system
        script = self.install(FileSystemPath('/usr/bin/debuginfo-install'))

        options = options or Options()

        script &= cast(  # type: ignore[redundant-cast]
            ShellScript,
            self._construct_install_debuginfo_script(  # type: ignore[reportGeneralIssues,unused-ignore]
                *installables, options=options
            ),
        )

        # Extra ignore/check for yum to workaround BZ#1920176
        if not options.skip_missing:
            script &= cast(  # type: ignore[redundant-cast]
                ShellScript,
                self._construct_presence_script(  # type: ignore[reportGeneralIssues,unused-ignore]
                    *tuple(Package(f'{installable}-debuginfo') for installable in installables),
                    what_provides=False,
                ),
            )

        return script

    def install_repository(self, repository: Repository) -> ShellScript:
        repo_path = f"/etc/yum.repos.d/{repository.filename}"
        return ShellScript(
            f"{self.guest.facts.sudo_prefix} tee {repo_path} <<'EOF'\n{repository.content}\nEOF"
        )

    def list_packages(self, repository: Repository) -> ShellScript:
        repo_ids = " ".join(f"--enablerepo={repo_id}" for repo_id in repository.repo_ids)
        return ShellScript(
            f"""
            {self.command.to_script()} repoquery --disablerepo='*' {repo_ids}
            """
        )

    def create_repository(self, directory: Path) -> ShellScript:
        """
        Create repository metadata for package files in the given directory.

        :param directory: The path to the directory containing RPM packages.
        :returns: A shell script to create repository metadata.
        """
        return ShellScript(f"createrepo {directory}")


# ignore[type-arg]: TypeVar in package manager registry annotations is
# puzzling for type checkers. And not a good idea in general, probably.
@provides_package_manager('dnf')  # type: ignore[arg-type]
class Dnf(PackageManager[DnfEngine]):
    NAME = 'dnf'

    _engine_class = DnfEngine

    bootc_builder = True

    probe_command = ShellScript(
        """
        type dnf && ((dnf --version | grep -E 'dnf5 version') && exit 1 || exit 0)
        """
    ).to_shell_command()
    # The priority of preference: `rpm-ostree` > `dnf5` > `dnf` > `yum`.
    # `rpm-ostree` has its own implementation and its own priority, and
    # the `dnf` family just stays below it.
    probe_priority = 50

    def check_presence(self, *installables: Installable) -> dict[Installable, bool]:
        try:
            output = self.guest.execute(self.engine.check_presence(*installables))
            stdout = output.stdout

        except RunError as exc:
            stdout = exc.stdout

        if stdout is None:
            raise GeneralError("rpm presence check provided no output")

        results: dict[Installable, bool] = {}

        for line, installable in zip(stdout.strip().splitlines(), installables):
            # Match for packages not installed, when "rpm -q PACKAGE" used
            match = re.match(rf'package {re.escape(str(installable))} is not installed', line)
            if match is not None:
                results[installable] = False
                continue

            # Match for provided rpm capabilities (packages, commands, etc.),
            # when "rpm -q --whatprovides CAPABILITY" used
            match = re.match(rf'no package provides {re.escape(str(installable))}', line)
            if match is not None:
                results[installable] = False
                continue

            # Match for filesystem paths, when "rpm -q --whatprovides PATH" used
            match = re.match(
                rf'error: file {re.escape(str(installable))}: No such file or directory', line
            )
            if match is not None:
                results[installable] = False
                continue

            results[installable] = True

        return results


class Dnf5Engine(DnfEngine):
    _base_command = Command('dnf5')
    skip_missing_packages_option = '--skip-unavailable'


# ignore[type-arg]: TypeVar in package manager registry annotations is
# puzzling for type checkers. And not a good idea in general, probably.
@provides_package_manager('dnf5')  # type: ignore[arg-type]
class Dnf5(Dnf):
    NAME = 'dnf5'

    _engine_class = Dnf5Engine

    probe_command = Command('dnf5', '--version')
    probe_priority = 60


class YumEngine(DnfEngine):
    _base_command = Command('yum')

    # TODO: get rid of those `type: ignore` below. I think it's caused by the
    # decorator, it might be messing with the class inheritance as seen by pyright,
    # but mypy sees no issue, pytest sees no issue, everything works. Silencing
    # for now.
    def install(
        self, *installables: Installable, options: Optional[Options] = None
    ) -> ShellScript:
        options = options or Options()

        script = cast(  # type: ignore[redundant-cast]
            ShellScript,
            self._construct_install_script(  # type: ignore[reportGeneralIssues,unused-ignore]
                *installables, options=options
            ),
        )

        # Extra ignore/check for yum to workaround BZ#1920176
        if options.skip_missing:
            script |= ShellScript('/bin/true')

        else:
            script &= cast(  # type: ignore[redundant-cast]
                ShellScript,
                self._construct_presence_script(  # type: ignore[reportGeneralIssues,unused-ignore]
                    *installables
                ),
            )

        return script

    def reinstall(
        self, *installables: Installable, options: Optional[Options] = None
    ) -> ShellScript:
        options = options or Options()

        script = cast(  # type: ignore[redundant-cast]
            ShellScript,
            self._construct_reinstall_script(  # type: ignore[reportGeneralIssues,unused-ignore]
                *installables, options=options
            ),
        )

        # Extra ignore/check for yum to workaround BZ#1920176
        if options.skip_missing:
            script |= ShellScript('/bin/true')

        else:
            script &= cast(  # type: ignore[redundant-cast]
                ShellScript,
                self._construct_presence_script(  # type: ignore[reportGeneralIssues,unused-ignore]
                    *installables
                ),
            )

        return script

    def refresh_metadata(self) -> ShellScript:
        return ShellScript(f'{self.command.to_script()} makecache')


# ignore[type-arg]: TypeVar in package manager registry annotations is
# puzzling for type checkers. And not a good idea in general, probably.
@provides_package_manager('yum')  # type: ignore[arg-type]
class Yum(Dnf):
    NAME = 'yum'

    _engine_class = YumEngine

    bootc_builder = False

    probe_command = ShellScript(
        """
        type yum && ((yum --version | grep -E 'dnf5 version') && exit 1 || exit 0)
        """
    ).to_shell_command()
    probe_priority = 40
