/* -*- mode: c++; indent-tabs-mode: nil -*- */
/** @file XmlDoc.qpp defines the XmlDoc class */
/*
    QC_XmlDoc.qpp

    Qore Programming Language

    Copyright 2003 - 2022 Qore Technologies, s.r.o.

    This library is free software; you can redistribute it and/or
    modify it under the terms of the GNU Lesser General Public
    License as published by the Free Software Foundation; either
    version 2.1 of the License, or (at your option) any later version.

    This library is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    Lesser General Public License for more details.

    You should have received a copy of the GNU Lesser General Public
    License along with this library; if not, write to the Free Software
    Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
*/

#include "qore-xml-module.h"

#include "QC_XmlDoc.h"
#include "QoreXPath.h"
#include "QoreXmlReader.h"
#include "QC_XmlNode.h"
#include "ql_xml.h"
#include "MakeXmlOpts.h"

#ifdef HAVE_XMLTEXTREADERRELAXNGSETSCHEMA
int QoreXmlDoc::validateRelaxNG(const char *rng, int size, ExceptionSink *xsink) {
   QoreXmlRelaxNGContext schema(rng, size, xsink);
   if (!schema) {
      if (!*xsink)
         xsink->raiseException("RELAXNG-SYNTAX-ERROR", "RelaxNG schema passed as argument to XmlDoc::validateRelaxNG() could not be parsed");
      return -1;
   }

   QoreXmlRelaxNGValidContext vcp(schema);
   int rc = vcp.validateDoc(ptr);

   if (!rc)
      return 0;
   if (*xsink)
      return -1;

   if (rc < 0)
      xsink->raiseException("RELAXNG-INTERNAL-ERROR", "an internal error occured validating the document against the RelaxNG schema passed; xmlRelaxNGValidateDoc() returned %d", rc);
   else if (rc)
      xsink->raiseException("RELAXNG-ERROR", "The document failed RelaxNG validation", rc);
   return -1;
}
#endif

#ifdef HAVE_XMLTEXTREADERSETSCHEMA
int QoreXmlDoc::validateSchema(const QoreString& xsd, ExceptionSink *xsink) {
    QoreXmlSchemaContext schema(xsd, xsink);
    if (*xsink)
        return -1;
    assert(schema);

    int rc = schema.validateDoc(ptr);

    if (!rc)
        return 0;

    if (rc < 0)
        xsink->raiseException("XSD-INTERNAL-ERROR", "an internal error occured validating the document against the XSD schema passed; xmlSchemaValidateDoc() returned %d", rc);
    else if (rc)
        xsink->raiseException("XSD-ERROR", "The document failed XSD validation", rc);
    return -1;
}
#endif

int QoreXmlDoc::validateDtd(const QoreString& dtd, ExceptionSink* xsink) {
    TempEncodingHelper str(dtd, QCS_UTF8, xsink);
    if (!str)
        return -1;

    xmlParserInputBufferPtr bptr = xmlParserInputBufferCreateMem(str->c_str(), str->size(), XML_CHAR_ENCODING_UTF8);
    if (!bptr) {
        xsink->raiseException("DTD-VALIDATION-ERROR", "failed to create buffer for DTD parsing: xmlParserInputBufferCreateMem() failed");
        return -1;
    }

    // xmlIOParseDTD() frees the bptr arg
    xmlDtdPtr xdp = xmlIOParseDTD(nullptr, bptr, XML_CHAR_ENCODING_UTF8);
    if (!xdp) {
        xsink->raiseException("DTD-SYNTAX-ERROR", "failed to parse DTD: xmlIOParseDTD() failed");
        return -1;
    }
    ON_BLOCK_EXIT(xmlFreeDtd, xdp);

    xmlValidCtxtPtr vctxt = xmlNewValidCtxt();
    if (!vctxt) {
        xsink->raiseException("DTD-VALIDATION-ERROR", "failed to create validation context for DTD parsing: xmlNewValidCtxt() failed");
        return -1;
    }
    ON_BLOCK_EXIT(xmlFreeValidCtxt, vctxt);

    if (!xmlValidateDtd(vctxt, ptr, xdp)) {
        xsink->raiseException("DTD-VALIDATION-ERROR", "the XML document failed DTD validation");
        return -1;
    }
    return 0;
}

QoreXmlNodeData *QoreXmlDocData::getRootElement() {
   xmlNodePtr n = xmlDocGetRootElement(ptr);
   if (!n) return 0;
   return new QoreXmlNodeData(n, this);
}

QoreStringNode *doString(xmlChar *str) {
   if (!str)
      return 0;
   QoreStringNode *rv = new QoreStringNode((const char *)str);
   xmlFree(str);
   return rv;
}

QoreXmlNodeData *doNode(xmlNodePtr p, QoreXmlDocData *doc) {
   if (!p)
      return 0;
   return new QoreXmlNodeData(p, doc);
}
/* Qore class Qore::Xml::XmlDoc */

qore_classid_t CID_XMLDOC;
QoreClass* QC_XMLDOC;

// XmlDoc::constructor(hash data, *hash opts) {}
static void XmlDoc_constructor_VhNh(QoreObject* self, const QoreListNode* args, q_rt_flags_t rtflags, ExceptionSink* xsink) {
    const QoreHashNode* data = HARD_QORE_VALUE_HASH(args, 0);
    const QoreHashNode* opts = get_param_value(args, 1).get<const QoreHashNode>();
# 149 "QC_XmlDoc.qpp"
SimpleRefHolder<QoreStringNode> xml;
   try {
       xml = make_xml(xsink, *data, MakeXmlOpts::createFromHash(opts));
   } catch (const MakeXmlOpts::InvalidHash &exc) {
      xsink->raiseException("MAKE-XML-OPTS-INVALID",
                            "the opts hash passed is not valid; invalid argument: '%s'",
                            exc.what());
      return;
   }
   if (!xml)
      return;
   SimpleRefHolder<QoreXmlDocData> xd(new QoreXmlDocData(*xml));
   if (!xd->isValid()) {
      xsink->raiseException("XMLDOC-CONSTRUCTOR-ERROR", "error parsing XML string");
      return;
   }

   self->setPrivate(CID_XMLDOC, xd.release());
}

// XmlDoc::constructor(string xml) {}
static void XmlDoc_constructor_Vs(QoreObject* self, const QoreListNode* args, q_rt_flags_t rtflags, ExceptionSink* xsink) {
    const QoreStringNode* xml = HARD_QORE_VALUE_STRING(args, 0);
# 178 "QC_XmlDoc.qpp"
SimpleRefHolder<QoreXmlDocData> xd(new QoreXmlDocData(xml));
   if (!xd->isValid()) {
      xsink->raiseException("XMLDOC-CONSTRUCTOR-ERROR", "error parsing XML string");
      return;
   }

   self->setPrivate(CID_XMLDOC, xd.release());
}

// XmlDoc::copy() {}
static void XmlDoc_copy(QoreObject* self, QoreObject* old, QoreXmlDocData* xd, ExceptionSink* xsink) {
# 194 "QC_XmlDoc.qpp"
self->setPrivate(CID_XMLDOC, new QoreXmlDocData(*xd));
}

// list XmlDoc::evalXPath(string xpath){}
static QoreValue XmlDoc_evalXPath_Vs(QoreObject* self, QoreXmlDocData* xd, const QoreListNode* args, q_rt_flags_t rtflags, ExceptionSink* xsink) {
    const QoreStringNode* xpath = HARD_QORE_VALUE_STRING(args, 0);
# 276 "QC_XmlDoc.qpp"
    QoreXPath xp(xd, xsink);
   if (!xp)
      return 0;

   return xp.eval(xpath->getBuffer(), xsink);
}

// *XmlNode XmlDoc::getRootElement(){}
static QoreValue XmlDoc_getRootElement(QoreObject* self, QoreXmlDocData* xd, const QoreListNode* args, q_rt_flags_t rtflags, ExceptionSink* xsink) {
# 290 "QC_XmlDoc.qpp"
    QoreXmlNodeData *n = xd->getRootElement();
   return !n ? 0 : new QoreObject(QC_XMLNODE, getProgram(), n);
}

// string XmlDoc::getVersion(){}
static QoreValue XmlDoc_getVersion(QoreObject* self, QoreXmlDocData* xd, const QoreListNode* args, q_rt_flags_t rtflags, ExceptionSink* xsink) {
# 204 "QC_XmlDoc.qpp"
    return new QoreStringNode(xd->getVersion());
}

// hash XmlDoc::toQore(int pflags = XPF_PRESERVE_ORDER){}
static QoreValue XmlDoc_toQore_Vi(QoreObject* self, QoreXmlDocData* xd, const QoreListNode* args, q_rt_flags_t rtflags, ExceptionSink* xsink) {
    int64 pflags = HARD_QORE_VALUE_INT(args, 0);
# 224 "QC_XmlDoc.qpp"
    QoreXmlReader reader(xd->getDocPtr(), xsink);
   if (*xsink)
      return 0;
   return reader.parseXmlData(QCS_UTF8, pflags, xsink);
}

// hash XmlDoc::toQoreData(*int pflags){}
static QoreValue XmlDoc_toQoreData_Ni(QoreObject* self, QoreXmlDocData* xd, const QoreListNode* args, q_rt_flags_t rtflags, ExceptionSink* xsink) {
    int64 pflags = HARD_QORE_VALUE_INT(args, 0);
# 245 "QC_XmlDoc.qpp"
    QoreXmlReader reader(xd->getDocPtr(), xsink);
   if (*xsink)
      return 0;
   return reader.parseXmlData(QCS_UTF8, pflags, xsink);
}

// string XmlDoc::toString(){}
static QoreValue XmlDoc_toString(QoreObject* self, QoreXmlDocData* xd, const QoreListNode* args, q_rt_flags_t rtflags, ExceptionSink* xsink) {
# 261 "QC_XmlDoc.qpp"
    return xd->toString(xsink);
}

// nothing XmlDoc::validateDtd(string dtd){}
static QoreValue XmlDoc_validateDtd_Vs(QoreObject* self, QoreXmlDocData* xd, const QoreListNode* args, q_rt_flags_t rtflags, ExceptionSink* xsink) {
    const QoreStringNode* dtd = HARD_QORE_VALUE_STRING(args, 0);
# 354 "QC_XmlDoc.qpp"
    xd->validateDtd(*dtd, xsink);
    return QoreValue();
}

// nothing XmlDoc::validateRelaxNG(string relaxng){}
static QoreValue XmlDoc_validateRelaxNG_Vs(QoreObject* self, QoreXmlDocData* xd, const QoreListNode* args, q_rt_flags_t rtflags, ExceptionSink* xsink) {
    const QoreStringNode* relaxng = HARD_QORE_VALUE_STRING(args, 0);
# 307 "QC_XmlDoc.qpp"
    #ifdef HAVE_XMLTEXTREADERRELAXNGSETSCHEMA
   // convert to UTF-8
   TempEncodingHelper nrng(relaxng, QCS_UTF8, xsink);
   if (!nrng)
      return 0;

   xd->validateRelaxNG(nrng->getBuffer(), nrng->strlen(), xsink);
#else
   xsink->raiseException("MISSING-FEATURE-ERROR", "the libxml2 version used to compile the xml module did not support the xmlTextReaderRelaxNGValidate() function, therefore XmlDoc::validateRelaxNG() is not available; for maximum portability, use the constant Option::HAVE_PARSEXMLWITHRELAXNG to check if this method is implemented before calling");
#endif
   return 0;
}

// nothing XmlDoc::validateSchema(string xsd){}
static QoreValue XmlDoc_validateSchema_Vs(QoreObject* self, QoreXmlDocData* xd, const QoreListNode* args, q_rt_flags_t rtflags, ExceptionSink* xsink) {
    const QoreStringNode* xsd = HARD_QORE_VALUE_STRING(args, 0);
# 334 "QC_XmlDoc.qpp"
    #ifdef HAVE_XMLTEXTREADERSETSCHEMA
   xd->validateSchema(*xsd, xsink);
#else
   xsink->raiseException("MISSING-FEATURE-ERROR", "the libxml2 version used to compile the xml module did not support the xmlTextReaderSchemaValidate() function, therefore XmlDoc::validateSchema() is not available; for maximum portability, use the constant Option::HAVE_PARSEXMLWITHSCHEMA to check if this method is implemented before calling");
#endif
   return 0;
}

DLLLOCAL void preinitXmlDocClass() {
    QC_XMLDOC = new QoreClass("XmlDoc", "::Qore::Xml::XmlDoc", QDOM_DEFAULT);
    CID_XMLDOC = QC_XMLDOC->getID();
    QC_XMLDOC->setSystem();
}

DLLLOCAL QoreClass* initXmlDocClass(QoreNamespace& ns) {
    if (!QC_XMLDOC)
        preinitXmlDocClass();

    // XmlDoc::constructor(hash data, *hash opts) {}
    QC_XMLDOC->addConstructor(XmlDoc_constructor_VhNh, Public, QCF_NO_FLAGS, QDOM_DEFAULT, 2, hashTypeInfo, QORE_PARAM_NO_ARG, "data", hashOrNothingTypeInfo, QORE_PARAM_NO_ARG, "opts");

    // XmlDoc::constructor(string xml) {}
    QC_XMLDOC->addConstructor(XmlDoc_constructor_Vs, Public, QCF_NO_FLAGS, QDOM_DEFAULT, 1, stringTypeInfo, QORE_PARAM_NO_ARG, "xml");

    // XmlDoc::copy() {}
    QC_XMLDOC->setCopy((q_copy_t)XmlDoc_copy);

    // list XmlDoc::evalXPath(string xpath){}
    QC_XMLDOC->addMethod("evalXPath", (q_method_n_t)XmlDoc_evalXPath_Vs, Public, QCF_RET_VALUE_ONLY, QDOM_DEFAULT, listTypeInfo, 1, stringTypeInfo, QORE_PARAM_NO_ARG, "xpath");

    // *XmlNode XmlDoc::getRootElement(){}
    QC_XMLDOC->addMethod("getRootElement", (q_method_n_t)XmlDoc_getRootElement, Public, QCF_CONSTANT, QDOM_DEFAULT, QC_XMLNODE->getOrNothingTypeInfo());

    // string XmlDoc::getVersion(){}
    QC_XMLDOC->addMethod("getVersion", (q_method_n_t)XmlDoc_getVersion, Public, QCF_CONSTANT, QDOM_DEFAULT, stringTypeInfo);

    // hash XmlDoc::toQore(int pflags = XPF_PRESERVE_ORDER){}
    QC_XMLDOC->addMethod("toQore", (q_method_n_t)XmlDoc_toQore_Vi, Public, QCF_RET_VALUE_ONLY, QDOM_DEFAULT, hashTypeInfo, 1, bigIntTypeInfo, QoreSimpleValue().assign((int64)XPF_PRESERVE_ORDER), "pflags");

    // hash XmlDoc::toQoreData(*int pflags){}
    QC_XMLDOC->addMethod("toQoreData", (q_method_n_t)XmlDoc_toQoreData_Ni, Public, QCF_RET_VALUE_ONLY, QDOM_DEFAULT, hashTypeInfo, 1, bigIntOrNothingTypeInfo, QORE_PARAM_NO_ARG, "pflags");

    // string XmlDoc::toString(){}
    QC_XMLDOC->addMethod("toString", (q_method_n_t)XmlDoc_toString, Public, QCF_RET_VALUE_ONLY, QDOM_DEFAULT, stringTypeInfo);

    // nothing XmlDoc::validateDtd(string dtd){}
    QC_XMLDOC->addMethod("validateDtd", (q_method_n_t)XmlDoc_validateDtd_Vs, Public, QCF_NO_FLAGS, QDOM_DEFAULT, nothingTypeInfo, 1, stringTypeInfo, QORE_PARAM_NO_ARG, "dtd");

    // nothing XmlDoc::validateRelaxNG(string relaxng){}
    QC_XMLDOC->addMethod("validateRelaxNG", (q_method_n_t)XmlDoc_validateRelaxNG_Vs, Public, QCF_NO_FLAGS, QDOM_DEFAULT, nothingTypeInfo, 1, stringTypeInfo, QORE_PARAM_NO_ARG, "relaxng");

    // nothing XmlDoc::validateSchema(string xsd){}
    QC_XMLDOC->addMethod("validateSchema", (q_method_n_t)XmlDoc_validateSchema_Vs, Public, QCF_NO_FLAGS, QDOM_DEFAULT, nothingTypeInfo, 1, stringTypeInfo, QORE_PARAM_NO_ARG, "xsd");

    return QC_XMLDOC;
}
