/* ###
 * IP: GHIDRA
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * 
 *      http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package ghidra.pcode.opbehavior;

import java.math.BigInteger;

import org.junit.Assert;
import org.junit.Test;

import ghidra.pcode.floatformat.FloatFormat;
import ghidra.pcode.floatformat.FloatFormatFactory;

public class OpBehaviorFloatRoundTest extends AbstractOpBehaviorTest {

	public OpBehaviorFloatRoundTest() {
		super();
	}

	@Test
	public void testEvaluateBinaryLong() {

		OpBehaviorFloatRound op = new OpBehaviorFloatRound();

		FloatFormat ff = FloatFormatFactory.getFloatFormat(8);

		long a = ff.getEncoding(2.5);
		long result = op.evaluateUnary(8, 8, a);
		Assert.assertEquals(3.0, ff.decodeHostFloat(result), 0);

		a = ff.getEncoding(2.25);
		result = op.evaluateUnary(8, 8, a);
		Assert.assertEquals(2.0, ff.decodeHostFloat(result), 0);

		a = ff.getEncoding(2.75);
		result = op.evaluateUnary(8, 8, a);
		Assert.assertEquals(3.0, ff.decodeHostFloat(result), 0);

		a = ff.getEncoding(-2.5);
		result = op.evaluateUnary(8, 8, a);
		Assert.assertEquals(-2.0, ff.decodeHostFloat(result), 0);

		a = ff.getEncoding(-2.25);
		result = op.evaluateUnary(8, 8, a);
		Assert.assertEquals(-2.0, ff.decodeHostFloat(result), 0);

		a = ff.getEncoding(-2.75);
		result = op.evaluateUnary(8, 8, a);
		Assert.assertEquals(-3.0, ff.decodeHostFloat(result), 0);

		a = ff.getEncoding(Double.POSITIVE_INFINITY);
		result = op.evaluateUnary(8, 8, a);
		Assert.assertEquals(Double.POSITIVE_INFINITY, ff.decodeHostFloat(result), 0);

		a = ff.getEncoding(Double.NEGATIVE_INFINITY);
		result = op.evaluateUnary(8, 8, a);
		Assert.assertEquals(Double.NEGATIVE_INFINITY, ff.decodeHostFloat(result), 0);

		a = ff.getEncoding(Double.NaN);
		result = op.evaluateUnary(8, 8, a);
		Assert.assertEquals(Double.NaN, ff.decodeHostFloat(result), 0);
	}

	@Test
	public void testEvaluateBinaryBigInteger() {

		OpBehaviorFloatRound op = new OpBehaviorFloatRound();

		FloatFormat ff = FloatFormatFactory.getFloatFormat(8);

		BigInteger a = ff.getEncoding(ff.getBigFloat(2.5d));
		BigInteger result = op.evaluateUnary(8, 8, a);
		Assert.assertEquals(ff.getBigFloat(3.0d), ff.decodeBigFloat(result));

		a = ff.getEncoding(ff.getBigFloat(2.25d));
		result = op.evaluateUnary(8, 8, a);
		Assert.assertEquals(ff.getBigFloat(2.0d), ff.decodeBigFloat(result));

		a = ff.getEncoding(ff.getBigFloat(2.75d));
		result = op.evaluateUnary(8, 8, a);
		Assert.assertEquals(ff.getBigFloat(3.0d), ff.decodeBigFloat(result));

		a = ff.getEncoding(ff.getBigFloat(-2.5d));
		result = op.evaluateUnary(8, 8, a);
		Assert.assertEquals(ff.getBigFloat(-2.0d), ff.decodeBigFloat(result));

		a = ff.getEncoding(ff.getBigFloat(-2.25d));
		result = op.evaluateUnary(8, 8, a);
		Assert.assertEquals(ff.getBigFloat(-2.0d), ff.decodeBigFloat(result));

		a = ff.getEncoding(ff.getBigFloat(-2.75d));
		result = op.evaluateUnary(8, 8, a);
		Assert.assertEquals(ff.getBigFloat(-3.0d), ff.decodeBigFloat(result));

		a = ff.getBigInfinityEncoding(false);
		result = op.evaluateUnary(8, 8, a);
		Assert.assertEquals(ff.getBigInfinity(false), ff.decodeBigFloat(result));

		a = ff.getBigInfinityEncoding(true);
		result = op.evaluateUnary(8, 8, a);
		Assert.assertEquals(ff.getBigInfinity(true), ff.decodeBigFloat(result));

		a = ff.getBigNaNEncoding(false);
		result = op.evaluateUnary(8, 8, a);
		Assert.assertEquals(ff.getBigNaN(false), ff.decodeBigFloat(result));

	}

}
