/*+		F P I L	

 * Module name:
     FPIL

 * Function:
     Fibre Positioner Instrument Library

 * Description:
     Provides a set of routines common to all fibre positioner instrumetns.

 * Language:
      C

 * Support: Tony Farrell, AAO

 *-

 * Sccs Id:     fpil.c, Release 1.1, 01/30/01

 * History:
     06-Dec-1999 - TJF - Original version
     28-Jan-1999 - TJF - Pass both unsquared and squared tolerance values
                         to each called routine.
     09-Mar-2000 - TJF - Add FpilGetMaxExtension as configure really
                         needs it rather then being able to use the
                         check routines.
     13-Mar-2000 - TJF - Add FpilGetField radius.
     15-Mar-2000 - TJF - Add FpilGetFieldRadiusIncPark function and
                         FieldRadIncPark item to structure.
     16-Mar-2000 - TJF - Add FpilDrawButFib function and DrawButFib
		         to structure.
     17-Mar-2000 - TJF - Add NumFids to structure and FpilGetNumFids().
     20-Mar-2000 - TJF - Include fpil_err.h.
     21-Aug-2000 - KS  - Make INLINE definition contingent on its not
                         already being defined.
     21-Dec-2002 - RSC - fibreType arg. added to FpilColInvPos().
 */
/*
 *  Include files
 */
#include <stdlib.h>
#include <stdio.h>
#include <math.h>
#include "fpil.h"
#include "fpil_err.h"

/* 
 * Constant definitions
 */

/*
 * Macro function definitions
 */
#define SQRD(x)           ((double)(x)*(double)(x))  

#ifndef INLINE
#   if defined(__GNUC__) || defined(GNUC)
#      define INLINE __inline__ static
#   else
#      define INLINE static
#   endif
#endif

/*#define FPIL_DEBUG 1*/
#ifdef FPIL_DEBUG
#warning "FPIL Debug enabled, checks will run much slower"
#endif
/*
 * type definitions
 */


typedef struct {
    long X1;
    long X2;
    long Y1;
    long Y2;
} BoundingBox;


/*
 * Functions imported (but not declared in include files)
 */




/*
 * Variables private to this module (static variables)
 */



/*
 * Function Definitions
 *
 */


/*
 * Internal Fuction name:
      FibreBox
  
 * Description:
      Constructs and returns a box about a fibre. 
  
 * History:
     06-Dec-1999 - TJF - Original version
  
 */
INLINE void FibreBox(
    const Fpil___InstType *inst,
    long fvpX,          /* The fibre button end - X ordinate */
    long fvpY,          /* The fibre button end - Y ordinate */
    long pivX,          /* The fibre pivot end  - X ordinate */
    long pivY,          /* The fibre pivot end  - Y ordinate */
    BoundingBox *box)
{
    if (pivX > fvpX)
    {
        box->X1 = fvpX - inst->maxButSize;
        box->X2 = pivX + inst->maxButSize;
    }
    else
    {
        box->X1 = pivX - inst->maxButSize;
        box->X2 = fvpX + inst->maxButSize;
    }
        
    if (pivY > fvpY)
    {
        box->Y1 = fvpY - inst->maxButSize;
        box->Y2 = pivY + inst->maxButSize;
    }
    else
    {
        box->Y1 = pivY - inst->maxButSize;
        box->Y2 = fvpY + inst->maxButSize;
    }
        
}
/*
 * Internal Fuction name:
      ButtonBox
  
 * Description:
      Constructs and returns a box about a fibre button. 
  
 * History:
     06-Dec-1999 - TJF - Original version
  
 */
INLINE void ButtonBox(
    const Fpil___InstType *inst,
    long x,          /* The fibre button end - X ordinate */
    long y,          /* The fibre button end - Y ordinate */
    BoundingBox *box)
{
    box->X1 = x - inst->maxButSize;
    box->X2 = x + inst->maxButSize;
    box->Y1 = y - inst->maxButSize;
    box->Y2 = y + inst->maxButSize;
}

/*
 * Internal Fuction name:
      BoxCollision
  
 * Description:
      Returns ture if two boxes might overlap
  
 * History:
     06-Dec-1999 - TJF - Original version
  
 */
INLINE int BoxCollision(
    const BoundingBox *box1,
    const BoundingBox *box2)
{
    int safe = 0;
   
    if (box1->X2 < box2->X1)
        safe = 1;                /* Box 1 is too far left of box 2 to overlap*/
    else if (box2->X2 < box1->X1)
        safe = 1;                /* Box 2 is too far left of box 1 to overlap*/
    if (box1->Y2 < box2->Y1)
        safe = 1;                /* Box 1 to too much lower of box 2 */
    else if (box2->Y2 < box1->Y1)
        safe = 1;                /* Box 2 is too much lower of box 1 */
             
    return !safe;                
}
/*
 * Internal Fuction name:
      FibreCollision
  
 * Description:
      Returns true if two fibres collied
  
 * History:
     13-Dec-1999 - TJF - Original version, based on 2dF tdFcollisionFibFib
                         routine, original written by JOS but modified
                         by JDW and TJF.
     04-May-2000 - TJF - Code was not correct for 6dF case, need to ensure
                         we check for the other end of the fibre.
  
 */
INLINE int FibreCollision(
        double  xapiv,
        double  yapiv,
        double  xa,
        double  ya,
        double  xbpiv,
        double  ybpiv,
        double  xb,
        double  yb)
{
/*
 *  We are trying to determine if there is an intercept between
 *  two line segements. Our approach is to rotate the origin
 *  such that fibre A lies on the X axies.  This makes the caculations
 *  easier
 */
    double  dxa, dya;
    double  ssa, sa;
    double  sina, cosa;
    double  xtb, ytb;
    double  xpb, ypb;
    double  xBr, yBr;
    double  xBPr, yBPr;
    double m;
    double xCross;

/*
 *  Detemine rotation required to put fibre A on X axis.
 */
    dxa  = xapiv - xa;
    dya  = yapiv - ya;
    ssa  = dxa*dxa + dya*dya;
    sa   = sqrt(ssa);
    cosa = dxa/sa;
    sina = dya/sa;

/*
 *  Determine coorindates of fibre B assuming fibre A starts at origin.
 */
    xtb = xb - xa;
    ytb = yb - ya;
    xpb = xbpiv - xa;
    ypb = ybpiv - ya;

/*
 *  Rotate fibre B coordinates such fibre A can be assumed to line on 
 *  the X axis.
 */
    xBr  = xtb*cosa + ytb*sina;
    yBr  = ytb*cosa - xtb*sina;
    xBPr = xpb*cosa + ypb*sina;
    yBPr = ypb*cosa - xpb*sina;

/*
 *  If fibre B is either fully above or fully below the X axis, there
 *  can be no collision.
 */
    if (yBr*yBPr > 0.0)                      
        return (0);
/*
 *  At this point, one end of fibre B is above and one below the X axis.
 *
 *  If both X components of fibre B are to the left of fibre A, then
 *  there can be no collision
 */ 
    else if (xBr > sa && xBPr > sa)             
        return (0); 
/*
 *  Now if both X components of fibre B are greter then zero, then 
 *  it must be the case the there is a collision
 */
    else if (xBr > 0.0 && xBPr > 0.0)             
        return (1);
/*
 *  Both X ends of fibre B are to the left of the Y axis.  No collision.
 */
    else if (xBr < 0.0 && xBPr < 0.0)             
        return (0);
/*
 *  At this point fibre B passes from quarter 1 to 3 or quarter 2 to 4 (or 
 *  vis-a-versa).
 *
 *  It intercepts with A only if the point at which is crosses the
 *  X axis (Y = 0) happens with the value of X positive and less then 
 *  the length of fibre A.   The point at which it crosses the X axis is 
 *  given by
 *
 *  X = x2 - y2 / m
 *
 *  This is positive if  x2 > y2/m, hence
 */
    m = (xBPr - xBr)/(yBPr - yBr);
    xCross = xBr - yBr*m;
    if ((xCross > 0)&&(xCross < sa))
        return (1);
/*
 *  No collision can occur
 */
    return(0);

}

/*
 * Internal Fuction name:
      FiducialFibreCollision
  
 * Description:
      Returns true if a fibre obscures a fiducial.
  
 * History:
     13-Dec-1999 - TJF - Original version, based on 2dF tdFcollisionMarkFib
                         routine, original written by JDW but modified
                         by TJF.
  
 */
INLINE int FiducialFibreCollision(
        long int  end1x,
        long int  end1y,
        long int  end2x,
        long int  end2y,
        long int  markx,
        long int  marky,
        double    tolerance,
        double    tolSqrd)
{
/*
 *  a is the length of the fibre
 *  b is the distance from the fiducial to the pivot
 *  c is the distance from the fiduical to the fibre end.
 */

    double a = SQRD((double)(end1x - end2x)) + SQRD((double)(end1y - end2y));
    double b = SQRD((double)(end1x - markx)) + SQRD((double)(end1y - marky));
    double c = SQRD((double)(end2x - markx)) + SQRD((double)(end2y - marky));
/*
 *  That if the distance from the pivot to the fiducial is longer than the
 *  fibre, then there can be no collision.  Likewise for the fibre end
 *  to the fiducial.
 */
    if (((a+tolSqrd) <= b) || ((a+tolSqrd) <= c))
        return 0;
    
    else if (end1x == end2x)
    {
/*
 *      Fibre lays along the Y axis  (the next bit of code would get
 *      a divide by zero error) and y of point is within the length
 *      of the fibre (ensure by the above check) 
 */
        long int d = markx - end2x;
        return ((d <= tolerance)? 1: 0);        
    }
    else
    {        
/*
 *      We can now use code which determines the distance between a line
 *      and a point since
 *        1.  The first check ensures a perpendicular from line to the
 *            point is within the line-segement of concern
 *        2.  end2x-end1x != 0
 *
 *     Given m is the gradient of the line between the fibre ends and
 *     b is the intercept, then, d the distance between the point x1,y1 and
 *     the line is given by
 *
 *      d = |  m.x1 - y + b    |   (Note, absolute value only)
 *          |------------------|
 *          | +/- sqrt(m^2 + 1)|
 *
 *
 *      Determine m and b.
 */
        double m = (double)(end2y - end1y)/(double)(end2x - end1x);
        double b = end2y - m*end2x;
        
/*
 *      Determine the denominator and then the numerator.  Note, sqrt()
 *      returns the positive square root only.
 */
        double dd = sqrt(m*m + 1);
        double dn = m * markx - marky + b;
        
        double d;
/*
 *      Determine the distance.
 */
        if (dn < 0)
            d = dn * -1 / dd;
        else
            d = dn / dd;         
        
        return ((d <= tolerance) ? 1: 0);
    }
    
}

/*
 *+			F p i l I n i t

 * Function name:
      FpilInit

 * Function:
      Initialise an Fpil variable.

 * Description:
      Supplies the details required to initialise an Fpil variable

 * Language:

      C

 * Call:
        (int) = FpilInit(clientData, version, routines, params, modelRoutines,
                        inst)

 * Parameters:   (">" input, "!" modified, "W" workspace, "<" output)
      (>) clientData (void *) Passed drectly to each user supplied routine when
                         it is invoked (as it's first argument).
      (>) version  (int) The FPI version number the caller was compiled
                         with.  You should allways specify this
                         as FPIL_VERSION.  It is assumed the source which
                         delcares the routines and params variables were
                         compiled at the same time.
      (>) routines (const FpilCollisionChecksType *) Contains pointers to
                         routines which perform various checks for the
                         instrument in question.  See below.
      (>) params   (const FpilInstParamsType *) Contains initial values for
                         various instrument parameters.  See below.
      (>) modelRoutines (const FpilModelRoutinesType *) Contains pointers to
                         routines use to implement the telescope model. 
                         see below.
      (<) inst     (FpilType *) The instrument description created by this
                         routine for use in other calls to Fpil routines.
                        

 * Returned value:
      Return 1 on success or 0 on error.  If an error occurs, more information
      is written to stderr.

 * FpilCollisionChecksType contents:
 
      The "routines" variables provides routines which do the required
      checks for the instrument in question.  In each case, the argument
      lists are minor varitions of the corresponding FpilCol call, normally
      with the exception that the FpilInstType argument is replaced by
      a void * type used to pass the clientData value and that the
      last argument is sometimes a clearance value to be used in the
      check.  The required routines are.

      ColButFib => A routine of type FpilColButFibRoutineType (see fpil.h)
                   which checks for collisions between a button and a fibre
      ColButBut => A routine of type FpilColButButRoutineType (see fpil.h)
                   which checks for collisions between two buttons.
      ColFidBut => A routine of type FpilColFidButRoutineType (see fpil.h)
                   which checks for collisions between a fiducial mark and
                   a button.
      ColInvPos => A routine of type FpilColInvPosRoutineType (see fpil.h)
                   which checks for an invalid plate position.  This is the
                   only one of these routines which is optional
      GetVirPiv => A routine of type FpilVirPivRoutineType (see fpil.h)
                   Given the position of a button - ie the x,y position of 
                   its origin, which is the position of the fibre end, and 
                   its theta value (the angle measured anticlockwise from the 
                   vertical through that origin to the fibre) returns
                   the coordinates of the virtual pivot point - the point at 
                   the end where the fibre is assumed to emerge.
      AccessCons => A routine of type FpilConsAccessRoutineType (see fpil.h)
                   used to access the instrument constant detail.  See
                   fpilcons.c for details on what must be in this.  Can
                   be specified as zero, which will cause FpilConsAccess()
                   to fail.  

      WhichSpec  => A routine of type FpilWhichSpecRoutineType (see fpil.h)
                   used to determine the spectrograph associated with
                   a pivot.  Can be specified as zero in which case
                   FpilConWhichSpec will fail.  

      DrawButFib => A routine of type FpilDrawRoutineType (see fpil.h)
                   used to provide information which allows an image of 
                   the button and fibre to be drawn on a Tcl/Tk canvas widget.
                   Can be specified az zero, which will cause FpilDrawButFib()
                   to fail. 

 * FpilInstParamsType contents:
      This parameter is used to set various values which describe details
      of the instrument, being

      ButtonClear => Min clearance between buttons (unsigned long, microns)
      FibreClear  => Min button/fibre clearance (unsigned long, microns)
      FidClear    => Min Clearance for a fiducal mark (unsigned long, microns)
      FieldRad    => Field radius (unsigned long, microns) outside which
                     fibres cannot be placed for observing (but they may
                     be parked outside this radius)
      FieldRadIncPark => Field radius (unsigned long, microns) outside
                     which fibres may never be placed, even when parked.
      Extension   => Max fibre extension (unsigned long, microns)
      MaxPivAngle => Max allow piv/fibre angle (double, radians)
      MaxFibAngle => Max allowed fibre/button angle (double, radians)
        
      FibAngleVar => Logical. If true, the fibre/button angle 
                                   can be varied to the limit MaxFibAngle.  
                                   Otherwise, the button rotates such that 
                                   this is always zero.                  

      MaxSize     => Maximum size of button in any planner 
                                   dimension (unsigned long, microns)
      NumFields   => Number of field plates (unsigned int). Note that
                     fpil.h provides a constant - FPIL_MAXFIELD (4)
                     which this value should not exceed (to allow programs to
                     statically defined arrays of things.).  
      NumPivots   => Number of pivots on a field plate (unsigned int).  Note
                     that fpil.h provides a constant - FPIL_MAXPIVOTS (1000), 
                     which this value should not exceed (to allow programs to
                     statically defined arrays of things.).  
      NumFids     => The number of fiducials on each plate (unsigned int). Note
                     that fpil.h provides a constant - FPIL_MAXFIDS (40), 
                     which this value should not exceed (to allow programs to
                     statically defined arrays of things.).  
      ParkMayCollide =>  Logical.  If true, then when fibres are
                     parked, they may collide with other fibres.
      Telescope   => Telescope name eg. AAT, UKST (char [20]).  This name
                     should be acceptable by the SLA library
                     sla_OBS routine.
      Instrument  => Instrument name eg 2dF, 6dF etc (char [20]).
      NumSpec     => Number of Spectrographs.  Only required if the
                     WhichSpec function, above, is defined.
      SpecNames   => Names of the spectrographs.  Only required if the
                     WhichSpec function, above, is defined.
                     This should be a pointer to an array of char pointers,
                     declared "const char * const array[]"
                     This should be equivalent to the names which might
                     be returned by the WhichSpec function in its SpecName
                     argument.

 * FpilInstModelRoutinesType contents:

    This parameter provides various routine used to access and implement
    the telescope model.

      Access => A routine of type FpilModelAccessRoutineType (see fpil.h)
                This is used to access the telescope model for a specified
                field.  This routine should access the model and 
                malloc and return the address of a structure of type
                FpilModelType.  By actually mallocing a different structure
                which contains as its first item a structure of type
                FpilModelType, the telescope model implementation routines
                can keep there own information in this structure.  It just
                casts to and from FpilModelType as required.

      Free => A routine of type FpilModelFreeRoutineType (see fpil.h)
                Used to close access to a telescoep model.  The
                telescope model data structure should be freeed.  This one
                is optional.  If not supplied, FpilModelFree(3) will
                simply call free(3) on the model.

      CvtInit => A routine of type FpilModelXyInitRoutineType (see fpil.h)
                This is used to initialize the telescope model for
                conversions at a given time , telescope model etc.  The
                FpilModel wrapper of this is defined such that it 
                may be called mutiple times for a model access and must
                be called at least once before the conversion routines
                are invoked.

       Rd2Xy  => A routine of type FpilModelRd2XyRoutineType (see fpil.h)
                Converts from RA/DEC to field X/Y.
       Xy2Pos => A routine of type FpilModelXy2PosRoutineType (see fpil.h)
                Converts from field X/Y to positioner X/Y
       Xy2Rd  => A routine of type FpilModelXy2RdRoutineType (see fpil.h)
                Converts from field X/Y to RA/DEC
       Pos2Xy => A routine of type FpilModelPos2XyRoutineType (see fpil.h)
                Converts from positioner X/Y to field X/Y.
       QuickAppToObs => A routine of type FpilModelQAORoutineType (see fpil.h)
                which does a quick conversion from apparent place to
                observed place using the telescope model.  Typically,
                the SLA routine slaAopqk() might be used to perform
                this conversion.
       NumberParams => The number of telescope model parameters required
                by the CvtInit routine.
       ParamNames => Simple symbolic names of the telescope model ParamDescrs.
                     This should be a pointer to an array of char pointers,
                     declared "const char * const array[]"
       ParamDescrs => Short descriptions of the telescope model parameters
                     This should be a pointer to an array of char pointers,
                     declared "const char * const array[]"
       ParamDefaults => Default values of each of the telescope
                      model parameters.
                     This should be a pointer to an array of doubles.

                

 * Include files: fpil.h

 * See Also: {references}

 * External functions used:
      malloc  (CRTL) Allocated memory.
      fprintf (CRTL) Used to write error messages to stderr.


 * Support: Tony Farrell, AAO

 *-

 * History:
     06-Dec-1999 - TJF - Original version
     13-Dec-1999 - TJF - I realized that fibre-fibre and fibre-fiducial
                         collisions are not instrument specific, so I
                         have moved the 2dF code into this module and
                         documented it better, removed the requirement
                         for the user to supply those routines.
 */




extern int FpilInit(
    void *clientData,
    int version,
    const FpilCollisionChecksType * routines,
    const FpilInstParamsType      * params,
    const FpilModelRoutinesType   * modelRoutines,
    FpilType *inst)
{
    Fpil___InstType *p;

/*
 *  We don't know how to deal with a version greater then this one.
 */
    if (version > FPIL_VERSION)
    {
        fprintf(stderr,
         "FpilInit:Caller version (%d) is greater then library version (%d)\n",
                version, FPIL_VERSION);
        return 0;
    }
    if ((!routines)||(!params))
    {
        fprintf(stderr,"FpilInit:Either routines or params argument null\n");
        return 0;
    }

    if ((!routines->ColButFib)||
        (!routines->ColButBut)||
        (!routines->ColFidBut)||
        (!routines->GetVirPiv))
    {
        fprintf(stderr,"FpilInit:Not all required routines specfied\n");
        return 0;
    }

    if ((p = malloc(sizeof(Fpil___InstType))) == 0)
    {
        fprintf(stderr,"FpilInit:Malloc failure\n");
        return 0;
    }
    p->clientData = clientData;
    p->routines = *routines;
    p->params   = *params;
/*
 *  Telescope model is optional
 */
    if (modelRoutines)
    {
        p->hasModel = 1;
        p->model    = *modelRoutines;
    }
    else
    {
        p->hasModel       = 1;
        p->model.Access  = 0;
        p->model.Free    = 0;
        p->model.CvtInit = 0;
        p->model.Rd2Xy   = 0;
        p->model.Xy2Pos  = 0;
        p->model.Xy2Rd   = 0;
        p->model.Pos2Xy  = 0;
    }
/*
 *  Calculate some values we use a lot
 */
    p->butTolSqrd   = SQRD(params->ButtonClear);
    p->fidTolSqrd   = SQRD(params->FidClear);
    p->fibreTolSqrd = SQRD(params->FibreClear);
    p->fieldRadSqrd = SQRD(params->FieldRad);
    p->extenSqrd    = SQRD(params->Extension);
    p->maxButSize   = params->ButtonClear +
                      params->FibreClear +
                      params->MaxSize;

/*
 *  If the fibre/button angle cannot be varied, set the max bend angle
 *  to zero.
 */
    if (!params->FibAngleVar)
        p->params.MaxFibAngle = 0;

/*
 *  It is not yet clear what we should do with the version value.
 */
    p->version  = version;
/*
 *  Return the instrument description as the dummy type FpilType.  This
 *  stops the caller being able to acess our data.
 */ 
    *inst = (FpilType)(p);

    return (1);
}


/*
 *+			F p i l F r e e

 * Function name:
      FpilFree

 * Function:
      Free data structure allocated by FpilInint

 * Description:
      Just frees the allocated memory.

 * Language:
      C

 * Call:
      (void) FpilInit(inst)

 * Parameters:   (">" input, "!" modified, "W" workspace, "<" output)
      (>) inst (FpilType) The Fpil Instrument data structure to free.

 * Include files: fpi.h

 * See Also: {references}

 * External functions used:
      free (CRTL) Free memory.

 * Support: Tony Farrell, AAO

 *-

 * History:
     06-Dec-1999 - TJF - Original version
 */
extern void FpilFree(FpilType inst)
{
    free(inst);
}


/* Collision routines */

/*
 *+			F p i l C o l F i b F i b

 * Function name:
      FpilColFibFib

 * Function:
      Checks for a collision in the xy plane between two fibres

 * Description:
      If a collision is detected the function will return true.  This
      routine is used to determine if two fibres cross and does not
      consider the width of the fibres involved.

      This routine first creates bounding boxes around the fibres and
      only uses the underlying collision algrothim if those boxes
      collide.  Thus, the underlying routine need not bother with
      this type of simple test.

 * Language:
      C

 * Call:
      (int) = FpilColFibFib (inst, pivAX,pivAY,fvpAX,fvpAY,
                                   pivBX,pivBY,fvpBX,fvpBY)

 *  Parameters:  (">" input, "!" modified, "W" workspace, "<" output)
      (>) inst            (FpilType)  The instrument as returned 
                                      by FpilInit(3).
      (>) pivAX           (double)    Fibre A Y ordinate.
      (>) pivAY           (double)    Fibre A Y ordinate.
      (>) fvpAX           (double)    Fibre A virtual pivot X ordinate.
      (>) fvpAY           (double)    Fibre A virtual pivot Y ordinate.
      (>) pivBX           (double)    Fibre B Y ordinate.
      (>) pivBY           (double)    Fibre b Y ordinate.
      (>) fvpBX           (double)    Fibre B virtual pivot X ordinate.
      (>) fvpBY           (double)    Fibre B virtual pivot Y ordinate.


 * Returned value:
      true if a collision is detected.

 * Include files: fpi.h

 * See Also: {references}

 * Prior requirements:
      FpilInit(3) must have been invoked for the FpilType variable.

 * Support: Tony Farrell, AAO

 *-

 * History:
     06-Dec-1999 - TJF - Original versio
     13-Dec-1999 - TJF - Since fibre collisions are indepent of instrument,
                         use our own version of the routine instead of
                         an instrument specific version.
 */

extern int  FpilColFibFib (
        const FpilType instrument,
        double  xapiv,
        double  yapiv,
        double  xa,
        double  ya,
        double  xbpiv,
        double  ybpiv,
        double  xb,
        double  yb)
{
    Fpil___InstType *inst = (Fpil___InstType *)instrument;

/*
 *  Create boxes about the fibres.
 */
    BoundingBox a;
    BoundingBox b;
    FibreBox(inst, xapiv,yapiv, xa, ya, &a);
    FibreBox(inst, xbpiv,ybpiv, xb, yb, &b);

/*
 *  Only if these boxes collide to be both trying the (expensive)
 *  detailed check routine.
 */
    if (BoxCollision(&a, &b))
        return FibreCollision(xapiv, yapiv, xa, ya, xbpiv, ybpiv, xb, yb);

    else
    {
/*
 *      If debuging, always all the underlying routine.
 */
#       ifdef FPIL_DEBUG
        {
            if (FibreCollision(xapiv, yapiv, xa, ya, xbpiv, ybpiv, xb, yb))
            {
                fprintf(stderr,
"FPIL ColFibFib error, bounding boxes don't collide but fibres do\n");
                return 1;
            }
        }
#       endif
        return 0;
        
    }
}

/*
 *+			F p i l C o l B u t F i b

 * Function name:
      FpilColButFib

 * Function:
      Checks for a collision in the xy plane between a button and a fibre.

 * Description:
      If a collision is detected the function will return true

      This routine first creates bounding boxes around the fibre and
      button and only uses the underlying collision routine if those boxes
      collide.  Thus, the underlying routine need not bother with
      this type of simple test.

      Invokes the instrument specific  button to fibre collision software
      using the specified arguments.

 * Language:
      C

 * Call:
      (int) = FpilColButFib (inst, butX, butY, theta, fvpX, fvpY, 
                             pivX, pivY)

 *  Parameters:  (">" input, "!" modified, "W" workspace, "<" output)
      (>) inst            (FpilType)  The instrument as returned 
                                      by FpilInit(3).
      (>) butX            (double)    Button X ordinate.
      (>) butY            (double)    Button Y ordinate.
      (>) theta           (double)    Button theta.
      (>) fvpX            (double)    Fibre virtual pivot X ordinate.
      (>) fvpY            (double)    Fibre virtual pivot Y ordinate.
      (>) pivX            (double)    Fibre pivot X ordinate.
      (>) pivY            (double)    Fibre pivot Y ordinate.


 * Returned value:
      true if a collision is detected.

 * Include files: fpi.h

 * See Also: {references}

 * Prior requirements:
      FpilInit(3) must have been invoked for the FpilType variable.

 * Support: Tony Farrell, AAO

 *-

 * History:
     06-Dec-1999 - TJF - Original version
 */

extern int  FpilColButFib (
        const FpilType instrument,
        double    butX,
        double    butY,
        double    theta,
        double    fvpX,
        double    fvpY,
        double    pivX,
        double    pivY)
{
    Fpil___InstType *inst = (Fpil___InstType *)instrument;

    BoundingBox button;
    BoundingBox  fibre;

    ButtonBox(inst, butX, butY, &button);
    FibreBox (inst, fvpX, fvpY, pivX, pivY, &fibre);
    
/*
 *  Only if these boxes collide to be both trying the (expensive)
 *  detailed check routine.
 */
    if (BoxCollision(&button, &fibre))
        return (*inst->routines.ColButFib)(inst->clientData,
                                       butX, butY, theta,
                                       fvpX, fvpY,
                                       pivX, pivY,
                                       inst->params.FibreClear,
                                       inst->fibreTolSqrd);
    else
    {
/*
 *      If debuging, always all the underlying routine.
 */
#       ifdef FPIL_DEBUG
        {
            if ((*inst->routines.ColButFib)(inst->clientData,
                                       butX, butY, theta,
                                       fvpX, fvpY,
                                       pivX, pivY,
                                       inst->params.FibreClear,
                                       inst->fibreTolSqrd))
            {
                fprintf(stderr,
"FPIL ColButFib error, bounding boxes don't collide but button/fibre does\n");
                return 1;
            }
        }
#       endif
        return 0;
    }

}


/*
 *+			F p i l C o l B u t B u t

 * Function name:
      FpilColButBut

 * Function:
      Checks for a collision in the xy plane between two buttons.

 

 * Description:
      If a collision is detected the function will return true
      
      This routine first creates bounding boxes around the buttons and
      only uses the underlying collision routine if those boxes
      collide.  Thus, the underlying routine need not bother with
      this type of simple test.
 
     Invokes the instrument specific  button to button collision software
      using the specified arguments.

 * Language:
      C

 * Call:
      (int) = FpilColButBut (inst,butXa,butYa,thetaa,butXb,butYb,thetab)

 *  Parameters:  (">" input, "!" modified, "W" workspace, "<" output)
      (>) inst            (FpilType)  The instrument as returned 
                                      by FpilInit(3).
      (>) butXa   (double)    Button A  X ordinate.
      (>) butYa   (double)    Button A  Y ordinate.
      (>) thetaa   (double)    Button A  theta.
      (>) butXb   (double)    Button B  X ordinate.
      (>) butYb   (double)    Button B  Y ordinate.
      (>) thetab   (double)    Button B  theta.

 * Returned value:
      true if a collision is detected.

 * Include files: fpi.h

 * See Also: {references}

 * Prior requirements:
      FpilInit(3) must have been invoked for the FpilType variable.

 * Support: Tony Farrell, AAO

 *-

 * History:
     06-Dec-1999 - TJF - Original version
 */
extern int  FpilColButBut (
        const FpilType instrument,
        double    butXa,
        double    butYa,
        double    thetaa,
        double    butXb,
        double    butYb,
        double    thetab)
{
    Fpil___InstType *inst = (Fpil___InstType *)instrument;

    BoundingBox a;
    BoundingBox b;
    ButtonBox(inst, butXa, butYa, &a);
    ButtonBox(inst, butXb, butYb, &b);

    
    if (BoxCollision(&a, &b))
        return (*inst->routines.ColButBut)(inst->clientData,
                                       butXa, butYa, thetaa,
                                       butXb, butYb, thetab,
                                       inst->params.ButtonClear,
                                       inst->butTolSqrd);
    else
    {
/*
 *      If debuging, always all the underlying routine.
 */
#       ifdef FPIL_DEBUG
        {
            if ((*inst->routines.ColButBut)(inst->clientData,
                                       butXa, butYa, thetaa,
                                       butXb, butYb, thetab,
                                       inst->params.ButtonClear,
                                       inst->butTolSqrd))
            {
                 fprintf(stderr,
"FPIL ColButBut error, bounding boxes don't collide but buttons do\n");
                return 1;
            }
        }
#       endif
        return 0;
        
        
    }

}



/*
 *+			F p i l C o l F i d u c i a l

 * Function name:
      FpilColFiducial

 * Function:
      Checks for a collision in the xy plane between a fibre/fibre button 
      and a fiducial.


 * Description:
      If a collision is detected the function will return true

      Invokes an internal fiducal-fibre collision detection routine
      (since that is not instrument specific, other then the
      tolerance).  It then invokes the instrument specific  fiducial to 
      button collision software using the specified arguments.

      Note that the reason for having fibre end points for both ends and
      the button coordinated is that the fibre end point at the button
      end and the button position are not necessarly the same point.

 * Language:
      C

 * Call:
      (int) = FpilColFiducial (inst,butx, buty, theta, 
                               end1x, end1y, 
                               end2x, end2y,
                               markx, marky)

 *  Parameters:  (">" input, "!" modified, "W" workspace, "<" output)
      (>) inst            (FpilType)  The instrument as returned 
                                      by FpilInit(3).
      (>) butx            (long int)   Fibre Button X ordinate.
      (>) buty            (long int)   Fibre Button Y ordinate.
      (>) theta           (double)     Fubre button theta orientation.
      (>) end1x           (long int)   Fibre end 1 X ordinate.
      (>) end1y           (long int)   Fibre end 1 Y ordinate.
      (>) end2x           (long int)   Fibre end 2 X ordinate.
      (>) end2y           (long int)   Fibre end 2 Y ordinate.
      (>) markx           (long int)   Fiducial X ordinate
      (>) marky           (long int)   Fiducial Y ordinate

 * Returned value:
      true if a collision is detected.

 * Include files: fpi.h

 * See Also: {references}

 * Prior requirements:
      FpilInit(3) must have been invoked for the FpilType variable.

 * Support: Tony Farrell, AAO

 *-

 * History:
     06-Dec-1999 - TJF - Original version
 */
extern int  FpilColFiducial (
        const FpilType instrument,
        long int  butx,
        long int  buty,
        double    theta,
        long int  end1x,
        long int  end1y,
        long int  end2x,
        long int  end2y,
        long int  markx,
        long int  marky)
{
/* 
 *  Combines Fiducial-Fibre and Fiducal-Button in one operation.
 */
    Fpil___InstType *inst = (Fpil___InstType *)instrument;

    if (FiducialFibreCollision(end1x, end1y, end2x, end2y,
                               markx, marky,
                               inst->params.FidClear,
                               inst->fidTolSqrd))
        return 1;

   return (*inst->routines.ColFidBut)(inst->clientData,
                                      butx, buty, theta, 
                                      markx, marky,
                                      inst->params.FidClear,
                                      inst->fidTolSqrd);
           
}


/*
 *+			F p i l C o l I n v P o s

 * Function name:
      FpilColInvPos

 * Function:
      Checks for an invalid position on the field plate 


 * Description:
      Used to check for things such as a collision with screws holding the 
      field plate on.

      Returns true if the position is invalid.

      Invokes the instrument specific  invalid position collision software
      using the specified arguments.

 * Language:
      C

 * Call:
      (int) = FpilColInvPos (inst,fibreType,butx,buty,theta)

 *  Parameters:  (">" input, "!" modified, "W" workspace, "<" output)
      (>) inst            (FpilType)  The instrument as returned 
                                      by FpilInit(3).
      (>) fibreType       (int)       Fibre type.
      (>) plate           (unsigned)  The plate to check for.
      (>) butx            (long int)  Button X ordinate.
      (>) buty            (long int)  Button Y ordinate.
      (>) theta           (double)    Button orientation.

 * Returned value:
      true if a collision is detected.

 * Include files: fpi.h

 * See Also: {references}

 * Prior requirements:
      FpilInit(3) must have been invoked for the FpilType variable.

 * Support: Tony Farrell, AAO

 *-

 * History:
     06-Dec-1999 - TJF - Original version
     19-Mar-2000 - TJF - Add plate argument.
     21-Dec-2002 - RSC - fibreType argument added.
 */
extern int  FpilColInvPos (
        const FpilType instrument,
        unsigned plate,
	int       fibreType,
        long int  butx,
        long int  buty,
        double    theta)
{
    Fpil___InstType *inst = (Fpil___InstType *)instrument;
    
/*
 *  There does not have to be an invalid position detection routine.
 */
    if (inst->routines.ColInvPos)
        return (*inst->routines.ColInvPos)(inst->clientData,
                                           plate,
					   fibreType,
                                           butx,  buty, theta,
                                           inst->params.ButtonClear,
                                           inst->butTolSqrd);
    else
        return 0;

}

/*
 *+			F p i l G e t V i r t P i v

 * Function name:
      FpilGetVirtPiv

 * Function:
       Return the virtual pivot position of the specified fibre.

 * Description:
       Given the position of a button - ie the x,y position of its origin, 
       which is the position of the fibre end, and its theta value (the angle 
       measured anticlockwise from the vertical through that origin to the 
       fibre) returns the coordinates of the virtual pivot point - the point 
       at the end where the fibre is assumed to emerge.

       Invokes the instrument specific virtual pivot point routine
       using the specified arguments.

 * Language:
      C

 * Call:
      (void) = FpilGetVirtPiv(inst, originX, originY, theta, 
                              virtPivX, virtPivY)

 *  Parameters:  (">" input, "!" modified, "W" workspace, "<" output)
      (>) inst            (FpilType)  The instrument as returned 
                                      by FpilInit(3).
      (>) originX         (double)    Position of fibre end - X ordinate, 
                                      microns
      (>) originY         (double)    Position of fibre end - Y ordinate,
                                      microns
      (>) theta           (double)    Button theta - radians
      (<) virtPivX        (double *)  The virtual fibre end - X ordinate.
      (<) virtPivy        (double *)  The virtual fibre end - Y ordinate.

 * Include files: fpi.h

 * See Also: {references}

 * Prior requirements:
      FpilInit(3) must have been invoked for the FpilType variable.

 * Support: Tony Farrell, AAO

 *-

 * History:
     06-Dec-1999 - TJF - Original version
 */
extern void   FpilGetVirtPiv (const FpilType instrument,
                              double OriginX,
                              double OriginY,
                              double Theta,
                              double *VirtPivX,
                              double *VirtPivY)
{
    Fpil___InstType *inst = (Fpil___InstType *)instrument;
    (*inst->routines.GetVirPiv)(inst->clientData,
                                OriginX,
                                OriginY,
                                Theta,
                                VirtPivX,
                                VirtPivY);
}


/*
 *  Other routines
 */

/*
 *+			F p i l G e t M a x P i v A  n g

 * Function name:
      FpilGetMaxPivAng

 * Function:
      Return the instrument maximum pivot angle


 * Description:
      Returns the value which indicates the maximum angle a fibre is 
      allowed to bend coming out of a retractor.  This can be set by
      FpilInit(3) or FpilSetMaxPivAng(3).

 * Language:
      C

 * Call:
      (double) = FpilGetMaxPivAng(inst)

 * Parameters:   (">" input, "!" modified, "W" workspace, "<" output)
      (>) inst            (FpilType)  The instrument as returned 
                                      by FpilInit(3).

 * Returned value:
      The value in radians.

 * Include files: fpi.h

 * See Also: {references}

 * Prior requirements:
      FpilInit(3) must have been invoked for the FpilType variable.

 * Support: Tony Farrell, AAO

 *-

 * History:
     06-Dec-1999 - TJF - Original version
 */
extern double        FpilGetMaxPivAng(const FpilType instrument)
{
    Fpil___InstType *inst = (Fpil___InstType *)instrument;
    return inst->params.MaxPivAngle;

}

/*
 *+			F p i l G e t M a x F i b A  n g

 * Function name:
      FpilGetMaxFibAng

 * Function:
      Return the instrument maximum fibre angle


 * Description:
      Returns the value which indicates the maximum angle a fibre is 
      allowed to bend comming out of a button (due to the button
      angle not aligning with the fibre).   This can be set by
      FpilInit(3) or FpilSetMaxPivAng(3).


      This is only ever non-zero if on some instrumets, as indicated by
      FpilGetFibAngVar(3).

 * Language:
      C

 * Call:
      (double) = FpilGetMaxFibAng(inst)

 * Parameters:   (">" input, "!" modified, "W" workspace, "<" output)
      (>) inst            (FpilType)  The instrument as returned 
                                      by FpilInit(3).

 * Returned value:
      The value in radians.

 * Include files: fpi.h

 * See Also: {references}

 * Prior requirements:
      FpilInit(3) must have been invoked for the FpilType variable.

 * Support: Tony Farrell, AAO

 *-

 * History:
     06-Dec-1999 - TJF - Original version
 */
extern double        FpilGetMaxFibAng(const FpilType instrument)
{
    Fpil___InstType *inst = (Fpil___InstType *)instrument;
    return inst->params.MaxFibAngle;

}
/*
 *+			F p i l G e t F i b A n g V a r

 * Function name:
       FpilGetFibAngVar

 * Function:
       Returns true if the button/fibre angle can vary.

 * Description:
        This function will return true on instruments such as 2dF where
        the button/fibre angle can be varied.  When it does so, 
        FpilGetMaxFibAng(3) indicates the maximum such beng.

 * Language:
      C

 * Call:
      (int) = FpilGetFibAngVar(inst)

 * Parameters:   (">" input, "!" modified, "W" workspace, "<" output)
      (>) inst            (FpilType)  The instrument as returned 
                                      by FpilInit(3).

 * Returned value:
      true/false.

 * Include files: fpi.h

 * See Also: {references}

 * Prior requirements:
      FpilInit(3) must have been invoked for the FpilType variable.

 * Support: Tony Farrell, AAO

 *-

 * History:
     06-Dec-1999 - TJF - Original version
 */
extern int           FpilGetFibAngVar(const FpilType instrument)
{
    Fpil___InstType *inst = (Fpil___InstType *)instrument;
    return inst->params.FibAngleVar;

}

/*
 *+			F p i l G e t M a x  B u t S i z e

 * Function name:
      FpilGetMaxButSize

 * Function:
      Returns the maximum size of a button including clearance.


 * Description:
      Returns the combination of fibre and button clearance with MaxSize

 * Language:
      C

 * Call:
      (unsigned long) = FpilGetMaxButSize(inst)

 * Parameters:   (">" input, "!" modified, "W" workspace, "<" output)
      (>) inst            (FpilType)  The instrument as returned 
                                      by FpilInit(3).

 * Returned value:
      The clearance in microns.

 * Include files: fpi.h

 * See Also: {references}

 * Prior requirements:
      FpilInit(3) must have been invoked for the FpilType variable.

 * Support: Tony Farrell, AAO

 *-

 * History:
     06-Dec-1999 - TJF - Original version
 */

extern unsigned long FpilGetMaxButSize(const FpilType instrument)
{
    Fpil___InstType *inst = (Fpil___InstType *)instrument;

    return  inst->maxButSize;

}

/*
 *+			F p i l G e t M a x E x t e n s i o n

 * Function name:
      FpilGetMaxExtension

 * Function:
      Returns the maximum extension length in microns.

 * Description:
      Returns the maximum extension length in microns.  

 * Language:
      C

 * Call:
      (unsigned long) = FpilGetMaxExtension(inst)

 * Parameters:   (">" input, "!" modified, "W" workspace, "<" output)
      (>) inst            (FpilType)  The instrument as returned 
                                      by FpilInit(3).

 * Returned value:
      The extension microns.

 * Include files: fpi.h

 * See Also: {references}

 * Prior requirements:
      FpilInit(3) must have been invoked for the FpilType variable.

 * Support: Tony Farrell, AAO

 *-

 * History:
     09-Mar-2000 - TJF - Original version
 */

extern unsigned long FpilGetMaxExtension(const FpilType instrument)
{
    Fpil___InstType *inst = (Fpil___InstType *)instrument;
    return inst->params.Extension;

}
/*
 *+			F p i l G e t F i e l d R a d i u s

 * Function name:
      FpilGetFieldRadius

 * Function:
      Returns the field radius in microns.

 * Description:
      Returns the field radius in microns - the radius outside which
      fibres cannot be placed for observing.

 * Language:
      C

 * Call:
      (unsigned long) = FpilGetFieldRadius(inst)

 * Parameters:   (">" input, "!" modified, "W" workspace, "<" output)
      (>) inst            (FpilType)  The instrument as returned 
                                      by FpilInit(3).

 * Returned value:
      The field radious in microns.

 * Include files: fpi.h

 * See Also: {references}

 * Prior requirements:
      FpilInit(3) must have been invoked for the FpilType variable.

 * Support: Tony Farrell, AAO

 *-

 * History:
     13-Mar-2000 - TJF - Original version
 */

extern unsigned long FpilGetFieldRadius(const FpilType instrument)
{
    Fpil___InstType *inst = (Fpil___InstType *)instrument;
    return inst->params.FieldRad;

}
/*
 *+			F p i l G e t F i e l d R a d i u s I n c P a r k

 * Function name:
      FpilGetFieldRadiusIncPark

 * Function:
      Returns the field radius including park space in microns.

 * Description:
      Returns the field radius in microns - the radius outside which
      fibres cannot be placed, even when parked.

 * Language:
      C

 * Call:
      (unsigned long) = FpilGetFieldRadiusIncPark(inst)

 * Parameters:   (">" input, "!" modified, "W" workspace, "<" output)
      (>) inst            (FpilType)  The instrument as returned 
                                      by FpilInit(3).

 * Returned value:
      The field radious in microns.

 * Include files: fpi.h

 * See Also: {references}

 * Prior requirements:
      FpilInit(3) must have been invoked for the FpilType variable.

 * Support: Tony Farrell, AAO

 *-

 * History:
     13-Mar-2000 - TJF - Original version
 */

extern unsigned long FpilGetFieldRadiusIncPark(const FpilType instrument)
{
    Fpil___InstType *inst = (Fpil___InstType *)instrument;
    return inst->params.FieldRadIncPark;

}
/*
 *+			F p i l G e t B u t C l e a r

 * Function name:
      FpilGetButClear

 * Function:
      Returns the clearance required between two buttons.


 * Description:
      Returns the minimum clearance required between two buttons.  This value
      can be set when calling FpiInit(3) or by using FpilSetButClear(3).

 * Language:
      C

 * Call:
      (unsigned long) = FpilGetButClear(inst)

 * Parameters:   (">" input, "!" modified, "W" workspace, "<" output)
      (>) inst            (FpilType)  The instrument as returned 
                                      by FpilInit(3).

 * Returned value:
      The clearance in microns.

 * Include files: fpi.h

 * See Also: {references}

 * Prior requirements:
      FpilInit(3) must have been invoked for the FpilType variable.

 * Support: Tony Farrell, AAO

 *-

 * History:
     06-Dec-1999 - TJF - Original version
 */

extern unsigned long FpilGetButClear(const FpilType instrument)
{
    Fpil___InstType *inst = (Fpil___InstType *)instrument;

    return inst->params.ButtonClear;

}
/*
 *+			F p i l G e t F i b C l e a r

 * Function name:
      FpilGetFibClear

 * Function:
      Returns the clearance required between two buttons.


 * Description:
      Returns the minimum clearance required between a fibre and a button.  
      This value can be set when calling FpiInit(3) or by using 
      FpilSetFibClear(3).

 * Language:
      C

 * Call:
      (unsigned long) = FpilGetFibClear(inst)

 * Parameters:   (">" input, "!" modified, "W" workspace, "<" output)
      (>) inst            (FpilType)  The instrument as returned 
                                      by FpilInit(3).

 * Returned value:
      The clearance in microns.

 * Include files: fpi.h

 * See Also: {references}

 * Prior requirements:
      FpilInit(3) must have been invoked for the FpilType variable.

 * Support: Tony Farrell, AAO

 *-

 * History:
     06-Dec-1999 - TJF - Original version
 */

extern unsigned long FpilGetFibClear(const FpilType instrument)
{
    Fpil___InstType *inst = (Fpil___InstType *)instrument;

    return inst->params.FibreClear;

}
/*
 *+			F p i l G e t F i d C l e a r

 * Function name:
      FpilGetFidClear

 * Function:
      Returns the clearance required between fiducials and fibres/buttons.


 * Description:
      Returns the clearance required between fiducials and fibres/buttons
      This value can be set when calling FpiInit(3) or by using 
      FpilSetFidClear(3).

 * Language:
      C

 * Call:
      (unsigned long) = FpilGetFidClear(inst)

 * Parameters:   (">" input, "!" modified, "W" workspace, "<" output)
      (>) inst            (FpilType)  The instrument as returned 
                                      by FpilInit(3).

 * Returned value:
      The clearance in microns.

 * Include files: fpi.h

 * See Also: {references}

 * Prior requirements:
      FpilInit(3) must have been invoked for the FpilType variable.

 * Support: Tony Farrell, AAO

 *-

 * History:
     06-Dec-1999 - TJF - Original version
 */

extern unsigned long FpilGetFidClear(const FpilType instrument)
{
    Fpil___InstType *inst = (Fpil___InstType *)instrument;

    return inst->params.FidClear;

}


/*+			F p i l S e t B u t C l e a r

 * Function name:
      FpilSetButClear

 * Function:
      Sets the clearance required between two buttons.


 * Description:
      Set the minimum clearance required between two buttons.  This value
      can be fetched FpilGetButClear(3) and is used when checking for
      collisions between buttons.

 * Language:
      C

 * Call:
      (void) = FpilSetButClear(inst,value)

 * Parameters:   (">" input, "!" modified, "W" workspace, "<" output)
      (>) inst            (FpilType)  The instrument as returned 
                                      by FpilInit(3).
      (>) value           (unsigned long) The new value in microns


 * Include files: fpi.h

 * See Also: {references}

 * Prior requirements:
      FpilInit(3) must have been invoked for the FpilType variable.

 * Support: Tony Farrell, AAO

 *-

 * History:
     06-Dec-1999 - TJF - Original version
 */

extern void          FpilSetButClear(const FpilType instrument, 
                                     unsigned long value)
{
    Fpil___InstType *inst = (Fpil___InstType *)instrument;
    inst->params.ButtonClear = value;
    inst->butTolSqrd = (double)value * (double)value;
    inst->maxButSize = inst->params.ButtonClear +
                       inst->params.FibreClear +
                       inst->params.MaxSize;
       
}
/*+			F p i l S e t F i b C l e a r

 * Function name:
      FpilSetFibClear

 * Function:
      Sets the clearance required between a fibre and a button.


 * Description:
      Set the minimum clearance required between a fibre and a button.  
      This value can be fetched FpilGetFibClear(3) and is used when checking 
      for collisions between fibres and buttons.

 * Language:
      C

 * Call:
      (void) = FpilSetFibClear(inst,value)

 * Parameters:   (">" input, "!" modified, "W" workspace, "<" output)
      (>) inst            (FpilType)  The instrument as returned 
                                      by FpilInit(3).
      (>) value           (unsigned long) The new value in microns


 * Include files: fpi.h

 * See Also: {references}

 * Prior requirements:
      FpilInit(3) must have been invoked for the FpilType variable.

 * Support: Tony Farrell, AAO

 *-

 * History:
     06-Dec-1999 - TJF - Original version
 */

extern void          FpilSetFibClear(const FpilType instrument, 
                                     unsigned long value)
{
    Fpil___InstType *inst = (Fpil___InstType *)instrument;
    inst->params.FibreClear = value;
    inst->fibreTolSqrd      = (double)value * (double)value;
    inst->maxButSize        = inst->params.ButtonClear +
                              inst->params.FibreClear +
                              inst->params.MaxSize;
       
}
/*+			F p i l S e t F i d C l e a r

 * Function name:
      FpilSetFidClear

 * Function:
      Sets the clearance required between a fiducial and button/fibre.


 * Description:
      Set the minimum clearance required between a fiducial and
      button/fibre.  This value
      can be fetched FpilGetFidClear(3) and is used when checking for
      collisions between fiducials and buttons/fiducials.

 * Language:
      C

 * Call:
      (void) = FpilSetFidClear(inst,value)

 * Parameters:   (">" input, "!" modified, "W" workspace, "<" output)
      (>) inst            (FpilType)  The instrument as returned 
                                      by FpilInit(3).
      (>) value           (unsigned long) The new value in microns


 * Include files: fpi.h

 * See Also: {references}

 * Prior requirements:
      FpilInit(3) must have been invoked for the FpilType variable.

 * Support: Tony Farrell, AAO

 *-

 * History:
     06-Dec-1999 - TJF - Original version
 */

extern void          FpilSetFidClear(const FpilType instrument, 
                                     unsigned long value)
{
    Fpil___InstType *inst = (Fpil___InstType *)instrument;
    inst->params.FidClear = value;
    inst->fidTolSqrd = (double)value * (double)value;
       
}

/*+			F p i l S e t M a x P i v A n g

 * Function name:
      FpilSetMaxPivAng

 * Function:
       Sets the instrument maximum pivot angle


 * Description:
      Set the value which indicates the maximum angle a fibre is 
      allowed to bend coming out of a retractor.  This can be fetched
      by FpilGetMaxPivAng(3).

 * Language:
      C

 * Call:
      (void) = FpilSetMaxPivAng(inst,value)

 * Parameters:   (">" input, "!" modified, "W" workspace, "<" output)
      (>) inst            (FpilType)  The instrument as returned 
                                      by FpilInit(3).
      (>) value           (double) The new value in radians


 * Include files: fpi.h

 * See Also: {references}

 * Prior requirements:
      FpilInit(3) must have been invoked for the FpilType variable.

 * Support: Tony Farrell, AAO

 *-

 * History:
     06-Dec-1999 - TJF - Original version
 */

extern void          FpilSetMaxPivAng(const FpilType instrument, 
                                      double value)
{
    Fpil___InstType *inst = (Fpil___InstType *)instrument;
    inst->params.MaxPivAngle = value;
       
}

/*+			F p i l S e t M a x F i b A n g

 * Function name:
      FpilSetMaxFibAng

 * Function:
       Sets the instrument maximum pivot angle


 * Description:
      Set the value which indicates the maximum angle a fibre is
      allowed to bend comming out of a button (due to the button
      angle not aligning with the fibre).  This can be fetched
      by FpilGetMaxFibAng(3).

      This is ignored on some instrumets, as indicated by
      FpilGetFibAngVar(3).

 * Language:
      C

 * Call:
      (void) = FpilSetMaxFibAng(inst,value)

 * Parameters:   (">" input, "!" modified, "W" workspace, "<" output)
      (>) inst            (FpilType)  The instrument as returned 
                                      by FpilInit(3).
      (>) value           (double) The new value in radians


 * Include files: fpi.h

 * See Also: {references}

 * Prior requirements:
      FpilInit(3) must have been invoked for the FpilType variable.

 * Support: Tony Farrell, AAO

 *-

 * History:
     06-Dec-1999 - TJF - Original version
 */

extern void          FpilSetMaxFibAng(const FpilType instrument, 
                                      double value)
{
    Fpil___InstType *inst = (Fpil___InstType *)instrument;
    if (inst->params.FibAngleVar)
        inst->params.MaxFibAngle = value;
       
}

/*+			F p i l O n F i e l d

 * Function name:
      FpilOnField

 * Function:
      Indicates if a point lies on the field.


 * Description:
      This functions returns true if the specified point is on the
      field (The field is assumed to be cicrular with origin at
      the center).

 * Language:
      C

 * Call:
      (void) = FpilOnField(inst,x,y)

 * Parameters:   (">" input, "!" modified, "W" workspace, "<" output)
      (>) inst            (FpilType)  The instrument as returned 
                                      by FpilInit(3).
      (>) x               (long) The X ordinate
      (>) y               (long) The Y ordinate


 * Include files: fpi.h

 * See Also: {references}

 * Returned value:
       true if the point is on the field.

 * Prior requirements:
      FpilInit(3) must have been invoked for the FpilType variable.

 * Support: Tony Farrell, AAO

 *-

 * History:
     06-Dec-1999 - TJF - Original version
 */
extern int  FpilOnField(const FpilType instrument, 
                        long x, long y)
{
    double radSqrd;
    Fpil___InstType *inst = (Fpil___InstType *)instrument;
    
    radSqrd = SQRD(x) + SQRD(y);
    return (radSqrd <= inst->fieldRadSqrd);
       
}
/*+			F p i l E x t e n s i o n C h e c k 1

 * Function name:
      FpilExtensionCheck1

 * Function:
      Returns true if a fibre of the given length is valid

 * Description:
      Supplied with the length of a fibre, this
      function returns true if that length is ok.

 * Language:
      C

 * Call:
      (void) = FpilExtensionCheck1(inst,length)

 * Parameters:   (">" input, "!" modified, "W" workspace, "<" output)
      (>) inst            (FpilType)  The instrument as returned 
                                      by FpilInit(3).
      (>) length          (double) The fibre length in microns


 * Include files: fpi.h

 * See Also: {references}
 
 * Returned value:
       True if the fibre length is ok.

 * Prior requirements:
      FpilInit(3) must have been invoked for the FpilType variable.

 * Support: Tony Farrell, AAO

 *-

 * History:
     06-Dec-1999 - TJF - Original version
 */
extern int  FpilExtensionCheck1(const FpilType instrument, 
                                double length)
{
    Fpil___InstType *inst = (Fpil___InstType *)instrument;

    return (length <= inst->params.Extension);
       
}
/*+			F p i l E x t e n s i o n C h e c k 2

 * Function name:
      FpilExtensionCheck2

 * Function:
      Returns true if a fibre of the given length is valid

 * Description:
      Supplied with the delta x and delta y of a fibre length.  This
      function returns true if that lenght is ok.

 * Language:
      C

 * Call:
      (void) = FpilExtensionCheck2(inst,deltaX, deltaY)

 * Parameters:   (">" input, "!" modified, "W" workspace, "<" output)
      (>) inst            (FpilType)  The instrument as returned 
                                      by FpilInit(3).
      (>) deltaX          (double) The fibre length delta in the X direction
      (>) deltaY          (double) The fibre length delta in the Y direction


 * Include files: fpi.h

 * See Also: {references}
 
 * Returned value:
       True if the fibre length is ok.

 * Prior requirements:
      FpilInit(3) must have been invoked for the FpilType variable.

 * Support: Tony Farrell, AAO

 *-

 * History:
     06-Dec-1999 - TJF - Original version
 */
extern int  FpilExtensionCheck2(const FpilType instrument, 
                                double deltaX, double deltaY)
{
    double distance;
    Fpil___InstType *inst = (Fpil___InstType *)instrument;
    
    distance = SQRD(deltaX) + SQRD(deltaY);
    return (distance <= inst->extenSqrd);
}


/*
 *+			F p i l C o l l i s i o n 1

 * Function name:
      FpilCollision1

 * Function:
      Checks for a collision in the xy plane between two fibres and buttons.

 * Description:
      This is a wrap of of a call too FpilColButBut(3) followed
      by calls to FpilColButFib(3) for each direction.

      If a collision is detected the function will return true.

      This version of this routine uses FpilGetVirtPiv(3) to determine
      the virtual pivot point given the button coordinates.

      This routine first creates bounding boxes around the fibres and
      buttons and only uses the underlying collision routines if those boxes
      collide.  Thus, the underlying routines need not bother with
      this type of simple test.

      Invokes the instrument specific button to fibre and  fibre to fibre 
      collision software.

 * Language:
      C

 * Call:
      (int) = FpilCollision1 (inst,
                              butXa, butYa, thetaa, pivXa, pivYa,
                              butXa, butYa, thetaa, pivXb, pivYb)

 *  Parameters:  (">" input, "!" modified, "W" workspace, "<" output)
      (>) inst            (FpilType)  The instrument as returned 
                                      by FpilInit(3).
      (>) butXa  (long int)  Button A X ordinate
      (>) butYa  (long int)  Button A Y ordinate
      (>) thetaa (double)    Button A theta
      (>) pivXa  (long int)  Coordinates of fibre A at pivot - X ordinate.
      (>) pivYa  (long int)  Coordinates of fibre A at pivot - Y ordinate.
      (>) fibXa  (long int)  Corrindates of fibre A at virtual pivot point - X.
      (>) fibYa  (long int)  Corrindates of fibre A at virtual pivot point - Y.
      (>) butXb  (long int)  Button B X ordinate
      (>) butYb  (long int)  Button B Y ordinate
      (>) thetab (double)    Button B theta
      (>) pivXb  (long int)  Coordinates of fibre B at pivot - X ordinate.
      (>) pivYb  (long int)  Coordinates of fibre B at pivot - Y ordinate.
      


 * Returned value:
      FPIL_NOCOL => (false) if no collision.
      FPIL_COL_BUTBUT => If the collision is between the buttons.
      FPIL_COL_BUTFIB => If the collision is between the button a and
                         fibre b.
      FPIL_COL_BUTFIB => If the collision is between the button b and
                         fibre a.
      true if a collision is detected.

 * Include files: fpi.h

 * See Also: {references}

 * Prior requirements:
      FpilInit(3) must have been invoked for the FpilType variable.

 * Support: Tony Farrell, AAO

 *-

 * History:
     06-Dec-1999 - TJF - Original version
 */

extern int FpilCollision1(
        const FpilType instrument,
        long int butXa,
        long int butYa,
        double thetaa,
        long int pivXa,
        long int pivYa,
        long int butXb,
        long int butYb,
        double thetab,
        long int pivXb,
        long int pivYb)
{

/*
 *  This is just a wrapper routine which calculates the virtual pivot
 *  points given the button locations and then calls other version
 *  of this routine.
 */
    double fibXa, fibYa, fibXb, fibYb;
    FpilGetVirtPiv(instrument, butXa, butYa, thetaa, &fibXa, &fibYa);
    FpilGetVirtPiv(instrument, butXb, butYb, thetab, &fibXb, &fibYb);
    return FpilCollision2(instrument,
                          butXa, butYa, thetaa, fibXa, fibYa, pivXa, pivYa,
                          butXb, butYb, thetab, fibXb, fibYb, pivXb, pivYb);
                          

}


/*
 *+			F p i l C o l l i s i o n 2

 * Function name:
      FpilCollision2

 * Function:
      Checks for a collision in the xy plane between two fibres and buttons.


 * Description:
      This is a wrap of of a call too FpilColButBut(3) followed
      by calls to FpilColButFib(3) for each direction.

      If a collision is detected the function will return true

      This routine first creates bounding boxes around the fibres and
      buttons and only uses the underlying collision routines if those boxes
      collide.  Thus, the underlying routines need not bother with
      this type of simple test.

      Invokes the instrument specific button to fibre and  fibre to fibre 
      collision software.

 * Language:
      C

 * Call:
      (int) = FpilCollision2 (inst,
                              butXa, butYa, thetaa, fibXa, fibYa, pivXa, pivYa,
                              butXa, butYa, thetaa, fibXa, fibYb, pivXb, pivYb)

 *  Parameters:  (">" input, "!" modified, "W" workspace, "<" output)
      (>) inst            (FpilType)  The instrument as returned 
                                      by FpilInit(3).
      (>) butXa  (long int)  Button A X ordinate
      (>) butYa  (long int)  Button A Y ordinate
      (>) thetaa (double)    Button A theta
      (>) fibXa  (long int)  Fibre A virtual pivot point - X ordinate.
      (>) fibYa  (long int)  Fibre A virtual pivot point - Y ordinate.
      (>) pivXa  (long int)  Fibre A pivot point - X ordinate.
      (>) pivXa  (long int)  Fibre A pivot point - Y ordinate.
      (>) butXb  (long int)  Button B X ordinate
      (>) butYb  (long int)  Button B Y ordinate
      (>) thetab (double)    Button B theta
      (>) fibXb  (long int)  Fibre B virtual pivot point - X ordinate.
      (>) fibYb  (long int)  Fibre B virtual pivot point - Y ordinate.
      (>) pivXb  (long int)  Fibre B pivot point - X ordinate.
      (>) pivXb  (long int)  Fibre B pivot point - Y ordinate.
      


 * Returned value:
      FPIL_NOCOL => (false) if no colliation
      FPIL_COL_BUTBUT => If the collision is between the buttons.
      FPIL_COL_BUTFIB => If the collision is between the button a and
                         fibre b.
      FPIL_COL_BUTFIB => If the collision is between the button b and
                         fibre a.
      true if a collision is detected.

 * Include files: fpi.h

 * See Also: {references}

 * Prior requirements:
      FpilInit(3) must have been invoked for the FpilType variable.

 * Support: Tony Farrell, AAO

 *-

 * History:
     06-Dec-1999 - TJF - Original version
 */

extern int FpilCollision2(
        const FpilType instrument,
        long int butXa,
        long int butYa,
        double thetaa,
        long int fibXa,
        long int fibYa,
        long int pivXa,
        long int pivYa,
        long int butXb,
        long int butYb,
        double thetab,
        long int fibXb,
        long int fibYb,
        long int pivXb,
        long int pivYb)
{
    Fpil___InstType *inst = (Fpil___InstType *)instrument;
    BoundingBox fibreA;
    BoundingBox fibreB;
    BoundingBox buttonA;
    BoundingBox buttonB;

    FibreBox(inst, pivXa, pivYa, fibXa, fibYa, &fibreA);
    FibreBox(inst, pivXb, pivYb, fibXb, fibYb, &fibreB);
    ButtonBox(inst, butXa, butYa, &buttonA);
    ButtonBox(inst, butXb, butYb, &buttonB);

/*
 *  If the two button boxes overlap, check them out.
 */
    if (BoxCollision(&buttonA, &buttonB))
    {
        if ((*inst->routines.ColButBut)(inst->clientData,
                                        butXa, butYa, thetaa,
                                        butXb, butYb, thetab,
                                        inst->params.ButtonClear,
                                        inst->butTolSqrd))
        {
            return FPIL_COL_BUTBUT;
        }
    }
/*
 *  If button A's box overlpas with fibre B, check them out.
 */
    if (BoxCollision(&buttonA, &fibreB))
    {
        if ((*inst->routines.ColButFib)(inst->clientData,
                                       butXa, butYa, thetaa,
                                       fibXb, fibYb,
                                       pivXb, pivYb,
                                       inst->params.FibreClear,
                                       inst->fibreTolSqrd))
        {
            return FPIL_COL_BUTFIB;
        }
    }
/*
 *  If button B's box overlaps with fibre A, check them out.
 */
    if (BoxCollision(&buttonB, &fibreA))
    {
        if ((*inst->routines.ColButFib)(inst->clientData,
                                       butXb, butYb, thetab,
                                       fibXa, fibYa,
                                       pivXa, pivYa,
                                       inst->params.FibreClear,
                                       inst->fibreTolSqrd))
        {
            return FPIL_COL_FIBBUT;
        }
    }
    return 0;

}








/*
 *+			F p i l G e t N u m F i e l d s

 * Function name:
      FpilGetNumFields

 * Function:
       Return the number of fields (plates) in the instrument.

 * Description:
       Return the number of fields (plates) in the instrument.
       

 * Language:
      C

 * Call:
      (unsigned) = FpilGetNumFields(inst)

 *  Parameters:  (">" input, "!" modified, "W" workspace, "<" output)
      (>) inst            (FpilType)  The instrument as returned 
                                      by FpilInit(3).

 * Returned value:
      The number of fields.

 * Include files: fpi.h

 * See Also: {references}

 * Prior requirements:
      FpilInit(3) must have been invoked for the FpilType variable.

 * Support: Tony Farrell, AAO

 *-

 * History:
     06-Dec-1999 - TJF - Original version
 */
extern unsigned int FpilGetNumFields(const FpilType instrument)
{
    Fpil___InstType *inst = (Fpil___InstType *)instrument;
    return inst->params.NumFields;
}

/*
 *+			F p i l G e t N u m P i v o t s

 * Function name:
      FpilGetNumPivots

 * Function:
       Return the number of pivots on a field plate.

 * Description:
       Return the number of pivots on a field plate.
       

 * Language:
      C

 * Call:
      (unsigned) = FpilGetNumPivots(inst)

 *  Parameters:  (">" input, "!" modified, "W" workspace, "<" output)
      (>) inst            (FpilType)  The instrument as returned 
                                      by FpilInit(3).

 * Returned value:
      The number of pivots on each plate.

 * Include files: fpi.h

 * See Also: {references}

 * Prior requirements:
      FpilInit(3) must have been invoked for the FpilType variable.

 * Support: Tony Farrell, AAO

 *-

 * History:
     06-Dec-1999 - TJF - Original version
 */
extern unsigned int FpilGetNumPivots(const FpilType instrument)
{
    Fpil___InstType *inst = (Fpil___InstType *)instrument;
    return inst->params.NumPivots;
}
/*
 *+			F p i l G e t N u m F i d u c i a l s

 * Function name:
      FpilGetNumFiducials

 * Function:
       Return the number of fiducials on a field plate.

 * Description:
       Return the number of fiducials on a field plate.
       

 * Language:
      C

 * Call:
      (unsigned) = FpilGetNumFiducials(inst)

 *  Parameters:  (">" input, "!" modified, "W" workspace, "<" output)
      (>) inst            (FpilType)  The instrument as returned 
                                      by FpilInit(3).

 * Returned value:
      The number of fiducials on each plate.

 * Include files: fpi.h

 * See Also: {references}

 * Prior requirements:
      FpilInit(3) must have been invoked for the FpilType variable.

 * Support: Tony Farrell, AAO

 *-

 * History:
     06-Dec-1999 - TJF - Original version
 */
extern unsigned int FpilGetNumFiducials(const FpilType instrument)
{
    Fpil___InstType *inst = (Fpil___InstType *)instrument;
    return inst->params.NumFids;
}

/*
 *+			F p i l G e t T e l e s c o p e

 * Function name:
      FpilGetTelescope

 * Function:
       Return the name of the telescope.

 * Description:
       Return the name of the telescope.
       

 * Language:
      C

 * Call:
      (const char *) = FpilGetTelescope(inst)

 *  Parameters:  (">" input, "!" modified, "W" workspace, "<" output)
      (>) inst            (FpilType)  The instrument as returned 
                                      by FpilInit(3).

 * Returned value:
        A pointer to the name of the telescope.  Valid for as long
        as "inst" is valid.  

 * Include files: fpi.h

 * See Also: {references}

 * Prior requirements:
      FpilInit(3) must have been invoked for the FpilType variable.

 * Support: Tony Farrell, AAO

 *-

 * History:
     06-Dec-1999 - TJF - Original version
 */
extern const char* FpilGetTelescope(const FpilType instrument)
{
    Fpil___InstType *inst = (Fpil___InstType *)instrument;
    return inst->params.Telescope;
}

/*
 *+			F p i l G e t I n s t N a m e

 * Function name:
      FpilGetInstName

 * Function:
       Return the name of the instrument.

 * Description:
       Return the name of the instrument.
       

 * Language:
      C

 * Call:
      (const char *) = FpilGetInstName(inst)

 *  Parameters:  (">" input, "!" modified, "W" workspace, "<" output)
      (>) inst            (FpilType)  The instrument as returned 
                                      by FpilInit(3).

 * Returned value:
        A pointer to the name of the instrument.  Valid for as long
        as "inst" is valid.  

 * Include files: fpi.h

 * See Also: {references}

 * Prior requirements:
      FpilInit(3) must have been invoked for the FpilType variable.

 * Support: Tony Farrell, AAO

 *-

 * History:
     06-Dec-1999 - TJF - Original version
 */
extern const char* FpilGetInstName(const FpilType instrument)
{
    Fpil___InstType *inst = (Fpil___InstType *)instrument;
    return inst->params.Instrument;
}



/*
 *+			F p i l D r a w B u t F i b

 * Function name:
      FpilDrawButFib

 * Function:
      Provides information about drawing a button and fibre.

 * Description:
      Note that the fibre is drawn from the pivot position to the
      position returned by this routine.  
   
       The caller needs to supply the scaling from microns to
       pixels as well as conversion routines for converting
       absolute values in microns on the field plate to
       absoluate values in the GUI in pixels.   The zoom and
       CvtData arguments are passed through to the appropriate
       arguments to the conversion routines.  The scaling should
       not include the zoom factor.
       

 * Language:
      C

 * Call:
      (void) = FpilDrawButFib(inst, scaling, zoom, CvtX, CvtY, CvtData,
	                    X, Y, Theta, Type, ButGrapLen, FibX, FibY,
	 		    ButGraphic1, ButGraphic2, status)

 * Parameters:   (">" input, "!" modified, "W" workspace, "<" output)
      (>) inst            (FpilType)  The instrument as returned 
                                      by FpilInit(3).
      (>) scaling         (double) The scaling from microns to pixels, 
                          excluding zoom factor.
      (>) zoom            (double)  Current zoom value.  
      (>) CvtX            (FpilCvtRoutineType) A routine for converting 
                          positions on the field plate in microns in X to
                          pixels on the GUI.
      (>) CvtY            (FpilCvtRoutineType) A routine for converting 
                          positions on the field plate in microns in X to
                          pixels on the GUI.
      (>) CvtData         (void *) Passed through to CvtX and CvtY as
	                  there first arguments.
      (>) X               (double) Fibre position in X.
      (>) Y               (double) Fibre position in Y.
      (>) Theta           (double) Fibre theta.
      (>) Type            (int)  The button type string.  0 = Guide bundle
                          1 = program object.
      (>) ButGrapLen      (unsigned) Length of the ButGraphic1 and ButGrapic2 
                          strings.
      (<) FibX            (double *) The fibre end position in X.
      (<) FibY            (double *) The fibre end position in X.
      (<) ButGraphic1     (char *) The routine will place here
                          the arguments to a Tk canvas widget create command 
                          required to draw a graphic of the button.  Only 
                          graphics which accept a "-fill" option will be used 
                          to allow the Tcl code sets colours appropiately 
                          depending on the type of object and if it has been 
                          selected.  For example, this may contain
                          something like "polygon x1 y1 x2 y2 x2 y3..."
                          or             "oval 10 10 80 80"
      (<) ButGraphic2     (char *) As per ButGraphic2, allowing a second
                          graphic to be used in creation of the button.  
                          Similar conditions apply.  If not required, will
                          be an empty string.
      
      (!) status    (StatusType *) modified status.
      

 * Include files: fpi.h

 *-

 * History:
     16-Mar-2000 - TJF - Original version
 */
extern void FpilDrawButFib (
    const FpilType instrument,
    double scaling,
    double zoom,
    FpilCvtRoutineType CvtX,
    FpilCvtRoutineType CvtY,
    void * CvtData,
    double X,
    double Y,
    double Theta,
    int Type,
    unsigned ButGrapLen,
    double *FibX,
    double *FibY,
    char *ButGraphic1,
    char *ButGraphic2,
    FpilStatusType *status)
{
    Fpil___InstType *inst = (Fpil___InstType *)instrument;
    if (inst->routines.DrawButFib)
    {
        /* Initialise the button graphic strings */
        if (ButGrapLen) 
        {
            ButGraphic1[0] = '\0';
            ButGraphic2[0] = '\0';
        }
        (*inst->routines.DrawButFib)(inst->clientData,
                                     scaling, zoom, CvtX, CvtY, CvtData,
                                     X, Y, Theta, Type, ButGrapLen,
                                     FibX, FibY, ButGraphic1, ButGraphic2,
                                     status);
    }
    else
    {
        *status = FPIL__NODRAW;
    }
}

/*
 *+			F p i l P a r k M a y C o l l i d e
 * Function name:
      FpilParkMayCollide

 * Function:
       Indicates if fibres which are parked may collide

 * Description:
       In some fibre instruments, fibres which are in there parked
       positions cannot collide with any other fibres.  In other instruments,
       they may collide.  This has significant effects on collision
       loops and delta algrothims.  This routine indicates what the
       case is for the instrument.
       

 * Language:
      C

 * Call:
      (int) = FpilParkMayCollide(inst)

 *  Parameters:  (">" input, "!" modified, "W" workspace, "<" output)
      (>) inst            (FpilType)  The instrument as returned 
                                      by FpilInit(3).

 * Returned value:
        True (non-zero) if park positions may collide.

 * Include files: fpi.h

 * See Also: {references}

 * Prior requirements:
      FpilInit(3) must have been invoked for the FpilType variable.

 * Support: Tony Farrell, AAO

 *-

 * History:
     06-Dec-1999 - TJF - Original version
 */
extern int FpilParkMayCollide(const FpilType instrument)
{
    Fpil___InstType *inst = (Fpil___InstType *)instrument;
    return inst->params.ParkMayCollide;
}
