

/*******************************************************************************
* E.S.O. - VLT project
*
* "@(#) $Id: fpcolMdlCalModel.c 74556 2002-11-24 09:25:26Z flamemgr $"
*
*
* who       when      what
* --------  --------  ----------------------------------------------
* tfarrell  08/10/02  Fix but in previous change - was pasing index
*                     rather then actual wavelength.
* tfarrell  30/09/02  fpcolMdlCvtInit() now needs field center wavelength
*                     argument to be supplied.
* kshortri  07/08/02  Needed stdlib.h to avoid warning message under Linux.
* kshortri  06/08/02  Allows for case where _HPUX_SOURCE is already defined,
*                     and includes stdio.h to ensure all routine defined.
* tfarrell  27/06/02  Drop command line arguments to fpcolMdlCalModels().
* tfarrell  09/06/02  Fix some bugs.  Check if we can convert the
*                     field centre RA/Dec to x/y just to ensure
*                     the conversion is ok.
* tfarrell  19/04/02  Renamed from fpcolMdlCalCreateModel.c to
*                     fpcolMdlCalModel.c. Revamp and add functions
*                     to allow us to support wavelength dependent
*                     model.
* ssmedley  07/09/01  created
*
*/

/************************************************************************
*   NAME
*	 fpcolMdlCalNewWavelength(), fpcolMdlCalCreateModels(), fpcolMdlCalSetPars()
*        fpcolMdlCalGetPars(), fpcolMdlCalWaveModel(), fpcllMdlCalShowModels() -Manage
*        a set of wavelength dependent astrometric model.
*
*   SYNOPSIS
*       #include "fpcolMdlCalModel.h"
*
*       int fpcolMdlCalNewWavelength(   double wavelength,
*                                       FpcalData *pFpcalData)
*
*       void fpcolMdlCalSetPars(        const FpcalPars *newPars,
*                                       FpcalMdl *model )
*
*	int fpcolMdlCalCreateModels  (FpcalData		*pFpcalData);
*
*       const FpcalPars * fpcolMdlCalGetPars(const FpcalMdl *model)
*
*       const FpilModelType *fpcolMdlCalWaveModel(FpcalMdl *models, 
*                                                 int waveIndex)
*       double fpcolMdlCalWavelength(FpcalMdl *models, int waveIndex)
*       void fpcolMdlCalShowModels(const FpcalMdl *model)
*
*   DESCRIPTION
*       Routines to manage a set of OzPoz telescope models to allow
*       wavelength dependent fitting.  The same set of model parameters
*       a are used for each model - allowing us to fit a set of parameters
*       across wavelengths.  Note that this assumes that for each
*       model there is a set of information initialised by "fpcolMdlCreateModel()",
*       in particular, the telescope model parameters, and a set of information
*       initalised by "fpcolMdlCvtInit()" - the refraction/dispersion parameters.
*       Each of the models we maintian have the same set of telescope model
*       parameters, but these can be changed (for all models).  But, the
*       wavelength supplied to fpcolMdlCvtInit() will be different for
*       each model.  (This approach could be extend to any other effect, such
*       as tempreature effects).
*      
*
*       fpcolMdlCalNewWavelength() adds a new wavelength (in nanometers)
*       to the wavelengths being managed.   There is a fixed number (10)
*       of wavelengths which can be managed at any time. (this should
*       be plenty).  If fpcolMdlCalCreateModels() had been invoked, the
*       new model is created.  The index of the new model is returned.
*       If the model already exists, then the index of the existing
*       model is returned.
*
*       fpcolMdlCalCreateModels() will create a telescope model for
*       each wavelength using the currently defined telescope model
*       parameters.  The result is a model which for each wavelength
*       handles the astmospheric effects which depend on wavelength.
*
*       fpcolMdlCalSetPars() will set the telescope model parameters,  This
*       will always set the internal record of the parameters to use.  If and
*       only if fpcolMdlCalCreateModels() has been invoked, it will also set
*       the parameters for each wavelength dependent model.
*          
*       fpcolMdlCalGetPars() will fetch the telescope parameters.
*
*       fpcolMdlCalWavelength() will fetch the wavelength for a given index.
*
*       fpcolMdlCalWaveModel() will return the telescope model for a specified
*       wavelength index.
*
*       fpcolMdlCalShowModels() will return details to stdout.
*
*
*   ENVIRONMENT
*
*   CAUTIONS
*
*   EXAMPLES
*
*   SEE ALSO
*
*   BUGS
*
*------------------------------------------------------------------------
*/
#define __EXTENSIONS__    /* Pick up extra include file stuff on solaris */
#ifndef _HPUX_SOURCE
#define _HPUX_SOURCE      /* Pick up extra include file stuff on HP UX */
#endif

#include <strings.h>	/* strcasecmp */
#include <stdio.h>
#include <stdlib.h>

#include "fpcolMdlCalModel.h"

#define CELSIUS_TO_KELVIN(_c) ((_c)+273.150)
#define NANO_TO_MICRO(_nano)  ((_nano)/1000.0)



int fpcolMdlCalNewWavelength(double wavelength,
                  FpcalData *pFpcalData)
{
  int index;
  FpcalMdl *model = &pFpcalData->model;
  /*
   * Determine the center wavelength to use (the one the telescope is pointed using)
   */
  int cenWaveIndex;
  if (pFpcalData->curWaveIndex.Giraffe >= 0)
      cenWaveIndex = pFpcalData->curWaveIndex.Giraffe;
  else if (pFpcalData->curWaveIndex.UVES >= 0)
      cenWaveIndex = pFpcalData->curWaveIndex.UVES;
  else
      cenWaveIndex = pFpcalData->curWaveIndex.FACB;

  /*
   * check if we already have this wavelength;
   */
  for (index = 0; index < model->wavelenCnt ; ++index)
    {
      if (model->wavelength[index] == wavelength)
        {
          /*printf("Wavelength of %g nanometers already known\n", wavelength);*/
          return index;
        }
    }
      
  /*
   * Need to add a new wavelength -> do so if we have space.
   */
  if (model->wavelenCnt < MAX_WLENS)
    {
      index = model->wavelenCnt++;
      model->wavelength[index] = wavelength;
      if (model->created)
        {
          StatusType status = STATUS__OK;
          fpcolMdlCreateModel(NUM_PARAMETERS,
                              model->pars.p,
                              pFpcalData->fieldPlate, /* field plate number */
                              &model->models[index],
                              &status);
          fpcolMdlCvtInit(model->models[index],
                          pFpcalData->observationDate,
                          0.0,  /* TODO: delta UT */
                          CELSIUS_TO_KELVIN(pFpcalData->atmTemp),
                          pFpcalData->atmPressure,
                          pFpcalData->atmHumidity/100.0,
                          NANO_TO_MICRO(model->wavelength[cenWaveIndex]),
                          NANO_TO_MICRO(wavelength),
                          &(pFpcalData->posAngle),
                          &status);
          if (status != STATUS__OK)
            return -1;
        }
      return index;
    }
  else 
    {
      fprintf(stderr,"Can't add any more wavelengths.  Maximum of %d already\n",MAX_WLENS);
      return -1;
    }
}


/*
 * create a model & initialise it.
 * Note: fpcalData must be 'load'ed beforehand.
 */
int fpcolMdlCalCreateModels(FpcalData *pFpcalData)
{
	StatusType	status = STATUS__OK;
        FpcalMdl *model = &pFpcalData->model;
        int index;
        int cenWaveIndex;


	if (!pFpcalData->bInitialised)
	{
		fprintf(stderr, 
                        "Model creation failed, must load fpcaldata first! Use \"load\"\n");
		return -1;
	}
        /*
         * We should have at least one wavelength if bInialised is true, but just
         * check that.
         */
        if (model->wavelenCnt <= 0)
        {
          fprintf(stderr,"No wavelengths have been allocated (should not occur)\n" );
          return -1;
        }
        if (!model->parsInited)
          {
            fprintf(stderr,"No telescope model parameters have been specified\n");
            return -1;
          }

        /*
         * Determine the center wavelength to use (the one the telescope is pointed using)
         *  (Note - this probably indicates we can't really do a model with
         *   multiple operations at different wavenelgths unless we always point
         *   the telescope using one such wavelength - which we specify last.
         */
        
        if (pFpcalData->curWaveIndex.Giraffe >= 0)
            cenWaveIndex = pFpcalData->curWaveIndex.Giraffe;
        else if (pFpcalData->curWaveIndex.UVES >= 0)
            cenWaveIndex = pFpcalData->curWaveIndex.UVES;
        else
            cenWaveIndex = pFpcalData->curWaveIndex.FACB;

        /*
         * Loop around for each wavelength;
         */
        for (index = 0; index < model->wavelenCnt ; ++index)
            {


              /* delete existing models (if they have been created) */
              if (model->created)
                {
                  fpcolMdlFree(model->models[index], &status);
                  if (status != STATUS__OK)
                    {
                      fprintf(stderr, "Couldn't free old telescope model index %d\n", 
                              index);
                      return -1;
                    }
                }

              /*
               * Create new model
               */
              fpcolMdlCreateModel(NUM_PARAMETERS,
                                  model->pars.p,
                                  pFpcalData->fieldPlate, /* field plate number */
                                  &model->models[index],
                                  &status);
              fpcolMdlCvtInit(model->models[index],
                              pFpcalData->observationDate,
                              0.0,  /* TODO: delta UT */
                              CELSIUS_TO_KELVIN(pFpcalData->atmTemp),
                              pFpcalData->atmPressure,
                              pFpcalData->atmHumidity/100.0,
                              NANO_TO_MICRO(model->wavelength[cenWaveIndex]),
                              NANO_TO_MICRO(model->wavelength[index]),
                              &(pFpcalData->posAngle),
                              &status);

              if (status != STATUS__OK)
                {
                  fprintf(stderr,
                          "Error creating model for wavelength %g nanometers, index %d\n",
                          model->wavelength[index], index);
                return -1;
                }
              /*
               * At this point we throw in a validity check.  If we can't convert the
               * field centre to an x and y position, then we will fail later when
               * the FIT is run.
               */
              if (index == 0)
                {
                  double xPos;
                  double yPos;
                  fpcolMdlRd2Xy(model->models[index],
                                 pFpcalData->fieldCentreAppRA,
                                 pFpcalData->fieldCentreAppDEC,
                                 pFpcalData->fieldCentreAppRA,
                                 pFpcalData->fieldCentreAppDEC,
                                 pFpcalData->observationDate,
                                 &xPos,
                                 &yPos,
                                 &status);
                  if (status != STATUS__OK)
                    {
                      fprintf(stderr, "Cannot convert field center to plate position\n");
                      return -1;
                    }
                }
            }
        model->created = 1;
        return 0;
}

/*
 * Change the telescope model paramters.
 */
void fpcolMdlCalSetPars(const FpcalPars *newPars,FpcalMdl *model )
{
  /*
   * We always save the new parameters.
   */
  model->pars = *newPars;
  model->parsInited = 1;
  /*
   * Now for each wavelength dependent model we have set up - save set the
   * new parameters.
   */
  if (model->created)
    {
      int index;
      StatusType status = STATUS__OK;
      for (index = 0; index < model->wavelenCnt ; ++index)
        {
          fpcolMdlSetModel(model->models[index],
                           NUM_PARAMETERS,
                           model->pars.p,
                           &status);
          if (status != STATUS__OK)
            {
              fprintf(stderr, "Programming Error:Error setting telescope model parameters\n");
              fprintf(stderr, "Index = %d, Wavelength = %g, status = %lX\n",
                      index, model->wavelength[index], (long int)status);
              exit(-1);
            }
        }
    }
}
/*
 * Fetch the telescope model parameeter we are currently using.
 */
const FpcalPars * fpcolMdlCalGetPars(const FpcalMdl *model)
{
  return &model->pars;
}



/*
 * Return the telescope model for the specified wavelength index
 */
FpilModelType *fpcolMdlCalWaveModel(FpcalMdl *models, int waveIndex)
{
  if ((models->created)&&(waveIndex >= 0)&&(waveIndex < models->wavelenCnt))
    return models->models[waveIndex];
  return 0;
}

/*
 * Return the wavelength for the specified wavelength index
 */
double fpcolMdlCalWavelength(FpcalMdl *models, int waveIndex)
{
  if ((waveIndex >= 0)&&(waveIndex < models->wavelenCnt))
    return models->wavelength[waveIndex];
  return 0;
}



/*
 * Output details on telescope models.
 */
void fpcolMdlCalShowModels(const FpcalMdl *model)
{
  if (model->wavelenCnt)
    {
      int index;
      printf("There have been %d wavelengths registered\n", model->wavelenCnt);
      for (index = 0; index < model->wavelenCnt ; ++index)
        {
          printf("Wavelength for index %d =  %g nanometers\n", index, model->wavelength[index]);
        }
    }

  if (model->created)
    printf("Telescope models have been created for each wavelength\n");
  else
    printf("The telescope models have not yet been created\n");
}
