/*+			M E S S

 *  Module name:
      MESS

 *  Function:
      Portable message handling routines

 *  Description:
      MessPutFacility - Enable a facility by add it to the list of current
			facilities.
      MessPutFlags - Set the components of a message that are returned.
      MessGetMsg   - Get the text assocated with a message code
      MessFacilityKnown - Returns true if a facility is known to Mess
      MessSeverity -  Return the severity of a message code.
      MessFacility - Return the facility of a message code.
      MessNumber   - Return the number of a message code.
      MessStatusToSysExit - Convert a status code to something appropaite
                     for returning as the main() functions status

 *  Language:
      C

 *  Support: Tony Farrell, AAO

 *-

 *  Copyright (c) Anglo-Australian Telescope Board, 1995.
    Not to be used for commercial purposes without AATB permission.
  
 * Sccs Id:     mess.c, Release 1.15, 05/27/98
  
 *  History:
      27-Aug-1992 - TJF - Original version
      11-Apr-1994 - TJF - Use StatusType in the union.  Support Alpha
      02-Aug-1996 - TJF - Tidy up compilation under DEC on Alpha.
      04-Nov-1997 - TJF - Add MessFacilityKnown routine.
      05-Dec-1997 - TJF - Support Intel x86 (M_I86 macro).  This is
			  a little endian machine.
      17-Mar-1998 - TJF - Add MessStatusToSysExit().
      27-May-1998 - TJF - Fix linux compilation warnings.
      {@change entry@}
 */
/*
 *  Include files
 */
#include "status.h"
#include "drama.h"
#include "mess.h"
#include <stdio.h>
#include <string.h>
#ifdef VMS
#   include <descrip.h>
#   include <jpidef.h>
#   include <ssdef.h>
#endif
#ifdef VxWorks
#   include <vxWorks.h>
#   include <semLib.h>
#   include <taskLib.h>
#endif
#define NIL 0
static MessFacType  *MessFacilities = NIL;
#ifdef VMS
    static long int MessDefFlags = -1;
#else
    static long int MessDefFlags = MESS_M_TEXT | MESS_M_IDENT |
				   MESS_M_SEVERITY | MESS_M_FACILITY;
#endif

#ifdef VxWorks 
    SEM_ID Semaphore = 0;
#endif

#ifdef vax				/* Get byte order		*/
#   define BIGEND 0
#elif defined(__alpha)
#   define BIGEND 0
#elif defined DECSTATION
#   define BIGEND 0
#elif defined M_I86
#   define BIGEND 0
#else
#   define BIGEND 1
#endif

/* This structure helps us build a message code.  Since it used fields, it
 *	depends on the the representation
 */
struct Message {
#if BIGEND
	unsigned cntrl     : 4;		/* Control bits (leave clear */
	unsigned userfac   : 1;		/* Set if user facility */
	unsigned facility  :11;		/* Facility code ( 1 - 2047) */
	unsigned notsystem : 1;		/* Set if not system wide   */
	unsigned number   : 12;		/* Message number (1 - 4095) */
	unsigned severity : 3;		/* Severity codes	*/
#else 
	unsigned severity : 3;		/* Severity codes	*/
	unsigned number   : 12;		/* Message number (1 - 4095) */
	unsigned notsystem : 1;		/* Set if not system wide   */
	unsigned facility  :11;		/* Facility code ( 1 - 2047) */
	unsigned userfac   : 1;		/* Set if user facility */
	unsigned cntrl     : 4;		/* Control bits (leave clear */
#endif
	};

/*
 *  This union is used to convert from the above bit map into an integer
 */
	
union MessageUnion {
	struct Message mm;
	StatusType  im;
	};
/*
 *  This array gives the text which is output for each of the 7 possible
 *  severity values.  (Only the first 5 are actually used)
 */
DCONSTV char *SeverityChars[] = {"W","S","E","I","F","X","X","X"};
/*
 *  Function prototypes.  Any function not prototypes in include files is
 *  prototyped here.
 */
#ifdef VMS
#pragma nostandard
    extern int lib$getjpi (long int *item_code,
			   unsigned long int process_id,
			   struct dsc$descriptor_s *process_name,
			   void *out_value,
			   struct dsc$descriptor_s *out_string,
			   short out_len);
     extern int sys$getmsg(DCONSTV unsigned long int smgid,  
		          unsigned short int *msglen,
		          struct dsc$descriptor_s    *bufadr,
			  DCONSTV unsigned long int flags,
			  void * outadr);
			      
#pragma standard
#endif
/*
 *+			M e s s F a c i l i t y

 *  Function name:
      MessFacility

 *  Function:
      Return the facility number associated with a message code.

 *  Description:
	Analyzes a mesage code and returns the facility assocaited with
	the message.


 *  Language:
      C

 *  Call:
      (int) = MessFacility (code)

 *  Parameters:   (">" input, "!" modified, "W" workspace, "<" output)
      (>) code	(StatusType)	The message code.

 *  Include files: mess.h


 *  Support: Tony Farrell, AAO

 *-

 *  History:
      23-Aug-1993 - TJF - Original version
      {@change entry@}
 */

extern int MessFacility (StatusType code)
{
    union MessageUnion m;
    m.im = code;
    return(m.mm.facility);
}
/*
 *+			M e s s S e v e r i t y

 *  Function name:
      MessSeverity

 *  Function:
      Return the severity of a message

 *  Description:
	Analyzes a message code and returns the severity of the message


 *  Language:
      C

*   Call:
      (int) = MessSeverity(code)

 *  Parameters:   (">" input, "!" modified, "W" workspace, "<" output)
      (>) code	(StatusType)	The message code.

 

 *  Include files: mess.h


 *  Support: Tony Farrell, AAO

 *-

 *  History:
      23-Aug-1993 - TJF - Original version
      {@change entry@}
 */

extern int MessSeverity (StatusType code)
{
    union MessageUnion m;
    m.im = code;
    return(m.mm.severity);
}
/*
 *+			M e s s N u m b e r

 *  Function name:
      MessNumber

 *  Function:
      Return the number of a message

 *  Description:
	Analyzes a message code and returns the number of the message
	within the facility


 *  Language:
      C

*   Call:
      (int) = MessNumber(code)

 *  Parameters:   (">" input, "!" modified, "W" workspace, "<" output)
      (>) code	(StatusType)	The message code.

 

 *  Include files: mess.h


 *  Support: Tony Farrell, AAO

 *-

 *  History:
      23-Aug-1993 - TJF - Original version
      {@change entry@}
 */

extern int MessNumber (StatusType code)
{
    union MessageUnion m;
    m.im = code;
    return(m.mm.number);
}




/*
 *+			M e s s P u t F a c i l i t y

 *  Function name:
      MessPutFacility

 *  Function:
      Add a new message facility to the list of known facilities

 *  Description:
      The new message facility is added to the beginning of the list.  Any
      other facility of the same number will be ignored.

      Under UNIX and VMS, this list is kept on a process specific basis.
	Under VxWorks, it is kept on a global basis.
	
      Under VxWorks, a mutual exclusion semaphore is used to protect the 
	data structure against two or more tasks accessing the list at 
        the same time.  If the semaphore has already been taken by  a
        task, other tasks will block until it becomes available.  Since the
        Semaphore is priority inversion safe, task deletion safe, queues 
        pended tasks on the bases of their priority and the operation is 
        quick, no task should block for any length of time.  This makes the
        use of this routine safe in all cases, except from interrupt handlers, 
        thus, this routine should not be called from interrupt handlers.


 *  Language:
      C

 *  Call:
      (Void) = MessPutFacility (facility)

 *  Parameters:   (">" input, "!" modified, "W" workspace, "<" output)
      (>) facility	(MessFacType *) The new facility as generated by 
				the messgen utility.

 *  Include files: mess.h

 *  External functions used:
	semMCreate	(VxWorks) VxWorks only, create a mutual-exclusion
					semaphore.
	semTake		(VxWorks) VxWorks only, take a semaphore.
	semGive		(VxWorks) VxWorks only, give a semaphore.

 *  External values used:
      None

 *  Prior requirements:
      None

 *  Support: Tony Farrell, AAO

 *-

 *  History:
      27-Aug-1992 - TJF - Original version
      21-Oct-1994 - TJF - Catch insertion of the same facility a second type,
				(same address, not just same facility number).
      10-Dec-1994 - TJF - The previous fix is not correct.  See notes in
			code.  We have a temp fix where we do not re-insert
			a facility which has already been inserted.
      {@change entry@}
 */

extern void MessPutFacility (MessFacType *facility)
{
    MessFacType *previousFacility;
/*
 *	Under VxWorks, with we don't already have a Semaphore, create one.
 *	IF we can't get a Semaphore, then NULL is returned and we just
 *	continue, ignore errors.
 */
#   ifdef VxWorks
	taskLock();
	if (Semaphore == 0)
	    Semaphore = semMCreate(SEM_Q_PRIORITY | 
				   SEM_INVERSION_SAFE |
				   SEM_DELETE_SAFE);
        taskUnlock();
/*
 *	Wait for any other task which has the semaphore to give it, and then
 *	take it ourselves.
 */
        semTake(Semaphore,WAIT_FOREVER);
#   endif
/*
 *   Has this facility already been inserted.  Try and find it.
 */
    previousFacility = MessFacilities;
    while ((previousFacility != NIL)&&
           (previousFacility->facnum != facility->facnum))
        previousFacility = previousFacility->next;

/*
 *  Three possibilities.
 *	1.  There is not previous facility of the same name - we insert the
 *		new facility.
 *	2.  There is a previous facility of the same name but it has a 
 *	        different address.  We just insert the new one at the 
 *		begining of the list, it will override the old one.
 *		THIS IS IN ERROR, WE SHOULD DELETE THE OLD ONE.
 *
 *	3.  There is a previous facility of the same name and it is actually
 *		the same one.  Here we don't want to insert the new one as
 *		we would be inserting the same thing twice, clobbering the
 *		next field.
 *
    CURRENTLY, WE DON'T INSERT IF THE IS ANY PREVIOUS FACILITY OF THE SAME
 *  NAME - THIS FIXES A PROBLEM WHEN RESTARTING PROGRAMS. WE SHOULD DELETE
 *  THE PREVIOUS FACILITY AND INSERT THE NEW ONE.
 */
    if (!previousFacility)
    {
/*
 *     Insert the new facility at the beginning of the list.
 */
        facility->next = MessFacilities;
        MessFacilities =  facility;
    }
/*
 *  Give the semaphore back
 */
#   ifdef VxWorks
	semGive(Semaphore);
#   endif
}

/*
 *+			M e s s P u t F l a g s

 *  Function name:
      MessPutFlags

 *  Function:
	Sets the default components when a message is returned.

 *  Description:
	This routine only has effect on {\tt MessGetMsg} if that routine is
	called with its {\tt flags} argument equal to 0.

	If this routine is not called, then all components of a message
	are returned by {\tt MessGetMsg}, otherwise this routine sets the 
	components to be returned.

	An exception is under VMS.  Under VMS, if this routine is not called,
	{\tt MessGetMsg} uses the default VMS message flags.
	

 *  Language:
      C

 *  Call:
      (Void) = MessPutFlags (int flags)

 *  Parameters:   (">" input, "!" modified, "W" workspace, "<" output)
      (>) flags     (Int) Message components to be returned. 
		    If flags is specified as 0, then components are return.
		    Set the following bits true for include the component
		    MESS_M_TEXT =>   Message text.
		    MESS_M_IDENT =>  Message identifier.
		    MESS_M_SEVERITY => Message severity indicator.
		    MESS_M_FACILITY => Facility name.
		

 *  Include files: mess.h

 *  External functions used:
      None

 *  External values used:
      None

 *  Prior requirements:
      None

 *  Support: Tony Farrell, AAO

 *-

 *  History:
      27-Aug-1992 - TJF - Original version
      {@change entry@}
 */

extern void MessPutFlags (DCONSTV int flags)
{
    if (flags == 0)
        MessDefFlags = MESS_M_TEXT | MESS_M_IDENT |
				   MESS_M_SEVERITY | MESS_M_FACILITY;
    else
	MessDefFlags = flags;
}


/*
 *+			M e s s G e t M s g

 *  Function name:
      MessGetMsg

 *  Function:
	Return message text for a given message number.

 *  Description:
       Searches the known message facilities for a message of the
       given number.  In this search, the severity of the supplied
       message number is ignored.  If found, a message string is constructed
       according to the details requested in the flags argument.  The
       format is-

	    "&FACILITY-S-NAME, {\tt text}"

      where-
	    FACILITY => The name of the correspond facility.
	    S	     => The severity of the message, as supplied in the
		       call to this routine (The same message can have
		       different severities).
	    NAME     => The message name.
	    text     => The text associated with the message.
		       

    If the message cannot be found, a suitable default text is returned.
    As a special case, the message number 0 is always defined and 
    returns an appropriate OK text.

    When running under VMS the behaviour changes.  If a message cannot
    be found, then the SYS$GETMSG system service is called to attempt to find
    the message in the VMS tables.  This will work if the message is a
    VMS system message or the message is known to the current process
    (either an object file generated by message/object is linked to the 
     current image or a "set message" command has been used to add the
     message to the current process)

 *  Language:
      C

 *  Call:
      (int) = MessGetMsg (msgid, flags, buflen, buffer)

 *  Parameters:   (">" input, "!" modified, "W" workspace, "<" output)
      (>) msgid	    (StatusType)   The identification of the message to be
			    retrieved.
      (>) flags     (Int) Message components to be returned.  Set the
		    following bits true to include the component -
		    MESS_M_TEXT =>   Message text.
		    MESS_M_IDENT =>  Message identifier.
		    MESS_M_SEVERITY => Message severity indicator.
		    MESS_M_FACILITY => Facility name.

		If flags is 0, then the default flags, set by
		MessSetFlags, is used.  If MessSetFlags has not been
		called, then all are set true, except under VMS, where
		the processes VMS message mask is used.
	(>) buflen (Int *)  The length of buffer.
        (>) buffer (Char *) The buffer to place the message in.  If it is
			    too long, then the message is truncated.

 *  Include files: mess.h

 *  External functions used:
	strncpy		(CRTL)	Copy one string to another.
	strncat		(CRTL)	Concentrate one string to another.
	sprintf		(CRTL)	Formated print to a string.

	lib$getjpi 	(VMS-RTL) (VMS version only) Get job process
				information.
	sys$getmsg	(VMS) (VMS version only) Get VMS message text.

 *  Function value:
	Return 1 if the message is found and 0 if it is not.

 *  External values used:
      None

 *  Prior requirements:
      None

 *  Support: Tony Farrell, AAO

 *-

 *  History:
      27-Aug-1992 - TJF - Original version
      {@change entry@}
 */

extern int MessGetMsg (DCONSTV StatusType msgid, int flag,
			DCONSTV int buflen, char  *buffer)
{
    int facility;	           /* The message facility code */
    int message;		   /* The message number (within facility)*/
    int severity;	           /* The message severity	*/
    int messIndex;		   /* The index into the facilities messages */
    MessFacType *facilityPointer;  /* Pointer to the facility table */
    union MessageUnion m;	   /* Used to convert the message id to its components*/
    int found = 0;		   /* Has the message been found	*/
    int first = 1;		   /* Indicates the first component of the string to be output */

/*
 *  Extract the message details, using the union for the conversion.
 */
    m.im = msgid;
    facility = m.mm.facility;
    severity = m.mm.severity;
    message = m.mm.number;
/*
 *  Clear the message buffer
 */
    if (buflen > 0)
	buffer[0] = '\0';

/*
 *  If no flags are specified, use the default, except under VMS where
 *  we try for the VMS process defaults.
 */
    if (flag == 0)
    {	
	flag = MessDefFlags;
#	ifdef VMS
#	pragma nostandard
	    if (flag == -1)
	    {
		int status;
		long int item = JPI$_MSGMASK;
		status = lib$getjpi(&item,0,0,&flag,0,0);
		if (status  != SS$_NORMAL)
		    flag  = MESS_M_TEXT | MESS_M_IDENT |
		            MESS_M_SEVERITY | MESS_M_FACILITY;
	    }
#	pragma standard
#	endif
    }
/*
 *  Handle as a special case, a message id of 0
 */
    if (msgid == 0)
    {
	if (flag & MESS_M_FACILITY)
	{
	    first = 0;
            strncpy(buffer,"&",buflen);
	    strncat(buffer,"ALL",buflen);
	}
	if (flag & MESS_M_SEVERITY)
	{
	    if (first)
                strncpy(buffer,"&",buflen);
            else
		strncat(buffer,"-",buflen);
	    strncat(buffer,"S",buflen);
	    first = 0;
	}
	if (flag & MESS_M_IDENT)
	{
	    if (first)
                strncpy(buffer,"&",buflen);
            else
		strncat(buffer,"-",buflen);
            strncat(buffer,"OK",buflen);
	    first = 0;
	}
	if (flag & MESS_M_TEXT)
	{
	    if (first)
                strncpy(buffer,"&",buflen);
            else
	        strncat(buffer,", ",buflen);
           strncat(buffer,"Status OK",buflen);
        }
	return(1);
    }
/*
 *   For all other cases, try to find the facility.  A facility of zero is
 *   an unknown message
 */

    if (facility == 0)
	facilityPointer = NIL;
    else
    {
        facilityPointer = MessFacilities;
        while ((facilityPointer != NIL)&&(facilityPointer->facnum != facility))
	    facilityPointer = facilityPointer->next;
    }

    if (facilityPointer != NIL)
    {
/*
 *	If we have a facility, try for the message
 */
	int num;
	messIndex = 0;
	while ((num = facilityPointer->mestable[messIndex].number) != 0)
	{
	    if (num == message)
	    {
		found = 1;
		break;
	    }
	    ++messIndex;
	}
/*
 *	Format the message
 */
	if (flag & MESS_M_FACILITY)
	{
	    first = 0;
            strncpy(buffer,"&",buflen);
	    strncat(buffer,facilityPointer->facname,buflen);
	}
	if (flag & MESS_M_SEVERITY)
	{
	    if (first)
                strncpy(buffer,"&",buflen);
            else
		strncat(buffer,"-",buflen);
	    strncat(buffer,SeverityChars[severity],buflen);
	    first = 0;
	}
	if (flag & MESS_M_IDENT)
	{
	    if (first)
                strncpy(buffer,"&",buflen);
            else
		strncat(buffer,"-",buflen);
	    if (found)
		strncat(buffer,facilityPointer->mestable[messIndex].name,buflen);
	    else
		strncat(buffer,"NOMSG",buflen);

	    first = 0;
	}
	if (flag & MESS_M_TEXT)
	{
	    if (first)
                strncpy(buffer,"&",buflen);
            else
		strncat(buffer,", ",buflen);
	    if (found)
		strncat(buffer,facilityPointer->mestable[messIndex].text,buflen);
	    else
	    {
		char s[30];
		sprintf(s,"Message Number %08lX",(long int)msgid);
		strncat(buffer,s,buflen);
	    }
	}
    }
#   ifdef VMS
/*
 *  If we have not found this message, then If under VMS, usr SYS$GETMSG 
 *  to get message details.
 */
    if (!found)
    {
	unsigned short msglen;
	long int status;
#	pragma nostandard   /* Auto init of structure	*/
	struct dsc$descriptor_s desc = { 0,
					 DSC$K_DTYPE_T,
					 DSC$K_CLASS_S,
					 0};
	desc.dsc$w_length = buflen;
	desc.dsc$a_pointer = buffer;
	status = sys$getmsg(msgid,&msglen,&desc,0,0);
	buffer[msglen] = '\0';
	if (status == SS$_MSGNOTFND)
	    return(1);
        else
	    return(0);
#       pragma standard
    }

#   else
/*
 *  If we have not found a facility and we are not running VMS, then
 *  setup a NONAME message.
 */
    else
    {
	int first = 1;
	if (flag & MESS_M_FACILITY)
	{
	    first = 0;
            strncpy(buffer,"&",buflen);
	    strncat(buffer,"NONAME",buflen);
	}
	if (flag & MESS_M_SEVERITY)
	{
	    if (first)
                strncpy(buffer,"&",buflen);
	    else
	  	strncat(buffer,"-",buflen);
	    strncat(buffer,SeverityChars[severity],buflen);
	    first = 0;
	}
	if (flag & MESS_M_IDENT)
	{
	    if (first)
                strncpy(buffer,"&",buflen);
	    else
		strncat(buffer,"-",buflen);
	    strncat(buffer,"NOMSG",buflen);

	    first = 0;
	}
	if (flag & MESS_M_TEXT)
	{
	    char s[30];
	    if (first)
                strncpy(buffer,"&",buflen);
	    else
		strncat(buffer,", ",buflen);
	    sprintf(s,"Message Number %08lX",(long int)msgid);
	    strncat(buffer,s,buflen);
	}
    }
    return(found);
#   endif

}



/*
 *+			M e s s F a c i l i t y K n o w n 

 *  Function name:
      MessFacilityKnown

 *  Function:
        Returns true if the specified facility is known to the system.

 *  Description:
        This routine returns true if the specified facility is known
        to the system.

 *  Language:
      C

 *  Call:
      (int) = MessFacilityKnown (int facilityNum)

 *  Parameters:   (">" input, "!" modified, "W" workspace, "<" output)
       (int) facilityNum (int) The number of the facility to check for.

 *  Include files: mess.h

 *  External functions used:
      None

 *  External values used:
      None

 *  Prior requirements:
      None

 *  Support: Tony Farrell, AAO

 *-

 *  History:
      04-Nov-1997 - TJF - Original version
      {@change entry@}
 */

extern int MessFacilityKnown (int facility)
{
    MessFacType *facilityPointer;
    if (facility == 0) return 0;

    facilityPointer = MessFacilities;
    while ((facilityPointer != NIL)&&(facilityPointer->facnum != facility))
	facilityPointer = facilityPointer->next;

    return ((facilityPointer != NIL) ? 1 : 0);


}

/*
 *+			M e s s S t a t u s T o S y s E x i t

 *  Function name:
      MessStatusToSysExit

 *  Function:
        Given a status code, return an exit value.

 *  Description:
        This routine returns an appropiate value to be used with the
        main() function return statement, given a status code.

 *  Language:
      C

 *  Call:
      (int) = MessStatusToSysExit (StatusType code)

 *  Parameters:   (">" input, "!" modified, "W" workspace, "<" output)
       (>) code (StatusType) The message code to convert.

 *  Include files: mess.h

 *  Prior requirements:
      None

 *  Support: Tony Farrell, AAO

 *-

 *  History:
      17-Mar-1998 - TJF - Original version
      {@change entry@}
 */

extern int MessStatusToSysExit (DCONSTV StatusType code)
{
/*
 *  Under VMS, convert STATUS__OK to SS$_NORMAL and return the final status.
 */
#   ifdef VMS
	if (code == STATUS__OK)
	{
#           pragma nostandard /* $ sign in identifier */
	    return (SS$_NORMAL);
#	    pragma standard
	}
	else
	    return (code);
/*
 *  Under VxWorks, return the actual status;
 */
#   elif defined(VxWorks)
	return(code);

/*
 *  WIN32 codes look a bit like our status codes.  We try to translate.
 */
#   elif defined(WIN32)
    {
/*
 *      This structure represents the WIN32 exit code structure.  It is
 *      very similar, drops a few unncessary bits and uses them to expand the
 *      ranges of the facility and number items.
 */
        struct winCodeStruct {
	    unsigned severity :  2;	/* Severity code	     */
   	    unsigned userfac  :  1;	/* Set if user facility      */
            unsigned reserved :  1;     /* Reserved                  */
	    unsigned facility : 12;	/* Facility code ( 0 - 4095) */
	    unsigned number   : 16;     /* Message number (0 - 65535 */
        } winCode;

        union {
            struct winCodeStruct winCode;
            int    integer;
        } winUnion;
/*
 *      Severity codes.  Note, we have one less then our scheme. We must
 *      combine the ERROR and SEVERE codes on WIN32's ERROR code.  
 */
        enum { WinSuccess = 0, WinInfo = 1, WinWarning = 2, WinError = 3 };

/*
 *      Use the Union to pull apart our error code.
 */
        union MessageUnion MessageCode;
        MessageCode.im = code;

/*
 *      Work out the WIN32 severity;
 */
        switch (MessageCode.mm.severity)
        {
            case STS_K_SUCCESS:
                winCode.severity = WinSuccess;
                break;
            case STS_K_INFO:
                winCode.severity = WinInfo;
                break;
            case STS_K_WARNING:
                winCode.severity = WinWarning;
                break;
            default:
                winCode.severity = WinError;
                break;
        }
/*
 *      Fill out the rest of the structure.
 */
        winCode.userfac  = MessageCode.mm.userfac;
        winCode.reserved = 0;
        winCode.facility = MessageCode.mm.facility;
        winCode.number   = MessageCode.mm.number;

        winUnion.winCode = winCode;
        return winUnion.integer;
    }

#   else
/*
 *  Otherwise (unix) we can only use the bottom byte.  So just return the
 *  severity.
 */
    {
	return(MessSeverity(code));
    }
#endif
}
