/*******************************************************************************
* E.S.O. - VLT project
*
* "@(#) $Id: convertPAF.c 290518 2016-11-24 15:14:57Z pbaksai $"
*
* who       when      what
* --------  --------  ----------------------------------------------
 * jpritcha 2016-10-04  removed no longer useful DEBUG statements
 *                      added during macOS X version development.
* jpr/pba   2016/06/22  PPRS-65719: Corrected error in proper motion.
* rschmutz  03/02/03  ignore trailing spaces before ';'.
* rschmutz  29/01/03  PPRS 8670: load INS.OSS.CHECKHA in PAF file.
* rschmutz  29/01/03  this header created - previous history see below.
*/

/*
 *                        c o n v e r t  P A F . c
 *
 *  Module name:
 *     convertPAF
 *
 *  Function:
 *     Read a PAF file and create the corresponding sds structure.
 *
 *  Description:
 *     This module contains the routines that handle reading in a PAF
 *     file written by the FP OSS (configure) program and creating the
 *     corresponding SDS structure. Normally the PAF file will have been
 *     written by FP OSS, but these routines don't assume this - they
 *     don't assume the order of items used by the FP OSS routines that
 *     write a PAF file, for example. However, this has not been written
 *     as an efficient general purpose PAF file access routine. In particular,
 *     the search for PAF keywords will only be efficient if they are 
 *     searched for in the order they appear in the file, so if a file using
 *     a different order to that written by FP OSS is used it will work, but
 *     relatively slowly.
 *
 *     Note that the only external routines defined in this file are
 *     PAF_CreateSds(), as this does all that is needed by FP OSS, and
 *     PAF_GetErrorText(). Of these, PAF_GetErrorText() can be used when
 *     PAF_CreateSds() returns bad status to get a description of the
 *     error, but since the error will also have been reported using the
 *     Ers system, this is probably unnecessary.
 *
 *  Language: C
 *
 *  Author(s): Keith Shortridge, AAO (KS)
 *-
 *
 *  History:
 *     20th Apr 2001.  KS.  Original version
 *      7th Aug 2001.  KS   Now creates an allocSky and allocGui item in 
 *                          the fieldData section, and an allocObj.
 *     16th Aug 2001.  KS.  Now supports ARGUS keywords.
 *     18th Aug 2001.  KS.  Minor changes to get a clean compilation under gcc
 *                          with the -ansi and Wall flags set.
 *     31st Oct 2001.  KS.  Now used Fpil for instrument-dependent code, and
 *                          corrects bug where a PAF file with no VLT guide
 *                          star assigned could not be read properly.
 *     22nd Aug 2002.  KS.  Counts of allocated guides, allocated objects,
 *                          and allocated sky targets now set correctly.
 *     28th Aug 2002.  KS.  Mode code is now deduced from the PAF.ID 
 *                          keyword.
 *     29th Aug 2002.  KS.  Optional keywords are now encoded into the comment
 *                          string.
 *     30th Aug 2002.  KS.  pId structure should have been an array of shorts
 *                          not of ints. Now fixed.
 *      7th Nov 2002.  KS.  Fixed order of tests in PAF_GetValue() to prevent
 *                          infinite loop when reading files written in expert
 *                          mode. See comment in that routine about efficiency
 *                          when there are optional keywords.
 *      3th Jan 2003.  RSC  PAF_CreateSds: load ArgusAngle without adding Offset.
 */
 
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <strings.h>
#include <assert.h>

#include "status.h"
#include "sds.h"
#include "slalib.h"
#include "slamac.h"
#include "tdFconvert.h"
#include "Ers.h"

#include "convertPAF.h"
 
/*  Global variables used by this module */

static char** PAF_Lines;            /* Pointers to start of each line -
                                     * these are all addresses in PAF_Array */
static char* PAF_Array;             /* Malloced array for file lines */
static char PAF_ErrorString[2048];  /* Error text - needs to be long enough to
                                     * hold a full file name and some text */
static int PAF_CurrentLine;         /* Index of the next line in the file */

#ifndef FALSE
#define FALSE 0
#endif

#ifndef TRUE
#define TRUE 1
#endif

/*  These flags are used to control some of the general routines defined
 *  in this file. DHDX_FLAG is set if the various delta values - used only
 *  for guide stars - are to be included. FIBRE_FLAG is set if the fibre
 *  number is to be obtained from the PAF keywords. THETA_FLAG is set if
 *  the theta value in an object structure is to be accessed. At the moment,
 *  Theta information is not saved in a PAF file, so although we have code in
 *  here to handle theta, all the Theta values are in fact set to zero and
 *  are sorted out later by higher level code.
 */
 
#define DHDX_FLAG  1
#define FIBRE_FLAG 2
#define THETA_FLAG 4

/*  A PAF_ObjectData structure is a convenient collection of the various
 *  data held in PAF keywords about a specific target object, and is 
 *  used as an argument to a number of the PAF_ routines. The names
 *  generally match those used by the corresponding SDS structure items
 *  and their meanings should be pretty obvious. The character string
 *  included is to provide work space in case optional keywords need
 *  to be included in the comment string. In this case, the expanded
 *  string is built up in CommentString and the Comment pointer is set
 *  to point to CommentString.
 */

typedef struct PAF_ObjectData {
   char* Name;
   double Ra;
   double Dec;
   int X;
   int Y;
   double Theta;
   char Type;
   char Spectrograph;
   short Priority;
   double Magnitude;
   short PId;
   char* Comment;
   int Fibre;
   char CommentString[1024];
} PAF_ObjectDataType;

/*  A PAF_StructIds structure is a convenient collection of the various
 *  SDS Ids corresponding to the various items in one of the SDS
 *  structures used by configure, and is used as an argument to a number
 *  of the PAF_ routines. The names generally match those used by the
 *  corresponding SDS structure items.
 */

typedef struct PAF_StructIds {
   SdsIdType nameId;
   SdsIdType raId;
   SdsIdType decId;
   SdsIdType xId;
   SdsIdType yId;
   SdsIdType dhdxId;
   SdsIdType dddxId;
   SdsIdType dhdyId;
   SdsIdType dddyId;
   SdsIdType thetaId;
   SdsIdType typeId;
   SdsIdType spectId;
   SdsIdType priorityId;
   SdsIdType magId;
   SdsIdType pId;
   SdsIdType commentId;
} PAF_StructIdType;

/*  External routines (defined in configure.c) that are used by the routines
 *  in this module.
 */
 
void DoMakeAllocated (
   SdsIdType topid,
   SdsIdType pivid,
   StatusType* status);
   
void ConfCSetArgusData(
    short InUse,
    double Angle,
    char* Scale,
    double Offset);
    
void ConfCGetArgusData(
    short* InUse,
    double* Angle,
    char* Scale,
    double* Offset);


/* -------------------------------------------------------------------------- */

/*                  P A F _  R e a d  I n i t
 *
 *  Function:
 *     Initialise the PAF file access routines.
 *
 *  Description:
 *     This routine should be called before any call is made to PAF_ReadFile()
 *     or to PAF_GetValue(). It initialises the various global variables
 *     used by these routines.
 */
 
static void PAF_ReadInit (void)
{
   PAF_Lines = (char**) NULL;
   PAF_Array = (char*) NULL;
   PAF_ErrorString[0] = '\0';
   PAF_CurrentLine = 0;
}

/* -------------------------------------------------------------------------- */

/*                  P A F _  R e a d  C l o s e
 *
 *  Function:
 *     Close down the PAF file access routines.
 *
 *  Description:
 *     This routine closes down and releases any resources allocated by
 *     PAF_ReadFile(). It should be called once all required calls to
 *     PAF_GetValue() have been made.
 */
 
static void PAF_ReadClose (void) 
{
   if (PAF_Array) {
      free (PAF_Array);
      PAF_Array = (char*) NULL;
   }
   if (PAF_Lines) {
      free (PAF_Lines);
      PAF_Lines = (char**) NULL;
   }
}

/* -------------------------------------------------------------------------- */

/*                  P A F _  G e t  E r r o r  T e x t
 *
 *  Function:
 *     Returns a pointer to text describing the last error in the PAF routines.
 *
 *  Description:
 *     If an error occurs in a PAF_ routine, a description of it is placed
 *     in a string whose address is returned by this routine.
 */
 
char* PAF_GetErrorText (void)
{
   return PAF_ErrorString;
}

/* -------------------------------------------------------------------------- */

/*                  P A F _  R e a d  L i s t
 *
 *  Function:
 *     Lists the keyword lines read by PAF_ReadFile().
 *
 *  Description:
 *     This is a diagnostic routine that lists all the non-comment lines
 *     that were read from the PAF file by PAF_ReadFile().
 */
 
static void PAF_ReadList (void)
{
   int LineIndex;                       /* Index into PAF_Lines array */
   
   if (PAF_Lines) {
      LineIndex = 0;
      while (PAF_Lines[LineIndex]) {
         printf ("%s\n",PAF_Lines[LineIndex]);
         LineIndex++;
      }
   }
}

/* -------------------------------------------------------------------------- */


/*                  P A F _  R e a d  F i l e
 *
 *  Function:
 *     Reads into memory the keywords from a specified PAF file.
 *
 *  Description:
 *     This routine reads all the non-comment lines from a specified
 *     PAF file and stores them in memory at locations given in the
 *     elements of the global array variable PAF_Lines. So PAF_Lines[0]
 *     has the address of the start of a string that contains the text
 *     (nul-terminated, but with no newline character) from the first
 *     non-comment line in the file, PAF_Lines[1] has the next line and
 *     so on. If there were n such non-comment lines read from the file,
 *     PAF_Lines[n] will contain a null address, indicating the end of the
 *     set of lines from the file. Once PAF_ReadFile() has been called,
 *     individual keyword values can be accessed through PAF_GetValue().
 */

static void PAF_ReadFile (
   char* Filename,
   StatusType* Status)
{
   /*  Local variables */
   
   int Bytes;                           /* Bytes needed for line ptr array */
   int BytesLeft;                       /* Bytes remaining in PAF_Array */
   int BytesUsed;                       /* Bytes needed for one line */
   char* LastCharPtr;                   /* Points to last char in line */
   char Line[512];                      /* Used to read file on first pass */
   int LineCount;                       /* # of non-comment lines in file */
   int LineIndex;                       /* Index into PAF_Lines array */
   int LineLength;                      /* Number of characters in line */
   int LineNumber;                      /* Line number - for error messages */
   char* LinePtr;                       /* Points to character in each line */
   FILE* TheFile = (FILE*) NULL;        /* The PAF file */
   char* PAF_LinePtr;                   /* Current pointer into PAF_Array */
   int TotalFileBytes;                  /* Bytes in non-comment lines */
   
   if (*Status != STATUS__OK) return;
   
   /*  First, we open the file */

   if ((TheFile = fopen(Filename, "r")) == (FILE*) NULL) {
      sprintf(PAF_ErrorString, "Cannot open file %s", Filename);
      ErsRep (0,Status,PAF_ErrorString);
      *Status = 1;
      goto Exit;
   }
         
   /*  To make it easier to search through the file, we read it into a
    *  malloced array of character strings. So first, we do a pass through
    *  the file to see how many records it has and how many bytes are
    *  needed to hold them all the lines we're interested in.
    */

   LineCount = 0;
   LineNumber = 0;
   TotalFileBytes = 0;
   for (;;) {
      if (fgets(Line,sizeof(Line),TheFile) == (char*)NULL) break;
      LineNumber++;
      
      /*  Look through the line for the first non-blank character. If there
       *  is one, and it isn't a comment character ('#'), then this is a
       *  line we're interested in. We also include the ("#1") pseudo-comment
       *  lines included by FP OSS, so have to test for them explicitly.
       */
       
      LinePtr = Line;
      while (*LinePtr) {
         if (*LinePtr == '\n') break;
         if (*LinePtr != ' ') {
            if ((*LinePtr != '#') ||
                  ((*LinePtr == '#') && (*(LinePtr + 1) == '!'))) {
               LineCount++;
               
               /*  Note that the number of bytes needed for this line has to
                *  allow for the terminating nul.
                */
                
               BytesUsed = strlen(Line) + 1;
               TotalFileBytes += BytesUsed;
            }
            break;
         }
         LinePtr++;
      }
   }
       
   /*  Now allocate an array to hold the non-comment lines, together with
    *  an index array that can be used to access each line individually.
    *  Note that we allow for one extra line of 80 bytes, just so we can
    *  safely issue the final fgets() call that ought to return the NULL
    *  at the end of the file.
    */
   
   LineCount++;
   Bytes = LineCount * sizeof(char*);
   PAF_Lines = (char**) malloc (Bytes);
   if (PAF_Lines == (char**) NULL) {
      sprintf(PAF_ErrorString, 
         "Cannot allocate %d bytes for line pointers for file %s",
                                                              Bytes,Filename);
      ErsRep (0,Status,PAF_ErrorString);
      *Status = 1;
      goto Exit;
   }
   TotalFileBytes += 80;    
   PAF_Array = (char*) malloc (TotalFileBytes);
   if (PAF_Array == (char*) NULL) {
      sprintf(PAF_ErrorString, "Cannot allocate %d bytes to read file %s",
                                                      TotalFileBytes,Filename);
      ErsRep (0,Status,PAF_ErrorString);
      *Status = 1;
      goto Exit;
   }

   /*  And now read the non-comment lines into the array.  We read each 
    *  line again, but we effectively only keep the non-comment lines,
    *  since it is only after reading one of them that we increment the
    *  various pointers.
    */ 
         
   LineIndex = 0;
   PAF_LinePtr = PAF_Array;
   BytesLeft = TotalFileBytes;
   rewind (TheFile);
   for (;;) {
      if (LineIndex >= LineCount) {
         sprintf(PAF_ErrorString,
            "Internal error reading file %s - size changed",Filename);
         ErsRep (0,Status,PAF_ErrorString);
         *Status = 1;
         goto Exit;
      }
      PAF_Lines[LineIndex] = PAF_LinePtr;
      if (fgets(PAF_LinePtr,BytesLeft,TheFile) == (char*)NULL) {
         PAF_Lines[LineIndex] = (char*) NULL;
         break;
      }
      
      /*  The test we do here to see if we're interested in the line is
       *  exactly the same as the test performed in the first pass.
       */
       
      LinePtr = PAF_LinePtr;
      while (*LinePtr) {
         if (*LinePtr == '\n') break;
         if (*LinePtr != ' ') {
            if ((*LinePtr != '#') ||
                  ((*LinePtr == '#') && (*(LinePtr + 1) == '!'))) {
                  
               /*  This is a line we're interested in. Since fgets() transmits
                *  newline characters, which we don't really want, we check for
                *  the presence of such a character and remove it. This is
                *  messy, but simplifies things later. We also remove any
                *  terminating semi-colons and quotes - see comments to
                *  PAF_GetValue() for an explanation of this. We just replace
                *  these in situ with nul characters, so most lines will
                *  end up with more than one nul at the end, but that 
                *  doesn't matter, since they're accessed through the
                *  PAF_Lines[] array, not by working through the characters
                *  from the file counting nuls to see which line we're at.
                */
               
               LineLength = strlen(PAF_LinePtr);
               BytesUsed = LineLength + 1;
               LastCharPtr = &(PAF_LinePtr[LineLength - 1]);
               if (LineLength > 0) {
                  LastCharPtr = &(PAF_LinePtr[LineLength - 1]);
                  if (*LastCharPtr == '\n') {
                     *LastCharPtr = '\0';
                     LastCharPtr--;
                  }
                  if (*LastCharPtr == ';') {
                     *LastCharPtr = '\0';
                     LastCharPtr--;
		     while (*LastCharPtr == ' ')
		     	{
			*LastCharPtr = '\0';
			LastCharPtr--;
			}
                  }
                  if (*LastCharPtr == '"') {
                     *LastCharPtr = '\0';
                     LastCharPtr--;
                  }
                  
                  /*  Finally, we have a line we want, the way we want it.
                   *  Modify the pointers and counters appropriately. Note
                   *  that BytesUsed has been manipulated so it includes
                   *  all the terminating nul character(s).
                   */
                      
                  LineIndex++;
                  BytesLeft -= BytesUsed;
                  PAF_LinePtr += BytesUsed;
               }
            }
            break;
         }
         LinePtr++;
      }
      
      /*  We make sure we haven't exhausted our array - the only way this
       *  can happen is if the file is changing underneath us, or if there's
       *  a bug. 
       */
       
      if (BytesLeft <= 0) {
         sprintf(PAF_ErrorString,
            "Internal error reading file %s - size changed",Filename);
         ErsRep (0,Status,PAF_ErrorString);
         *Status = 1;
         goto Exit;
      }
   }
   
   /*  We ought to end up with BytesLeft set to exactly the 80 extra bytes we
    *  allowed for the last read. If the code is changed significantly, 
    *  adding a test for that at this point might be a good idea.
    */
   
   /*  The next line to be looked at is obviously the first line in the file */
   
   PAF_CurrentLine = 0;
   
Exit:;

   if (TheFile != (FILE*) NULL) fclose(TheFile);
}

/* -------------------------------------------------------------------------- */

/*                      P A F _  G e t  V a l u e
 *
 *  Function:
 *     Gets the text associated with a specified PAF keyword.
 *
 *  Description:
 *
 *     This routine looks up the value associated with the specified
 *     keyword and returns a pointer to that value. If the value is
 *     numeric, the pointer will point to a number that is terminated
 *     by a nul character (not by a semi-colon, as in the original
 *     PAF file). If the value is a character string, the pointer will
 *     point to a leading '"' character. However, the string that follows
 *     this will end directly with a nul character - the terminating '"'
 *     and ';' in the original PAF file are removed, and the leading '"'
 *     (now unmatched!) is only retained to indicate that this is a string.
 *     This allows higher level code to step over the leading quote
 *     and to treat the rest as an ordinary nul-terminated string, which
 *     makes it possible to use the string value directly without having
 *     to copy it and take responsibility for memory management of the
 *     copied strings.
 * 
 *     This routine assumes that PAF_ReadFile() has been called and so all
 *     the text lines from a PAF file are indexed via the PAF_Lines[]
 *     global array.
 *
 *     This routine is not particularly clever in the way it searches the
 *     text lines read from the PAF file, and is very inefficient if used
 *     to look for keywords in an order that doesn't match that of the
 *     input file.
 */

static char* PAF_GetValue (
   char* Keyword)
{
   /*  Local variables */

   int Length;                          /* Number of characters in Keyword */
   int LineIndex;                       /* Index through lines read from file */
   char* LinePtr;                       /* Works through characters in line */
   char* ValuePtr;                      /* Points to required value string */  
   short WrappedRound;                  /* Flag to indicate gone past end */
   
   ValuePtr = (char*) NULL;
   
   if (PAF_Lines) {

      LineIndex = PAF_CurrentLine;
      WrappedRound = FALSE;
      
      for (;;) {
      
         /*  If at the line we started at and we've wrapped round then we've
          *  searched the whole file. Time to quit.
          */
          
         if (LineIndex == PAF_CurrentLine) {
            if (WrappedRound) break;
         }
         
         /*  If we find ourselves at the NULL pointer that marks the end of
          *  the line address array, start again at the bottom - and note
          *  that we've done so. If we've hit it twice, break out.
          */
          
         if (PAF_Lines[LineIndex] == (char*) NULL) {
            if (WrappedRound) break;
            WrappedRound = TRUE;
            LineIndex = 0;
         }
         
         /*  Now, see if this line is the one we want. If the line read from
          *  the file starts with a '#' it must be one of the '#1' pseudo-
          *  comments - they're the only comments PAF_FileRead() lets through,
          *  so we move along two characters. Then if we have a match we
          *  look for the start of the value string - the next non-blank
          *  character. Note that we have to match the keyword, then check 
          *  that the next character in the line is blank - otherwise we've
          *  just found another keyword of which ours is an abbreviation.
          */
          
         LinePtr = PAF_Lines[LineIndex];
         if (*LinePtr == '#') LinePtr += 2;
         Length = strlen(Keyword);
         if (!strncmp(LinePtr,Keyword,Length)) {
            LinePtr += Length;
            if (*LinePtr == ' ') {
               while (*LinePtr) {
                  if (*LinePtr != ' ') {
                     ValuePtr = LinePtr;
                     PAF_CurrentLine = LineIndex + 1;
                     break;
                  }
                  LinePtr++;
               }
               break;
            }
         }
         
         /*  And then look at the next line, if this wasn't the one we
          *  wanted. This is really pretty crude, and relies on the caller
          *  really knowing the order of things in the file. A possible
          *  speed-up might be to see if all of the keyword bar the last
          *  field matches. If it doesn't, but it did when we started, then
          *  we've probably gone past and might do better to search backwards.
          *
          *  Note that now that we look for the optional keywords - the
          *  ones such as PM-RA, PM-DEC, BAND, etc, - these are often not
          *  present and in that case this routine will search the whole
          *  file for each one. That is going to be very inefficient. The
          *  question must be, is it OK to assume that all keywords for the
          *  same target are in fact collected together, in which case you
          *  can go on to where all bar the last field matches, then back
          *  to find the start of the block for that object, and not look
          *  further for keywords you don't find? In any case, it's only
          *  the really long files written in expert mode, with all the
          *  unallocated targets, that are slow to process.
          */
          
         LineIndex++;
      }
      
   }
   
   return ValuePtr;
}

/* -------------------------------------------------------------------------- */

/*                      P A F _  G e t  R A
 *
 *  Function:
 *     Gets an RA value associated with a specified PAF keyword.
 *
 *  Description:
 *     This routine looks up the value associated with the specified
 *     keyword and interprets it as a right ascension value in the
 *     standard ESO PAF format. It returns this as a value in radians.
 */
 
static double PAF_GetRA (
   char* Keyword,
   StatusType* Status)
{
   int Istat;
   char* ValuePtr;
   double RaInRadians = 0.0;
   
   if (*Status != STATUS__OK) return 0.0;
   
   if ((ValuePtr = PAF_GetValue (Keyword)) != (char*) NULL) {
      
      /*  The PAF convention is that an RA is formatted as a number,
       *  but this is just a fancy encoding - the number is hhmmss.nn
       *  so an RA of 10:04:33.70 is formatted as 100433.70
       */
      
      int Hours = 0;
      int Minutes = 0;
      double Seconds = 0.0;
      sscanf (ValuePtr,"%2d%2d%5lf",&Hours,&Minutes,&Seconds);
      
      /*  Now convert to radians */
      
      slaDtf2r(Hours,Minutes,Seconds,&RaInRadians,&Istat);
      if ((Istat != 0) || (Hours > 23) || (Hours < 0)) {
         *Status = 1;
         RaInRadians = 0.0;
         sprintf (PAF_ErrorString,"Invalid Right Ascension - %s",ValuePtr);
         ErsRep (0,Status,PAF_ErrorString);
      }
   } else {
   
      *Status = 1;
      sprintf (PAF_ErrorString,"Unable to find keyword - %s",Keyword);
      ErsRep (0,Status,PAF_ErrorString);
   }
   return RaInRadians;
}
      
/* -------------------------------------------------------------------------- */

/*                      P A F _  G e t  D e c
 *
 *  Function:
 *     Gets an Declination value associated with a specified PAF keyword.
 *
 *  Description:
 *     This routine looks up the value associated with the specified
 *     keyword and interprets it as a declination value in the
 *     standard ESO PAF format. It returns this as a value in radians.
 */
 
static double PAF_GetDec (
   char* Keyword,
   StatusType* Status)
{
   int Istat;
   char* ValuePtr;
   double DecInRadians = 0.0;
   
   if (*Status != STATUS__OK) return 0.0;
   
   if ((ValuePtr = PAF_GetValue (Keyword)) != (char*) NULL) {
      
      /*  The PAF convention is that a declination is formatted as a number,
       *  but this is just a fancy encoding - the number is [-]ddmmss.n
       *  so a dec of -31.27.30.1 is formatted as -312730.1
       */
      
      short Negative = FALSE;
      int Degrees = 0;
      int Minutes = 0;
      double Seconds = 0.0;
      char* StringPtr = ValuePtr;
      
      if (*StringPtr == '+') {
         StringPtr++;
      } else if (*StringPtr == '-') {
         StringPtr++;
         Negative = TRUE;
      }
      sscanf (StringPtr,"%2d%2d%4lf",&Degrees,&Minutes,&Seconds);
      
      /*  Now convert to radians */
      
      slaDaf2r(Degrees,Minutes,Seconds,&DecInRadians,&Istat);
      if ((Istat != 0) || (Degrees > 90)) {
         *Status = 1;
         DecInRadians = 0.0; 
         sprintf (PAF_ErrorString,"Invalid Declination - %s",ValuePtr);
         ErsRep (0,Status,PAF_ErrorString);
      } else {
         if (Negative) DecInRadians = -DecInRadians;
      }
   } else {
   
      *Status = 1;
      sprintf (PAF_ErrorString,"Unable to find keyword - %s",Keyword);
      ErsRep (0,Status,PAF_ErrorString);
   }
   return DecInRadians;
}

/* -------------------------------------------------------------------------- */

/*                      P A F _  G e t  S t r i n g
 *
 *  Function:
 *     Gets a string associated with a specified PAF keyword.
 *
 *  Description:
 *     This routine looks up the value associated with the specified
 *     keyword, checks that it is a string value and returns a pointer
 *     to the string. The pointer points to the first character of the
 *     string (not to the leading '"' character in the PAF file) and
 *     the string is terminated with a nul character (there is no closing
 *     '"' or ';' or newline as in the PAF file). If the keyword cannot be
 *     found, a blank string is returned.
 */
 
static char* PAF_GetString (
   char* Keyword,
   StatusType* Status)
{
   char* ValuePtr = "";
   
   if (*Status != STATUS__OK) return (char*) NULL;
   
   if ((ValuePtr = PAF_GetValue (Keyword)) != (char*) NULL) {
      
      /*  The string should start with a (mis-matched) leading '"'.
       *  We check for that and skip over it if it is present. If it
       *  is missing we ignore that - maybe there should be an error
       *  condition raised?
       */
      
      if (*ValuePtr == '"') ValuePtr++;
   } else {
   
      *Status = 1;
      sprintf (PAF_ErrorString,"Unable to find keyword - %s",Keyword);    
      ErsRep (0,Status,PAF_ErrorString);
      ValuePtr = "";
   }
   
   return ValuePtr;
}

/* -------------------------------------------------------------------------- */

/*                      P A F _  G e t  N u m b e r
 *
 *  Function:
 *     Gets a numeric value associated with a specified PAF keyword.
 *
 *  Description:
 *     This routine looks up the numeric value associated with the specified
 *     keyword. If no such keyword exists, zero is returned.
 */
 
static double PAF_GetNumber (
   char* Keyword,
   StatusType* Status)
{
   double Value = 0.0;
   char* ValuePtr;
   
   if (*Status != STATUS__OK) return 0.0;
   
   if ((ValuePtr = PAF_GetValue (Keyword)) != (char*) NULL) {
      
      /*  The string should be a formatted number. Should check for
       *  a properly formatted number *********
       */
      
      sscanf (ValuePtr,"%lf",&Value);
      
   } else {
   
      *Status = 1;
      sprintf (PAF_ErrorString,"Unable to find keyword - %s",Keyword);      
      ErsRep (0,Status,PAF_ErrorString);
      Value = 0.0;
   }
   
   return Value;
}

/* -------------------------------------------------------------------------- */

/*                      P A F _  G e t  O b j e c t  D a t a
 *
 *  Function:
 *     Gets data about one object from a PAF file.
 *
 *  Description:
 *     Given the base name common to all the keywords for the object in
 *     question, this routine fills up the fields of a PAF_ObjectDataType
 *     structure with the information held in the current PAF file about
 *     the object.
 */

static void PAF_GetObjectData (
   FpilType *Instrument,                   /* Details of current instrument */
   char* BaseName,                         /* Base part of keyword spec */
   int Index,                              /* Object index - from 0 */
   int Flags,                              /* Flags controlling details */
   PAF_ObjectDataType* ObjectDataPtr,      /* Pointer to data structure */
   StatusType* Status)                     /* Inherited status */
{
   char Keyword[64];
   int KeyCount;
   int Key;
   char* StringPtr;
   char ValueString[128];
   
   /*  These define the optional keywords normally encoded into the
    *  comment string. These defintions need to be exactly the same
    *  as those included in the code for ConfParseComment() in the
    *  file configure.c.
    */
    
   static char* Keywords[] = {
       "BAND", "B-V", "V-R", "PM-RA", "PM-DEC", "SNR", "CATEGORY", "DIAMETER" };
   static char* PAFKeywords[] = {
       "BAND", "B_V", "V_R", "PM_RA", "PM_DEC", "SNR", "CATEGORY", "DIAMETER" };
   static short KeywordIsChar[] = {
        TRUE,   FALSE, FALSE, FALSE,   FALSE,    FALSE, TRUE,       FALSE };
       
   static int NumKeywords = sizeof(Keywords)/sizeof(char*);
    
   if (*Status != STATUS__OK) return;
   
   sprintf (Keyword,"%s%d.NAME",BaseName,Index + 1);
   ObjectDataPtr->Name = PAF_GetString(Keyword,Status);
   sprintf (Keyword,"%s%d.RA",BaseName,Index + 1);
   ObjectDataPtr->Ra = PAF_GetRA(Keyword,Status);
   sprintf (Keyword,"%s%d.DEC",BaseName,Index + 1);
   ObjectDataPtr->Dec = PAF_GetDec(Keyword,Status);
   /*  Note that we set X and Y to zero - we assume they'll be
    *  calculated properly by the higher levels */
   ObjectDataPtr->X = 0.0;
   ObjectDataPtr->Y = 0.0;
   sprintf (Keyword,"%s%d.TYPE",BaseName,Index + 1);
   FpilEncodeTargetType(*Instrument,PAF_GetString(Keyword,Status),
                 &(ObjectDataPtr->Type),&(ObjectDataPtr->Spectrograph));
   /*  We set Theta to zero as well. That will be calculated later */
   ObjectDataPtr->Theta = 0.0;
   sprintf (Keyword,"%s%d.PRIOR",BaseName,Index + 1);
   ObjectDataPtr->Priority = (short) PAF_GetNumber(Keyword,Status);
   sprintf (Keyword,"%s%d.MAG",BaseName,Index + 1);
   ObjectDataPtr->Magnitude = PAF_GetNumber(Keyword,Status);
   sprintf (Keyword,"%s%d.PRGID",BaseName,Index + 1);
   ObjectDataPtr->PId = (short) PAF_GetNumber(Keyword,Status);
   sprintf (Keyword,"%s%d.COMMENT",BaseName,Index + 1);
   ObjectDataPtr->Comment = PAF_GetString(Keyword,Status);
   if (Flags & FIBRE_FLAG) {
      sprintf (Keyword,"%s%d.FIBRE",BaseName,Index + 1);
      ObjectDataPtr->Fibre = (int) PAF_GetNumber(Keyword,Status);
   } else {
      ObjectDataPtr->Fibre = -1;
   }
   
   /*  Now we have to look for the optional keywords that are normally
    *  encoded at the start of the comment string. If we find any, we
    *  need to modify the comment string to include these optional
    *  values. We format any we find into the character string held
    *  in the object data structure for just this purpose.
    */
   
   KeyCount = 0;
   ObjectDataPtr->CommentString[0] = '\0';
   for (Key = 0; Key < NumKeywords; Key++) {
      sprintf (Keyword,"%s%d.%s",BaseName,Index + 1,PAFKeywords[Key]);
      StringPtr = PAF_GetValue(Keyword);
      if (StringPtr) {
         if (!strncmp(PAFKeywords[Key],"PM_RA",5)) {
            if (atof(PAF_GetValue("INS.OSS.VER")+1) > 186402. ) {
               sprintf(StringPtr,"%*f",(int) strlen(StringPtr),atof(StringPtr)*cos(ObjectDataPtr->Dec));
            }
         }
         if (KeyCount > 0) strcat(ObjectDataPtr->CommentString," ");
         KeyCount++;
         if (KeywordIsChar[Key]) {
            if (*StringPtr == '"') StringPtr++;
         }
         sprintf (ValueString,"(%s=%s)",Keywords[Key],StringPtr);
         strcat (ObjectDataPtr->CommentString,ValueString);
      }
   }
   
   /*  If there were additional optional keywords, append the actual
    *  comment string to the end of them, and set the Comment pointer
    *  in the object data structure to point to the resulting string.
    */
    
   if (KeyCount > 0) {
      if (ObjectDataPtr->Comment[0] != '\0') {
         strcat(ObjectDataPtr->CommentString," ");
         strcat (ObjectDataPtr->CommentString,ObjectDataPtr->Comment);
      }
      ObjectDataPtr->Comment = ObjectDataPtr->CommentString;
   }
}

/* -------------------------------------------------------------------------- */

/*                P A F _  C r e a t e  O b j  S t r u c t
 *
 *  Function:
 *     Creates an structure describing a set of objects.
 *
 *  Description:
 *     This routine creates the arrays in one of the unallocGuide or 
 *     unallocObject structures. It needs the Sds Id of the overall
 *     structure and the number of objects that have to be included.
 *     It creates the relevant arrays and sets their Sds Ids into
 *     the structure it is passed for the purpose.
 */

static void PAF_CreateObjStruct (
   SdsIdType LevelId,
   int Elements,
   int Flags,
   PAF_StructIdType* IdStructPtr,
   StatusType* Status)
{
   unsigned long dims[2];
   
   if (*Status != STATUS__OK) return;
   
   dims[1] = Elements;
   dims[0] = NAME_LENGTH;
   SdsNew(LevelId,"name",0,NULL,SDS_CHAR,2,dims,&IdStructPtr->nameId,Status);
   SdsNew(LevelId,"ra",0,NULL,SDS_DOUBLE,1,&dims[1],&IdStructPtr->raId,Status);
   SdsNew(LevelId,"dec",0,NULL,SDS_DOUBLE,1,&dims[1],&IdStructPtr->decId,
                                                                        Status);
   SdsNew(LevelId,"x",0,NULL,SDS_INT,1,&dims[1],&IdStructPtr->xId,Status);
   SdsNew(LevelId,"y",0,NULL,SDS_INT,1,&dims[1],&IdStructPtr->yId,Status);
   SdsNew(LevelId,"type",0,NULL,SDS_CHAR,1,&dims[1],&IdStructPtr->typeId,
                                                                        Status);
   if (Flags & DHDX_FLAG) {
      SdsNew(LevelId,"dhdx",0,NULL,SDS_DOUBLE,1,&dims[1],&IdStructPtr->dhdxId,
                                                                        Status);
      SdsNew(LevelId,"dddx",0,NULL,SDS_DOUBLE,1,&dims[1],&IdStructPtr->dddxId,
                                                                        Status);
      SdsNew(LevelId,"dhdy",0,NULL,SDS_DOUBLE,1,&dims[1],&IdStructPtr->dhdyId,
                                                                        Status);
      SdsNew(LevelId,"dddy",0,NULL,SDS_DOUBLE,1,&dims[1],&IdStructPtr->dddyId,
                                                                        Status);
   } else {
      IdStructPtr->dhdxId = 0;
      IdStructPtr->dddxId = 0;
      IdStructPtr->dhdyId = 0;
      IdStructPtr->dddyId = 0;
   }
   if (Flags & THETA_FLAG) {
      SdsNew(LevelId,"theta",0,NULL,SDS_DOUBLE,1,&dims[1],&IdStructPtr->thetaId,
                                                                     Status);
   } else {
      IdStructPtr->thetaId = 0;
   }
   SdsNew(LevelId,"spectrograph",0,NULL,SDS_CHAR,1,&dims[1],
                                            &IdStructPtr->spectId,Status);
   SdsNew(LevelId,"priority",0,NULL,SDS_SHORT,1,&dims[1],
                                          &IdStructPtr->priorityId,Status);
   SdsNew(LevelId,"magnitude",0,NULL,SDS_DOUBLE,1,&dims[1],&IdStructPtr->magId,
                                                                     Status);
   SdsNew(LevelId,"pId",0,NULL,SDS_SHORT,1,&dims[1],&IdStructPtr->pId,Status);
   dims[0] = COMMENT_LENGTH;
   SdsNew(LevelId,"comment",0,NULL,SDS_CHAR,2,dims,&IdStructPtr->commentId,
                                                                      Status);
}

/* -------------------------------------------------------------------------- */

/*                P A F _  L o c a t e  O b j  I d s
 *
 *  Function:
 *     Locates arrays in an object structure and returns their Sds Ids.
 *
 *  Description:
 *     This routine locates the arrays in one of the unallocGuide or 
 *     unallocObject or objects structures. It needs the Sds Id of the
 *     overall.  It locatess the relevant arrays and sets their Sds Ids into
 *     the structure it is passed for the purpose.
 *     At the moment, this is only used for the 'objects' structure, but
 *     it could be used for any of the structures.
 */

static void PAF_LocateObjIds (
   SdsIdType LevelId,
   int Flags,
   PAF_StructIdType* IdStructPtr,
   StatusType* Status)
{
   if (*Status != STATUS__OK) return;
   
   SdsFind(LevelId,"name",&IdStructPtr->nameId,Status);
   SdsFind(LevelId,"ra",&IdStructPtr->raId,Status);
   SdsFind(LevelId,"dec",&IdStructPtr->decId,Status);
   SdsFind(LevelId,"x",&IdStructPtr->xId,Status);
   SdsFind(LevelId,"y",&IdStructPtr->yId,Status);
   if (Flags & THETA_FLAG) {
      SdsFind(LevelId,"theta",&IdStructPtr->thetaId,Status);
   } else {
      IdStructPtr->thetaId = 0;
   }
   SdsFind(LevelId,"type",&IdStructPtr->typeId,Status);
   if (Flags & DHDX_FLAG) {
      SdsFind(LevelId,"dhdx",&IdStructPtr->dhdxId,Status);
      SdsFind(LevelId,"dddx",&IdStructPtr->dddxId,Status);
      SdsFind(LevelId,"dhdy",&IdStructPtr->dhdyId,Status);
      SdsFind(LevelId,"dddy",&IdStructPtr->dddyId,Status);
   } else {
      IdStructPtr->dhdxId = 0;
      IdStructPtr->dddxId = 0;
      IdStructPtr->dhdyId = 0;
      IdStructPtr->dddyId = 0;
   }
   SdsFind(LevelId,"spectrograph",&IdStructPtr->spectId,Status);
   SdsFind(LevelId,"priority",&IdStructPtr->priorityId,Status);
   SdsFind(LevelId,"magnitude",&IdStructPtr->magId,Status);
   SdsFind(LevelId,"pId",&IdStructPtr->pId,Status);
   SdsFind(LevelId,"comment",&IdStructPtr->commentId,Status);
}


/* -------------------------------------------------------------------------- */

/*                      P A F _  F r e e  I d s
 *
 *  Function:
 *     Releases the SDS ids held in a PAF_StructIdType structure.
 *
 *  Description:
 *     Given a pointer to a structure of type PAF_StructIdType, this routine
 *     releases all the SDS Ids that it contains.
 */
 
static void PAF_FreeIds (
   PAF_StructIdType* IdStructPtr,        /* Structure containing SDS Ids */
   StatusType* Status)                   /* Inherited status */
{
   if (*Status != STATUS__OK) return;
   
   SdsFreeId(IdStructPtr->nameId,Status);
   SdsFreeId(IdStructPtr->raId,Status);
   SdsFreeId(IdStructPtr->decId,Status);
   SdsFreeId(IdStructPtr->xId,Status);
   SdsFreeId(IdStructPtr->yId,Status);
   SdsFreeId(IdStructPtr->typeId,Status);
   if (IdStructPtr->dhdxId) SdsFreeId(IdStructPtr->dhdxId,Status);
   if (IdStructPtr->dhdyId) SdsFreeId(IdStructPtr->dhdyId,Status);
   if (IdStructPtr->dddxId) SdsFreeId(IdStructPtr->dddxId,Status);
   if (IdStructPtr->dddyId) SdsFreeId(IdStructPtr->dddyId,Status);
   if (IdStructPtr->thetaId) SdsFreeId(IdStructPtr->thetaId,Status);
   SdsFreeId(IdStructPtr->spectId,Status);
   SdsFreeId(IdStructPtr->priorityId,Status);
   SdsFreeId(IdStructPtr->magId,Status);
   SdsFreeId(IdStructPtr->pId,Status);
   SdsFreeId(IdStructPtr->commentId,Status);
}
/* -------------------------------------------------------------------------- */

/*                   P A F _  S e t  O b j e c t  D a t a
 *
 *  Function:
 *     Creates an SDS structure based on the contents of a PAF file.
 *
 *  Description:
 *     Given the details about an object in a PAF_ObjectDataType structure,
 *     and a set of SDS Ids for the arrays in a standard configure program
 *     structure, this routine enters the details into the arrays, at a
 *     specified index position.
 */
 
static void PAF_SetObjectData (
   PAF_StructIdType* IdStructPtr,        /* Structure containing SDS Ids */
   int Index,                            /* Object index - from 0 */
   PAF_ObjectDataType* ObjectDataPtr,    /* Structure containing object data */
   StatusType* Status)                   /* Inherited status */
{
   double DZero = 0.0;
   
   if (*Status != STATUS__OK) return;
   
   /*  These items are common to all structures */
   
   SdsPut(IdStructPtr->nameId,NAME_LENGTH*sizeof(char),Index*NAME_LENGTH,
                                               ObjectDataPtr->Name,Status);
   SdsPut(IdStructPtr->raId,sizeof(double),Index,&ObjectDataPtr->Ra,Status);
   SdsPut(IdStructPtr->decId,sizeof(double),Index,&ObjectDataPtr->Dec,Status);
   SdsPut(IdStructPtr->xId,sizeof(int),Index,&ObjectDataPtr->X,Status);
   SdsPut(IdStructPtr->yId,sizeof(int),Index,&ObjectDataPtr->Y,Status);
   SdsPut(IdStructPtr->typeId,sizeof(char),Index,&ObjectDataPtr->Type,Status);
   SdsPut(IdStructPtr->spectId,sizeof(char),Index,&ObjectDataPtr->Spectrograph,
                                                                        Status);
   SdsPut(IdStructPtr->priorityId,sizeof(short),Index,&ObjectDataPtr->Priority,
                                                                        Status);
   SdsPut(IdStructPtr->magId,sizeof(double),Index,&ObjectDataPtr->Magnitude,
                                                                        Status);
   SdsPut(IdStructPtr->pId,sizeof(short),Index,&ObjectDataPtr->PId,Status);
   SdsPut(IdStructPtr->commentId,COMMENT_LENGTH*sizeof(char),
                           Index*COMMENT_LENGTH,ObjectDataPtr->Comment,Status);
   
   /*  If the structure contains the dhdx and related items, we set their
    *  values to zero, assuming they'll be set properly at some later date
    */
    
   if (IdStructPtr->dhdxId) {
      SdsPut(IdStructPtr->dhdxId,sizeof(double),Index,&DZero,Status);
   }
   if (IdStructPtr->dhdyId) {
      SdsPut(IdStructPtr->dhdyId,sizeof(double),Index,&DZero,Status);
   }
   if (IdStructPtr->dddxId) {
      SdsPut(IdStructPtr->dddxId,sizeof(double),Index,&DZero,Status);
   }
   if (IdStructPtr->dddyId) {
      SdsPut(IdStructPtr->dddyId,sizeof(double),Index,&DZero,Status);
   }
   
   /*  If the structure contains 'theta' we set that as well. */
   
   if (IdStructPtr->thetaId) {
      SdsPut(IdStructPtr->thetaId,sizeof(double),Index,&ObjectDataPtr->Theta,
                                                                        Status);
   }
}

/* -------------------------------------------------------------------------- */

/*                      P A F _  C r e a t e  S d s
 *
 *  Function:
 *     Creates an SDS structure based on the contents of a PAF file.
 *
 *  Description:
 *     This is the main external routine provided in the PAF input code.
 *     It reads in a PAF file whose name is passed to it, and returns
 *     the Sds Id of an Sds structure as required by Configure. It also
 *     returns details of the VLT guide probe position, if one was specified.
 *     It needs to be passed the details of the instrument in use, so it
 *     can make use of Fpil routines and it also needs to be passed the
 *     Sds Id of the pivot information structure. It also returns an integer
 *     code giving the fibre combination in use, if this can be determined.
 */
 
void PAF_CreateSds (
   FpilType *Instrument,
   char* FileName,
   SdsIdType PivId,
   SdsIdType* FileId,
   PAF_VLTGuideInfoType* GuideInfoPtr,
   int *ModeCode,
   double *checkHA,
   StatusType* Status)
{
   short ArgusInUse;
   char* ArgusUsedString;
   double ArgusAngle;
   char* ArgusScale;
   char Scale[16];
   double ArgusOffset;
   int Index;
   char Keyword[64];
   char* ValuePtr;
   short UnallocObjects;
   short UnallocGuides;
   short UnallocSky;
   short AllocGuides;
   short AllocObjects;
   short AllocSky;
   short VLTGuiderTargets;
   short UnallocFACBTargets;
   double CenRa;
   double CenDec;
   double AppRa;
   double AppDec;
   double Mjd;
   double AppEpoch;
   char* Label;
   StatusType LocalStatus;
   SdsIdType fId;
   SdsIdType ugId;
   SdsIdType uoId;
   SdsIdType ObjId;
   SdsIdType tmpId;
   unsigned long dims[2];
   PAF_ObjectDataType ObjectData;
   PAF_StructIdType IdStruct;
   int NumTypes;
   int IType;
   char InsCode;
   char* AbbrNames[10];
   char* TypeNames[10];
   int TypeCount[10];
   char* PPosPtr;
   short GuideStarSel;
   double GuideRA;
   double GuideDec;
   int Length;
   int LastDot;
   int FirstDot;
      
   if (*Status != STATUS__OK) return;
   
   /*  Get the list of PAF types and the abbreviated names used in the
    *  PAF keywords.
    */
   
   FpilPAFTypes (*Instrument,sizeof(TypeNames)/sizeof(char*),&NumTypes,
                                                    TypeNames,AbbrNames);   
   
   PAF_ReadInit();
   PAF_ReadFile(FileName,Status);
   if (*Status != STATUS__OK) goto Exit;
   
   /*  A call to PAF_ReadList() at this point can be a useful diagnostic.
    *  Having it coded like this prevents a compiler complaining that
    *  PAF_ReadList() is static but unused, and changing the flag value
    *  will enable the diagnostics.
    */
   
   if (0) PAF_ReadList();
   
   /*  We need to know the total number of unallocated objects, both
    *  normal program targets and guide stars, and then the total number
    *  of allocated objects. Once we know these, we can create the SDS
    *  structure we need.
    */
    
   UnallocObjects = 0;
   UnallocGuides = 0;
   AllocObjects = 0;
   AllocGuides = 0;
   AllocSky = 0;
   UnallocSky = 0;
   VLTGuiderTargets = 0;
   UnallocFACBTargets = 0;
   
   /*  We calculate these values by looking at all the keywords describing
    *  the various target types and noting the point at which we can't
    *  find the a keyword with the specified index. It's also the case that
    *  most of this information is held in other keywords in the PAF file,
    *  and we could just have looked at those. Perhaps we should really
    *  do both and check that they match..
    */
    
   Index = 1;
   for (;;) {
      sprintf (Keyword,"INS.XOBJ%d.NAME",Index);
      if ((ValuePtr = PAF_GetValue (Keyword)) == (char*) NULL) {
         UnallocObjects = Index - 1 - UnallocSky;
         break;
      }
      sprintf (Keyword,"INS.XOBJ%d.TYPE",Index);
      if ((ValuePtr = PAF_GetValue (Keyword)) != (char*) NULL) {
         if (FpilIsTargetSky(*Instrument,ValuePtr[1])) UnallocSky++;
      }
      Index++;
   }
   Index = 1;
   for (;;) {
      sprintf (Keyword,"TEL.PGS%d.NAME",Index);
      if ((ValuePtr = PAF_GetValue (Keyword)) == (char*) NULL) {
          VLTGuiderTargets = Index - 1;
          UnallocGuides += Index - 1;
          break;
      }
      Index++;
   }
   Index = 1;
   for (;;) {
      sprintf (Keyword,"INS.XREF%d.NAME",Index);
      if ((ValuePtr = PAF_GetValue (Keyword)) == (char*) NULL) {
          UnallocFACBTargets = Index - 1;
          UnallocGuides += Index - 1;
          break;
      }
      Index++;
   }
   Index = 1;
   for (;;) {
      sprintf (Keyword,"INS.REF%d.NAME",Index);
      if ((ValuePtr = PAF_GetValue (Keyword)) == (char*) NULL) {
          AllocGuides = Index - 1;
          break;
      }
      Index++;
   }
   for (IType = 0; IType < NumTypes; IType++) {    
      InsCode = *(AbbrNames[IType]);
      Index = 1;
      for (;;) {
         sprintf (Keyword,"INS.%cFIB%d.NAME",InsCode,Index);
         if ((ValuePtr = PAF_GetValue (Keyword)) == (char*) NULL) {
            TypeCount[IType] = Index - 1;
            break;
         } else {
            sprintf (Keyword,"INS.%cFIB%d.TYPE",InsCode,Index);
            if ((ValuePtr = PAF_GetValue (Keyword)) != (char*) NULL) {
               if (ValuePtr[1] == 'S') {
                  AllocSky++;
               } else {
                  AllocObjects++;
               }
            }
         }
         Index++;
      }
   }
   
   /*  We need to collect together all the field details. */
   
   CenRa = PAF_GetRA ("TEL.TARG.RA",Status);
   CenDec = PAF_GetDec ("TEL.TARG.DEC",Status);
   Label = PAF_GetString ("PAF.NAME",Status);
   Mjd = PAF_GetNumber ("INS.TARG.MJD",Status);
   AppEpoch = PAF_GetNumber ("INS.TARG.APPEPOCH",Status);
   AppRa = PAF_GetRA ("INS.TARG.APPRA",Status);
   AppDec = PAF_GetDec ("INS.TARG.APPDEC",Status);
   
   /*  From the label - the value of the PAF.NAME string, we can extract
    *  the mode. It may be that we should write a special comment PAF 
    *  keyword to get this, but doing it this way has the advantage of
    *  being backwards-compatible with older PAF files written by FPOSS.
    *  The name is a string consisting of the file name, a dot, the 
    *  mode, a dot and the date. (See the code for WritePAF{}.) We
    *  extract the mode string and see which of the various modes it
    *  matches. It should match the start of one of the mode strings
    *  returned by FpilFibreCombos(). The index of the matching mode
    *  string is the mode code we return to the caller of this routine.
    *  As part of this process, we truncate the label at the dot that
    *  separates the field name from the mode and date information,
    *  because we don't want that included in the field name in the
    *  SDS structure.
    */
   
   *ModeCode = -1; 
   Length = strlen(Label);
   LastDot = -1;
   FirstDot = -1;
   for (Index = Length - 1; Index >= 0; Index--) {
      if (Label[Index] == '.') {
         if (LastDot == -1) {
            LastDot = Index;
         } else {
            FirstDot = Index;
            Label[Index] = '\0';
            break;
         }
      }
   }
   if (FirstDot != -1) {
      int IMode;
      int NumCombos;
      char* ComboNames[24];
      char* ModeNames[24];
      int ModeLength = LastDot - FirstDot - 1;
      if (ModeLength > 0) {
         FpilFibreCombos (*Instrument,24,&NumCombos,ComboNames,ModeNames);
         for (IMode = 0; IMode < NumCombos; IMode++) {
            if (!strncmp(&(Label[FirstDot + 1]),ModeNames[IMode],ModeLength)) {
               *ModeCode = IMode;
               break;
            }
         }
      }
   }     
         
   /*  See if there is a VLT guide star specified. If there is, get its
    *  RA and Dec values and the orientation of the guide probe. What we
    *  actually want is the index into the unallocated guide objects
    *  for the guide star, but we won't know that until we create that
    *  structure.
    */
    
   GuideStarSel = FALSE;
   GuideRA = GuideDec = 0.0;
   GuideInfoPtr->SelGuideIndex = 0;
   strcpy (GuideInfoPtr->Posneg,"");
   LocalStatus = 0;
   ErsPush();
   PPosPtr = PAF_GetString ("TEL.GS.PPOS",&LocalStatus);
   if (LocalStatus == 0) {
      GuideStarSel = TRUE;
      strncpy(GuideInfoPtr->Posneg,PPosPtr,sizeof(GuideInfoPtr->Posneg));
      GuideInfoPtr->Posneg[sizeof(GuideInfoPtr->Posneg) - 1] = '\0';
      GuideRA = PAF_GetRA ("TEL.GS.RA",Status);
      GuideDec = PAF_GetDec ("TEL.GS.DEC",Status);
   } else {
      ErsAnnul(&LocalStatus);
   }
   ErsPop();
   
   /*  Look for the ARGUS keywords. If ARGUS is in use, we need an angle
    *  and a scale. We have to set these into the system, using the C
    *  routine ConfCSetArgusData(), implemented in configure.c. We don't
    *  want to change the offset, so we get that first using the routine
    *  ConfCGetArgusData(), also in configure.c.
    */
    
   ConfCGetArgusData(&ArgusInUse,&ArgusAngle,Scale,&ArgusOffset);
   ArgusUsedString = PAF_GetValue ("INS.ARGS.USED");
   ArgusInUse = FALSE;
   ArgusAngle = 0.0;
   ArgusScale = "1:1";
   if (ArgusUsedString != (char*) NULL) {
      ArgusInUse = (*ArgusUsedString == 'T');
      if (ArgusInUse) {
         ArgusScale = PAF_GetString ("INS.ARGS.SCALE",Status);
         ArgusAngle = PAF_GetNumber ("INS.ARGS.ANGLE",Status);
      }
   }
   ConfCSetArgusData(ArgusInUse,ArgusAngle,ArgusScale,ArgusOffset);
   
   /*
    *  Get the +/- hour angle range that the config. is valid
    */
   LocalStatus = 0;
   ErsPush();
   *checkHA = PAF_GetNumber ("INS.OSS.CHECKHA",&LocalStatus);
   if (LocalStatus == 1) {
      ErsAnnul(&LocalStatus);
   }
   ErsPop();

   /*  Note. Most of the SDS code in this routine is lifted directly from
    *  the equivalent code in tdFconvertSds.c. This made it easier to
    *  be sure I didn't miss anything. It then had to be modified so it
    *  got the data from where this routine had put it, but the overall
    *  set of calls should be the same in both routines.
    */
        
   /*  Create the overall SDS structure */
   
   SdsNew(0,"OzPoz_field",0,NULL,SDS_STRUCT,0,NULL,FileId,Status);
   SdsNew(*FileId,"fieldData",0,NULL,SDS_STRUCT,0,NULL,&fId,Status);
   SdsNew(*FileId,"unallocGuide",0,NULL,SDS_STRUCT,0,NULL,&ugId,Status);
   SdsNew(*FileId,"unallocObject",0,NULL,SDS_STRUCT,0,NULL,&uoId,Status);
   
   /*  Store field details. */
    
   SdsNew(fId,"cenRa",0,NULL,SDS_DOUBLE,0,NULL,&tmpId,Status);
   SdsPut(tmpId,sizeof(double),0,&CenRa,Status);
   SdsFreeId(tmpId,Status);
   SdsNew(fId,"cenDec",0,NULL,SDS_DOUBLE,0,NULL,&tmpId,Status);
   SdsPut(tmpId,sizeof(double),0,&CenDec,Status);
   SdsFreeId(tmpId,Status);
   SdsNew(fId,"appRa",0,NULL,SDS_DOUBLE,0,NULL,&tmpId,Status);
   SdsPut(tmpId,sizeof(double),0,&AppRa,Status);
   SdsFreeId(tmpId,Status);
   SdsNew(fId,"appDec",0,NULL,SDS_DOUBLE,0,NULL,&tmpId,Status);
   SdsPut(tmpId,sizeof(double),0,&AppDec,Status);
   SdsFreeId(tmpId,Status);
   SdsNew(fId,"appEpoch",0,NULL,SDS_DOUBLE,0,NULL,&tmpId,Status);
   SdsPut(tmpId,sizeof(double),0,&AppEpoch,Status);
   SdsFreeId(tmpId,Status);
   SdsNew(fId,"configMjd",0,NULL,SDS_DOUBLE,0,NULL,&tmpId,Status);
   SdsPut(tmpId,sizeof(double),0,&Mjd,Status);
   SdsFreeId(tmpId,Status);
   SdsNew(fId,"unallocObj",0,NULL,SDS_SHORT,0,NULL,&tmpId,Status);
   SdsPut(tmpId,sizeof(short),0,&UnallocObjects,Status);
   SdsFreeId(tmpId,Status);
   SdsNew(fId,"unallocGui",0,NULL,SDS_SHORT,0,NULL,&tmpId,Status);
   SdsPut(tmpId,sizeof(short),0,&UnallocGuides,Status);
   SdsFreeId(tmpId,Status);
   SdsNew(fId,"unallocSky",0,NULL,SDS_SHORT,0,NULL,&tmpId,Status);
   SdsPut(tmpId,sizeof(short),0,&UnallocSky,Status);
   SdsFreeId(tmpId,Status);
   dims[0] = COMMENT_LENGTH;
   SdsNew(fId,"label",0,NULL,SDS_CHAR,1,dims,&tmpId,Status);
   SdsPut(tmpId,dims[0]*sizeof(char),0,Label,Status);
   SdsFreeId(tmpId,Status);
   dims[0] = ID_LENGTH;
   SdsNew(fId,"progId",0,NULL,SDS_CHAR,1,dims,&tmpId,Status);
   SdsPut(tmpId,dims[0]*sizeof(char),0," ",Status);
   SdsFreeId(tmpId,Status);
   dims[0] = MODE_LENGTH;
   SdsNew(fId,"mode",0,NULL,SDS_CHAR,1,dims,&tmpId,Status);
   SdsPut(tmpId,dims[0]*sizeof(char),0,"NORMAL",Status);
   SdsFreeId(tmpId,Status);
   SdsNew(fId,"allocSky",0,NULL,SDS_SHORT,0,NULL,&tmpId,Status);
   SdsPut(tmpId,sizeof(short),0,&AllocSky,Status);
   SdsFreeId(tmpId,Status);
   SdsNew(fId,"allocGui",0,NULL,SDS_SHORT,0,NULL,&tmpId,Status);
   SdsPut(tmpId,sizeof(short),0,&AllocGuides,Status);
   SdsFreeId(tmpId,Status);
   SdsNew(fId,"allocObj",0,NULL,SDS_SHORT,0,NULL,&tmpId,Status);
   SdsPut(tmpId,sizeof(short),0,&AllocObjects,Status);
   SdsFreeId(tmpId,Status);
   
   /*  Now we have to create the unallocated objects structure, the
    *  unallocated guide objects structure, and finally the allocated
    *  objects structure. All of these have basically the same format,
    *  with minor differences, so we use a set of utility routines to
    *  do this, with flag options to control the minor variations.
    *  PAF_CreateObjStruct() creates the structure, and puts the SDS Ids for
    *  the various items into IdStruct. PAF_GetObjectData() gets the data
    *  for an object described by a set of keywords in the PAF file. (For
    *  example, all the INS.XOBJn.xxx keywords have the details for the
    *  nth unallocated object, the INS.XREFn.xxx keywords have the details
    *  for the unallocated guide targets.) It puts the data into ObjectData.
    *  PAF_SetObjectData() then sets the data in ObjectData into the various
    *  SDS arrays whose Ids are in IdStruct at one specified index value.
    *  Finally, PAF_FreeIds() releases the SDS Ids in IdStruct.
    */
   
   /*  Create unallocated objects structure. */
   
   PAF_CreateObjStruct (uoId,UnallocObjects + UnallocSky,0,&IdStruct,Status);
   
   /*  Now pass through the PAF file, picking up all the unallocated objects
    *  and setting their data into the unallocated objects structure. All of
    *  these are in the INS.XOBJn section of the PAF file, which includes
    *  both unallocated sky and target objects - but not unallocated
    *  guide objects.
    */
    
   for (Index = 0; Index < UnallocObjects + UnallocSky; Index++) {         
      PAF_GetObjectData (Instrument,"INS.XOBJ",Index,0,&ObjectData,Status);
      PAF_SetObjectData (&IdStruct,Index,&ObjectData,Status);  
   }   
   PAF_FreeIds (&IdStruct,Status);
   
   /*  Create unallocated guide structure. */
    
   PAF_CreateObjStruct (ugId,UnallocGuides,DHDX_FLAG,&IdStruct,Status);
   
   /*  Now pass through the PAF file, picking up all the unallocated guide
    *  objects and setting their data into the unallocated guide objects
    *  structure.  Note that most of these objects are described in the
    *  PAF file under the unallocated guide targets (in INS.XREF keywords)
    *  but that the VLT guide probe targets (in TEL.PGS) are also classed as 
    *  unallocated guide objects, so they need to be included as well. We
    *  pull in the VLT guide probe targets first, then the ordinary guide
    *  targets.
    */
    
   for (Index = 0; Index < VLTGuiderTargets; Index++) {  
      PAF_GetObjectData (Instrument,"TEL.PGS",Index,0,&ObjectData,Status);
      PAF_SetObjectData (&IdStruct,Index,&ObjectData,Status);
      
      /*  As we go through the VLT guide probe targets, see which of them
       *  is the selected guide probe target, if one was indeed selected.
       */
       
      if (GuideStarSel) {
         if ((ObjectData.Ra == GuideRA) && (ObjectData.Dec == GuideDec)) {
            GuideInfoPtr->SelGuideIndex = Index + 1;
         }
      }
   }
   
   for (Index = 0; Index < UnallocFACBTargets; Index++) {     
      PAF_GetObjectData (Instrument,"INS.XREF",Index,0,&ObjectData,Status);
      PAF_SetObjectData (&IdStruct,Index + VLTGuiderTargets,&ObjectData,Status);  
   } 
   PAF_FreeIds (&IdStruct,Status);
   
   /*  We use the DoMakeAllocated routine in configure.c to create the
    *  objects structure and initialise it with the proper pivot positions
    */
    
   DoMakeAllocated (*FileId,PivId,Status);
   
   SdsFind(*FileId,"objects",&ObjId,Status);
   PAF_LocateObjIds (ObjId,DHDX_FLAG | THETA_FLAG,&IdStruct,Status);
   
   /*  Now, go through each allocated object - the allocated guide targets
    *  are described by the INS.REFi keywords, the program allocated targets
    *  are described by the INS.GFIBi, INS.UFIBi and INS.AFIBi keywords.
    *  (We don't assume these names, however, we get them through Fpil.)
    */
    
   for (Index = 0; Index < AllocGuides; Index++) {     
      PAF_GetObjectData (Instrument,"INS.REF",Index,FIBRE_FLAG,&ObjectData,Status);
      PAF_SetObjectData (&IdStruct,ObjectData.Fibre - 1,&ObjectData,Status);
   }  
   
   for (IType = 0; IType < NumTypes; IType++) {    
      InsCode = *(AbbrNames[IType]);
      Index = 1;
      sprintf (Keyword,"INS.%cFIB",InsCode);
      for (Index = 0; Index < TypeCount[IType]; Index++) {     
         PAF_GetObjectData (Instrument,Keyword,Index,FIBRE_FLAG,
                                                           &ObjectData,Status);
         PAF_SetObjectData (&IdStruct,ObjectData.Fibre - 1,&ObjectData,Status);
      }
   }  
   PAF_FreeIds (&IdStruct,Status);
            
   SdsFreeId (fId,Status);
   SdsFreeId (ugId,Status);
   SdsFreeId (uoId,Status);
   SdsFreeId (ObjId,Status);
   
Exit:;        
      
   PAF_ReadClose();
   
}

/*
         1         2         3         4         5         6         7         8
12345678901234567890123456789012345678901234567890123456789012345678901234567890
*/
