/*+
 *                         C O N F I G _ M E T H O D
 *
 *  Function:
 *     'Algorithm-specific' routines for 2dF fibre allocations.
 *
 *  Description:
 *     This file contains a set of routines that between them provide the
 *     default fibre instrument configuration algorithm. These are intended 
 *     to be called from the 'housekeeping and interface' (config.c) 
 *     layer of routines described in the document 'Fibre Instrument 
 *     Configuration Algorithm Routines'. 
 *
 *     This module implements the default allocation algorithm described
 *     in that document and the section "The default algorithm" in that
 *     document should be read as in introduction to this module. This
 *     original algorithm (which has a number of variations) is sometimes
 *     described as the 'Taylor' algorithm, being based on an idea by Keith
 *     Taylor. This module also implements the 'Oxford' algorithm as devised 
 *     by Gavin Dalton, sometimes described as 'iterative target selection',
 *     or 'cone allocation'.
 *
 *  Version date: 2nd October 2002.
 *
 *  Support: K. Shortridge, Tony Farrell, AAO.  Gavin Dalton, Oxford.
 *-
 *  Sccs Id:     config_method.c, Release 1.20, 10/02/02
 *
 *  History:
 *     23rd Sep 1994.  Original version. KS/AAO.
 *     17th Oct 1994.  Now no longer uses explicit dimensions or clearances.
 *                     Gets these from the housekeeping layer now. KS/AAO.
 *      9th Jul 1996.  Fixed priority code so low numbers are higher
 *                     priority. KS/AAO.
 *     30th Sep 1996.  Reverted to original priority conventions. KS/AAO.
 *     30th Oct 1996.  Working Version of Oxford Code installed GBD/OXF.
 *     14th Nov 1996.  Added a new pathstate 'OUTSIDE' to prevent errors
 *                     cropping up if the target list goes beyond the 
 *                     nominal field radius... GBD
 *     19th Jun 1998.  Added a new pathstate 'SCREWED' to check for the
 *                     two fixing screws that hold the plate in place. GBD
 *     22nd Jan 1999.  Merge in Gavin Dalton's changes to main source. 
 *                     Previous version was SCCS id 1.9.  TJF/AAO.
 *     28th Jan 1999.  Add Sccs Id line.  TJF/AAO.
 *     12th Feb 1999.  Rename Config.h to tdFconfig.h to avoid conflicts
 *                     with DRAMA's config.h in case insentivie enviroments. 
 *                     TJF/AAO
 *     07th Mar 2000.  Major changes to support new style where the
 *                     algorithm routines are called via function pointer. 
 *                     All routines and variables now static except for
 *                     the new routine ConfigMethodDetails().  Init()
 *                     now checks for a priority of CONFIG_PRIO_INVALID
 *                     as set by ConfigSetTargets() instead of setting
 *                     priority to 100.   TJF/AAO.
 *                     NoteDeallocation() now handles objects which
 *                     were preallocated, removing the need for
 *                     config_method.c to play with TargetInfo. Remove
 *                     NUnGuide and NUnTarget from CONFIG_METHOD_Global
 *                     and associated code.  The addition housekeeping
 *                     routine NoteDeallocatoinBroken allows us to
 *                     avoid touching AllocInfo.State.  
 *                     Remove use of most DRAMA includes files, only
 *                     need SDS and ERS.   Use housekeeping info report
 *                     routine instead of MsgOut().
 *                     Remove use of 2dF specific collision detection
 *                     routines and instead use new housekeeping layer
 *                     routines. 
 *                     SHOULD NOW BE INDEPENDENT OF 2DF. TJF/AAO.
 *     08th Mar 2000.  Remove calls to ErsRep and ErsOut, use new 
 *                     ErrRepRoutine in MethodInfo structure.
 *                     Include tdFconfigMethod.h instead of tdFconfig.h
 *                     so that we are not dependent on as many things.
 *                     Now only need DRAMA's status.h include file and
 *                     our own definition of DUNUSED.   TJF/AAO.
 *     09th Mar 2000.  Remove usage of explict 404, using malloc to
 *                     allocate the ICross variable in 
 *                     ConfigMethodUncrossFibres and Zapped in
 *                     ConfigMethodAutoDeallocate(). TJF/AAO.
 *     10th Mar 2000.  Improve some of the parameter releated documentation
 *                     based on information from the 2dF manual.  Still don't
 *                     know must about the "expert" options (see configure
 *                     program).  Add comment in module description
 *                     about description of algorithm in tdfconfig.doc. 
 *                     TJF/AAO.
 *     20th Mar 2000.  Add CONFIG_METHOD_IGNORE, used for ignoring sky
 *                     values and items with PivotInfo.AllowAllocChange
 *                     set false.  TJF/AAO.
 *     31th Mar 2000.  Allocations can not be tentative, make use of that
 *                     in ConfigMethodSetAlloc in Oxford algorithm case
 *                     to avoid lots of calls to the user interface on
 *                     tentative allocation changes.  TJF/AAO.
 *     06th Apr 2000.  Use MethodInfo->Collision()  in ConfigMethodNewAlloc()
 *                     amd ConfigMethodCrashTest().  Macro CHECK_FPIL can
 *                     be enabled to check old way versions new approach.
 *                     When we are confident, can remove all code marked
 *                     by CHECK_FPIL.
 *                     Check for collisions against parked fibres as
 *                     in 6dF this is possible. (Later, may make this 
 *                     optional, needs support from FPIL.) TJF/AAO.
 *     07th Apr 2000.  Ensure MaxMoves global item and max values of
 *                     FREEKSY and RECOVER parameters initialised to
 *                     values based on the maximum number of pivots.  Some
 *                     more limitations noted below. TJF/AAO.
 *     11th Apr 2000.  ConfigMethodInit() returns a status of 
 *                     CONFIG__INVAL_PARM_VALUE instead of CANCELLED 
 *                     for invalid parameter combinations. TJF/AAO.
 *     18th Apr 2000.  Fixed apparent bug in ConfigMethodDefineCones()
 *                     which causes it to fail to allocate anything if
 *                     the first object in a priority range was a sky. TJF/AAO.
 *     19th Apr 2000.  Make use of new MethodInfo->ParkMayCollide() routine
 *                     when checking for collisions.
 *      9th Aug 2000.  The 'Spect' field in a CONFIG_PivotInfo structure is
 *                     now the more general 'FibreType'. Made changes to 
 *                     support this change and to use Fpil routines to handle
 *                     the values it contains. KS/AAO.
 *     14th Aug 2000.  Reformatted code to fix tab-induced and other layout
 *                     problems and to constrain to 80-character lines. KS/AAO.
 *     05th Sep 2000.  CrashTest was much slower.  Change back to previous
 *                     approach instead of using MethodInfo->Collision.
 *                     Uncross was much slower due the now work required by
 *                     NoteDeallocation.  Instead, make use of a new
 *                     routine ConfigMethodTryUncross which uses a new
 *                     MethodInfo->SetFibrePos() function.  This allows
 *                     use to try CrashTest on a uncross without actually
 *                     deallocating and allocating. TJF/AAO.
 *     05th Sep 2000.  Removed or commented out all DEBUG statements. KS.
 *     20th Sep 2000.  Only use #warning if system supports it. KS.
 *     22nd Nov 2000.  Replaced use of PivotTheta by Alpha - this corrects a
 *                     long standing bug in the 'straightest' target allocation
 *                     mode. Added code to calculate a 'crowded' metric for
 *                     each target, in order to allow a 'most crowded' 
 *                     allocation later if necessary. Commented some of the
 *                     code in the cone definition routine. KS.
 *     28th Nov 2000.  Modified parameter names to include "(expert)" to 
 *                     indicate they should only be made available via the
 *                     user interface in 'expert' mode. Introduced the
 *                     USE_OXFORD_ALGORITHM pre-processor variable, and - for
 *                     the moment - set this false if the system is being
 *                     built for FLAMES. Removed the code relating to the
 *                     unimplemented CONFIG_METHOD_SEL_RECOVER selection
 *                     method. Replaced the 'closest' selection option by a
 *                     new 'most crowded' option. KS.
 *      4th Dec 2000.  ConfDoSwapAllocate() now reports every second, making
 *                     it possible to cancel an allocation that has too many
 *                     targets to complete in a reasonable time. Removed the
 *                     magic numbers of 600,-399 and -400 from the code i
 *                     ConfigMethodDefineCones(). Introduced the code that
 *                     deallocates fibres after an allocation in order to get
 *                     an unbiassed set of spare fibres for sky. KS.
 *      2nd Feb 2001.  Introduced the CONFIG_METHOD_PARK_CLASH code for the
 *                     allocation matrix and implemented the appropriate tests
 *                     at initialisation. KS.
 *      3rd Feb 2001.  Introduced use of rand() instead of random() on systems
 *                     that only support the former. KS.
 *     21st Feb 2001.  Now uses FibreLength field of the PivotInfo structure
 *                     where the maximum extension of a specific fibre is 
 *                     needed. KS.
 *     18th Aug 2001.  Minor changes to get a clean compilation under gcc
 *                     with the -ansi and Wall flags set. KS.
 *     22nd Aug 2001.  Set the sky fibres parameter default value down to zero
 *                     for in the FLAMES-only case. KS.
 *     24th Oct 2001.  Was always leaving one fibre free for sky even when the
 *                     'leave for sky' parameter was set to zero. Fixed. KS.
 *     13th Nov 2001.  Added ConfigMethodDisableSky(), ConfigMethodReEnable(),
 *                     ConfigMethodDisableNonSky(), ConfigMethodMaxAlloc().
 *                     Tentative deallocation stage of allocation now only
 *                     happens if the maximum possible number of pivots have
 *                     not been allocated. (Used to test against total number
 *                     of pivots, not allowing for ones that could not possibly
 *                     be allocated.) KS.
 *     14th Nov 2001.  Added ConfigMethodReAssignToSky(), and removed the unused
 *                     ConfigMethodDisableNonSky(). KS.
 *     20th Nov 2001.  Modified ConfigMethodUncrossFibres() so it checks that
 *                     the allocation matrix allows a given fibre swop. This
 *                     stops it failing when sky targets are disabled. KS.
 *     28th Nov 2001.  Removed DEBUG output introduced during previous
 *                     change. KS.
 *      6th Dec 2001.  Set default value of fibres to leave for sky to zero
 *                     for all systems - this is now obsolete, replaced in
 *                     practice by the explicit sky assignment on a per fibre
 *                     type basis. KS.
 *     17th Dec 2001.  Fixed bug that was preventing 'expert' options in a
 *                     parameter with string choices from being accepted.
 *                     'Only allocate sky' is no longer an expert option, but
 *                     if invoked uses the Taylor algorithm rather than the
 *                     Oxford algorithm. KS.
 *     19th Dec 2001.  Reassigning of fibres to sky targets now works properly
 *                     if the Oxford algorithm is used for the main allocation.
 *                     ConfigMethodUpdateMatrix() added for this purpose. KS.
 *      3rd Jan 2002.  The FieldCheck function provided by config.c now takes
 *                     an additional 'PositionerTolerances' argument. KS.
 *      5th Aug 2002.  Corrected use of Pivot index 0's path status to check if
 *                     a target is outside the field. The target info Priority
 *                     field is now used instead. Removed the use of the
 *                     FLAMES_ONLY and USE_OXFORD_ALGORITHM variables, replacing
 *                     them with a test against number of pivots to identify
 *                     FLAMES. This is unsatisfactory, but has the advantage
 *                     of working in both the FLAMES-only and all-systems
 *                     builds. KS.
 *     23rd Aug 2002.  Corrected apparent bug in setting of TargetPivot during
 *                     allocations - was being set even if an attempted
 *                     allocation failed, meaning that a target appeared to be
 *                     allocated, and so would be missed out of the swapping
 *                     phase. Result may be a better swap phase, but one that
 *                     takes longer. Modified progress report during swap
 *                     phase so that progress bar doesn't blank out. Swap
 *                     phase now only occurs at end of allocation, not 
 *                     after each priority level, in the interests of speed.
 *     27th Aug 2002.  Added use of ConfigMethodSortPivots() to encourage the
 *                     iterative algorithm to allocate with minimal fibre 
 *                     bending. Comprehensive reworking of code for
 *                     ConfigMethodConeAllocate() to try to clarify its
 *                     operation. KS.
 *      2nd Sep 2002.  Completed commenting of the swap phase of the Oxford
 *                     algorithm, particularly ConfigMethodBlockSwapAllocate()
 *                     and ConfigMethodDoSwap(). Introduced an immediate return
 *                     from ConfigMethodDoSwap() when maximum number of moves
 *                     exceeded
 *      5th Sep 2002.  Don't allow allocation of a fibre behind its park 
 *                     position - picks up problems with OzPoz FACB's which are
 *                     currently parked on the plate.  TJF fix originally 
 *                     applied at Paranal 21stAug 2002, now incorporated into
 *                     main code by KS.
 *     11th Sep 2002.  Added ConfigMethodAllocCount() for diagnostic purposes.
 *                     When sky targets are disabled, actual current allocations
 *                     are now no longer flagged as not possible. This corrects
 *                     some odd effects in the allocation count if an allocation
 *                     is repeated. KS.
 *     14th Sep 2002.  Introduced MeritValue to try to prevent gains being made
 *                     at the expense of high priority targets. KS.
 *     17th Sep 2002.  Tentative deallocation now works even if the 'Oxford'
 *                     algorithm is being used. KS.
 *     18th Sep 2002.  Uncrossing now seems to work even if the 'Oxford' 
 *                     algorithm is not being used. This combination is no
 *                     longer rejected. 'Oxford' algorithm now can allocate
 *                     in the case where only guide fibres are selected. KS.
 *     26th Sep 2002.  Corrected problem when fibres were supposed to be left
 *                     unallocated for 6dF and no sky targets were specified. 
 *                     KS.
 *      2nd Oct 2002.  Fixed malloc leaks due to the allocation of Pivot and
 *                     ConeTargets arrays. KS.
 *     21st Dec 2002.  fibreType arg. added to *ColInvPos functions.
 * 
 * 
 * 
 * jpritcha 2016-10-05 Bug fix at lie 5885
 *
 *  Bugs and limitations:
 *     o  The Oxford algorithm was rather bolted onto this code and in places
 *        the joins are very evident. This is slowly being sorted out.
 *     o  There is no code that attempts to use twist angles to increase the
 *        number of allocated fibres. This is not an option for OzPoz and 6dF,
 *        but could be used for 2dF, in principle.
 *     o  The Oxford algorithm is very slow when used in cases where there
 *        are many more targets than fibres. The swap phase runs agonisingly
 *        slowly in this case. It might be that it should be disabled in
 *        such cases.
 *     o  FREEKSKY parameter max value and MaxMoves global item parameter need
 *        the number of guide fibres subtracted from their values.
 *     o  ConfigMethodUncrossFibres() does not work for instruments on which
 *        parked fibres may be crossed by others (e.g. 6dF) and results
 *        in collisions.  As a result, the UNCROSS parameter value must be
 *        set to NEVER instead of ATEND for 6dF.
 *+
 */

/* GBD's comments: 

 * Changed all tests of AllocInfo[IPiv].State == 0 for broken fibres to 
 * PivotInfo[IPiv].Spect == 0 to allow for auto-deallocation of broken
 * fibres.. */


/*  Include files used by CONFIG_METHOD  */

#include <math.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>

/*  DRAMA include files, only status.h needed */

#include "status.h"

/*  Debugging macro */
#ifdef DEBUG
#undef DEBUG
#endif
#define DEBUG (void) printf


/*#define CHECK_FPIL*/
/*  The following messages are handy until the FPIL stuff is deemed to be
 *  OK, but the use of #warning is non-standard. To date, the only system
 *  not to support it is HP-UX (of course!), so we omit the messages on that
 *  system.
 */

#ifndef _HPUX_SOURCE
#ifdef CHECK_FPIL
#warning "Checking Fpil wrap up routines against original sequence"
#else
/* #warning "Not checking FPIL wrap up routines" */
#endif
#endif

/*  Random() isn't supported under Metrowerks CodeWarrior, so we use rand()
 *  instead - note that the code that uses random() clips off all bar the 
 *  lowest 16 bits, so we can get away with this. Some systems (Linux)
 *  don't seem to declare it properly in stdlib.h, so we provide a
 *  declaration for it.
 */
 
#ifdef macintosh
#define random rand
#else
long int random (void);
#endif

/*
 * DUNUSED Flag which under GCC, prevents warnings about an argument being
 * unused - should be added where we know it will be unused
 */
#ifndef DUNUSED
#if defined(__GNUC__) || defined(GNUC)
#   define DUNUSED /* Default definition */
#   ifdef __GNUC_MINOR__
#   if __GNUC_MINOR__ >= 7
#       undef  DUNUSED
#       define DUNUSED __attribute__ ((unused))
#   endif
#   endif
#else
#   define DUNUSED /* */
#endif
#endif /* DUNUSED */


/*  Include files specific to CONFIG itself */

#include "tdFconfigMethod.h"
#include "config_err.h"

/* -------------------------------------------------------------------------- */

/*            M e t h o d - s p e c i f ic  d e c l a r a t i o n s
 *
 *  'Method' in this case refers to anything specific to the actual allocation
 *  used.  Because of the uncertainty about the best such algorithm to be used, 
 *  the code here is arranged so that all the method-specific code can be 
 *  removed to a separate file if necessary so that alternative algorithms
 *   ('methods') can be substituted.
 */

typedef struct CONFIG_METHOD_MatrixElement {
   unsigned char PathStatus;     /* Flags state of the possible path */
   int CollidingPivot;           /* A fibre that collides with the path */
   double Theta;                 /* The theta value for the button */
   double Alpha;                 /* The alpha value for the pivot */
} CONFIG_METHOD_MatrixElement;

/*  Want to define a target based matrix as well as a pivot based matrix...
 *  this is a little heavier on memory, but will come in useful later on.
 */

typedef struct CONFIG_METHOD_TargetElement {
   int TargetPivot;           /* The pivot currently used */
   double TargetT;            /* A direction angle for the target */
   double TargetR;            /* A radial distance for the target */
   int NumPivots;             /* The number of fibres that could be used */
   int NConeTargets;          /* The number of targets inside our cone */
   int *Pivot;                /* Array containing the possible pivot numbers */
   int *ConeTargets;          /* The list of targets inside our cone */
} CONFIG_METHOD_TargetElement;

/*  A little more detail here about some of the fields of this structure
 *  may make it easier to understand the Oxford iterative allocation
 *  algorithm. CONFIG_METHOD_Global.TargetMatrix contains the address of
 *  an array of these structures, one for each target.
 *
 *  TargetPivot  is the index number of the pivot (if any) currently
 *               allocated to the target in question. A -ve number (usually
 *               -1) indicates that the target is not allocated to a pivot.
 *               This is the index number into the MethodInfo->PivotInfo
 *               array.
 *  TargetT      is the angle between the X-axis and a line drawn from the
 *               center of the field to the target.
 *  TargetR      is the distance from the center of the field to the target.
 *  NumPivots    is the number of pivots that can be allocated to the target,
 *               calculated simply on the basis of fibre/target compatability
 *               and physical constraints such as maximum bend angle and
 *               fibre length.
 *  NConeTargets is the number of targets that fall within the cone defined
 *               by the target and the two extreme fibres that can be allocated
 *               to it.
 *  Pivots       contains the index numbers of the pivots that can be allocated
 *               to the target. These are index numbers into the array
 *               MethodInfo->PivotInfo.
 *  ConeTargets  contains the index numbers of all the targets that are within
 *               the cone with the target at the apex. These are index numbers
 *               into the array MethodInfo->TargetInfo.
 */

/*  These are the codes used for the PathStatus flag in the matrix. These
 *  indicate either that a given path is possible, or the reasons why it
 *  isn't. Originally, all elements are set either to POSSIBLE, WON'T_REACH
 *  BAD_ALPHA, BROKEN, PARK_CLASH or INCOMPATIBLE. Nothing can alter these
 *  conditions, which are physical. As the program progresses, fibres get 
 *  allocated and some of the POSSIBLEs change into IN_USE or ALLOCATED or 
 *  COLLISION, and may change back again if fibres are deallocated.
 *  OUTSIDE means the targets too far out and would be badly vignetted...
 *  SCREWED means it would fall on one of the plate screws, which 
 *  would be very bad news!  IGNORE means no attempt should be made to
 *  allocate the pivot. DISABLED means that this is an allocation that is
 *  possible, but is not being allowed at the moment (for example, to
 *  prevent any sky target allocations).
 */

#define CONFIG_METHOD_POSSIBLE 0
#define CONFIG_METHOD_ALLOCATED 1
#define CONFIG_METHOD_WONT_REACH 2
#define CONFIG_METHOD_BROKEN 3
#define CONFIG_METHOD_FIBRE_IN_USE 4
#define CONFIG_METHOD_INCOMPATIBLE 5
#define CONFIG_METHOD_BAD_ALPHA 6
#define CONFIG_METHOD_COLLISION 7
#define CONFIG_METHOD_TARGET_IN_USE 8
#define CONFIG_METHOD_OUTSIDE 9
#define CONFIG_METHOD_SCREWED 10
#define CONFIG_METHOD_IGNORE  11
#define CONFIG_METHOD_PARK_CLASH 12
#define CONFIG_METHOD_DISABLED 13

/*  Method parameter constants. Note that having these statically defined
 *  like this makes it very easy to implement ConfigMethodParamDetails(),
 *  and also helps to code ConfigMethodSetParam(). However, it is important
 *  that the information held here is consistent - each parameter has a
 *  symbolic name that equates to an integer index, and the entries in
 *  the ConfigMethodParamXxxx arrays must match those indices. It isn't quite
 *  possible to make the whole parameter business 'table-driven' (controlled
 *  entirely by the entries in here). Some explicit code is needed, but the 
 *  more we can do from the definitions here the easier the code is to modify.
 *  At present, to add a new parameter, 1) Increase the value of CONFIG_METHOD-
 *  _NPARMS, 2) Define a new symbolic value for its index (a new parameter
 *  called CONFIG_METHOD_xxxx_PARM, 3) Add new entries to all the CONFIG_METHOD
 *  _ParmXxxx tables. 4) For a character parameter with a fixed set of possible
 *  values, define a new array of the possible values (see CONFIG_METHOD_-
 *  SelectChoices) and symbolic indices for the values, 5) Add initialisation
 *  code to ConfigMethodGlobalInit(), 6) Add whatever code is needed to make
 *  use of the new parameter.
 */

#define CONFIG_METHOD_NPARMS 6

/*  A array of structures like this goes into the method-specific global
 *  variables. For each parameter, there is a double precision value (used
 *  only for floating point parameters) and an integer value. For logical 
 *  parameters the integer value will be 0 (false) or 1 (true). For integer
 *  parameters the integer value is that of the parameter. For character
 *  parameters with an unlimited number of possible values, when the parameter
 *  is set to a new value an array is allocated in memory (using malloc())
 *  for a static copy of that new value string, and the address of this
 *  allocated string is held in common. For character parameters with a
 *  limited set of possible values, all possible values are held in a static
 *  array and the address of the value currently set is held in common. In
 *  this case, the integer value is also used to hold the index number of
 *  the current value being used.
 */

typedef struct CONFIG_METHOD_ParamValue {
   int IntegerValue;                         /* Integer value - see above */
   char *StringAddress;                      /* Address of string */
   double DoubleValue;                       /* Floating point value */
} CONFIG_METHOD_ParamValue;


/*  First parameter - index = CONFIG_METHOD_SELECT_PARM = 0.    
 *
 *  The first parameter refers to the means whereby a target is selected
 *  for a given pivot. The choices all have symbolic names
 *  CONFIG_METHOD_SEL_Xxxxx and these must match the order of character
 *  strings in CONFIG_METHOD_SelectChoices.
 *
 *  It's worth explicitly documenting what's involved in implementing a
 *  new option for this parameter.
 *  1) Add a new CONFIG_METHOD_SEL_xxx definition to the following section.
 *  2) Add a new entry into the array CONFIG_METHOD_SelectChoices[] defined
 *     below. The index into this array for the new entry has to match the
 *     value for the new CONFIG_METHOD_SEL_xxx definition. If this is to be
 *     an expert-only option, make the description end with "(expert)".
 *  3) Change the value given in CONFIG_METHOD_ParmNValues for the number of
 *     possible values for CONFIG_METHOD_SELECT_PARM.
 *  4) If you want this new option to be the default, change the code in
 *     ConfigMethodGlobalInit() that sets the initial value of
 *     CONFIG_METHOD_Global.ParameterValues[CONFIG_METHOD_SELECT_PARM],
 *     noting that you have to set both .IntegerValue and .StringAddress.
 *  5) Add whatever code is needed to make it work.
 *
 *  Note that at the moment, unfortunately, the configure.tcl interface code
 *  has a number of places where assumptions are made about the order of
 *  the possible choices for the selection method (so that the tcl code can 
 *  explicitly invoke a specific method such as auto-reallocation). This means
 *  that changing the order of the choices is unsafe. If you really want to do
 *  it, look at the places in the code where "ParameterDetails" is used, for
 *  example in the Reallocate{} procedure, and make sure you understand
 *  exactly what's going on. This should be changed - it breaks all the
 *  carefully constructed encapsulation of the allocation code - and it will
 *  be, but it's going to need a bit of work. 
 */

#define CONFIG_METHOD_SELECT_PARM 0

#define CONFIG_METHOD_SEL_CROWD 0    /* Allocate 'hardest' pivot to the most
                                      * crowded target. Repeat until done. */
#define CONFIG_METHOD_SEL_FAR 1      /* Allocate 'hardest' pivot to the most
                                      * distant target. Repeat until done. */
#define CONFIG_METHOD_SEL_HARD 2     /* Allocate 'hardest' pivot to the
                                      * 'hardest' target. Repeat until done. */
#define CONFIG_METHOD_SEL_STRAIGHT 3 /* Allocate 'hardest' pivot to the most
                                      * straight target. Repeat until done. */
#define CONFIG_METHOD_SEL_GBD 4      /* Iterative target allocation -
                                        recursive swapping of targets
                                        to get maximum number of targets */
#define CONFIG_METHOD_SEL_BROKEN 5   /* Deallocate fibres in the current
                                        configuration which are invalid:  
                                        This could happen because they were 
                                        disabled in the instrument constants
                                        file or are too close according to the
                                        current verson of the constants file.
                                        Once thse fibres have been deallocated,
                                        an attempt is made to reallocate
                                        the target which were lost. */
#define CONFIG_METHOD_SEL_SKYONLY 6  /* Only allocate sky positions */
#define CONFIG_METHOD_SEL_IMPORT 7
#define CONFIG_METHOD_SEL_IMPORTONLY 8
                                        
static char *CONFIG_METHOD_SelectChoices[] = {
    "by most crowded target",                /* CONFIG_METHOD_SEL_CROWD      */
    "by furthest target (expert)",           /* CONFIG_METHOD_SEL_FAR        */
    "by hardest target",                     /* CONFIG_METHOD_SEL_HARD       */
    "by straightest target",                 /* CONFIG_METHOD_SEL_STRAIGHT   */
    "iterative target allocation",           /* CONFIG_METHOD_SEL_GBD        */
    "auto-reallocate (expert)",              /* CONFIG_METHOD_SEL_BROKEN     */
    "only allocate sky",                     /* CONFIG_METHOD_SEL_SKYONLY    */
    "import ascii allocation table (expert)",/* CONFIG_METHOD_SEL_IMPORT     */
    "import with no checks (expert)"};       /* CONFIG_METHOD_SEL_IMPORTONLY */

/*  First parameter - index = CONFIG_METHOD_DEALLOC_PARM = 1.    
 *
 *  The second parameter refers to the use of a 'tentative deallocation'
 *  pass to try to improve the allocation coverage. This is a character
 *  parameter - the deallocation pass is never attempted, or it depends
 *  on whether or not there are fibres unallocated at the end of the
 *  initial pass. Sometimes the reallocation process can produce a 
 *  preferable result.
 */

#define CONFIG_METHOD_DEALLOC_PARM 1

#define CONFIG_METHOD_DEALLOC_NEVER 0
#define CONFIG_METHOD_DEALLOC_ONLY 1
#define CONFIG_METHOD_DEALLOC_ALWAYS 2

static char *CONFIG_METHOD_DeallocChoices[] = {
    "never",                             /* CONFIG_METHOD_DEALLOC_NEVER  */
    "if fibres are unallocated",         /* CONFIG_METHOD_DEALLOC_ONLY   */
    "even if all fibres are allocated (expert)"
    };                                   /* CONFIG_METHOD_DEALLOC_ALWAYS */


/*  Third parameter - index = CONFIG_METHOD_UNCROSS_PARM = 2.    
 *
 *  The third parameter controls aspects of the fibre uncrossing scheme.
 *  The default is to uncross at the end of the allocation, which
 *  will suffice for most fields.  If there are many priority levels of 
 *  object one can sometimes gain in allocations by selecting uncrossing
 *  after each priority pass, though this is more time consuming. It is also 
 *  possible to uncross exisiting configurations.
 */

#define CONFIG_METHOD_UNCROSS_PARM 2

#define CONFIG_METHOD_UNCROSS_NEVER 0
#define CONFIG_METHOD_UNCROSS_EACH 1
#define CONFIG_METHOD_UNCROSS_ATEND 2
#define CONFIG_METHOD_UNCROSS_ONLY 3
#define CONFIG_METHOD_UNCROSS_THEN_ALLOCATE 4

static char *CONFIG_METHOD_UncrossChoices[] = {
    "never",                           /* CONFIG_METHOD_UNCROSS_NEVER */
    "after each priority pass (expert)", 
                                       /* CONFIG_METHOD_UNCROSS_EACH  */
    "when allocation completed",       /* CONFIG_METHOD_UNCROSS_ATEND */
    "uncross only",                    /* CONFIG_METHOD_UNCROSS_ONLY  */
    "uncross first then allocate extras (expert)"}; 
                                       /* CONFIG_METHOD_UNCROSS_THEN_ALLOCATE */

/*  Fourth parameter - index = CONFIG_METHOD_FREESKY_PARM = 3.    
 *
 *  The fourth parameter indicates the number of fibres left
 *  parked using normal allocation so they can be used for Sky fibres 
 *  later. For 2dF (and 6dF?) 16 seems a reasonable value. For FLAMES,
 *  where there are fewer fibres (very few, in some configurations) we 
 *  set it to zero and let the user set it up if they want. This parameter
 *  is now obsoleted by the explicit per-fibre type control over sky 
 *  allocations, so we set it to zero for all systems.
 */
 
#define CONFIG_METHOD_FREESKY_PARM 3
static char *CONFIG_METHOD_FreeSkyChoices[]={"  "};

#define FREESKY_PARAM_DEFAULT 0

/*  Fifth parameter - index = CONFIG_METHOD_RECOVER_PARM = 4.    
 *
 *  The fifth parameter - fibre to recover, mode CONFIG_METHOD_SEL_RECOVER
 *    NOT IMPLEMENTED
 */
#define CONFIG_METHOD_RECOVER_PARM 4
static char *CONFIG_METHOD_FibreToRecover[]={"  "};

/*  Sixth parameter - index = CONFIG_METHOD_BONUS_PARM = 5.    
 *
 *  The Sixth parameter - priorities below this threshold are a bonus.  When
 *  Don't look at these until autoreallocate when we allocate any left
 *  over fibres to these targets.  The idea seems to be that targets below
 *  this priority level are regarded as a bonus if we can allocate them,
 *  but we don't go out of our way to get them - right at the end in 
 *  auto-reallocate they may be allocated as a bonus. (Comment by KS).
 */
#define CONFIG_METHOD_BONUS_PARM 5
static char *CONFIG_METHOD_BonusPrio[]={"  "};

/*  
 *  The following arrays contain the parameter details as used by
 *  ConfigMethodParamDetails() and ConfigMethodSetParam().
 */

static char *CONFIG_METHOD_ParmNames[CONFIG_METHOD_NPARMS] = {
    "Target selection",                       /* CONFIG_METHOD_SELECT_PARM   */
    "Tentative deallocation",                 /* CONFIG_METHOD_DEALLOC_PARM  */
    "Fibre Uncrossing",                       /* CONFIG_METHOD_UNCROSS_PARM  */
    "Number of fibres to leave for sky (expert)",
                                              /* CONFIG_METHOD_FREESKY_PARM  */
    "Fibre to recover (expert)",              /* CONFIG_METHOD_RECOVER_PARM  */
    "Priority for Bonus Targets (expert)"};   /* CONFIG_METHOD_BONUS_PARM    */

static char CONFIG_METHOD_ParmTypes[CONFIG_METHOD_NPARMS] = {
    'C',                                      /* CONFIG_METHOD_SELECT_PARM   */
    'C',                                      /* CONFIG_METHOD_DEALLOC_PARM  */
    'C',                                      /* CONFIG_METHOD_UNCROSS_PARM  */
    'I',                                      /* CONFIG_METHOD_FREESKY_PARM  */
    'I',                                      /* CONFIG_METHOD_RECOVER_PARM  */
    'I'};                                     /* CONFIG_METHOD_BONUS_PARM    */

static double CONFIG_METHOD_ParmLimits[CONFIG_METHOD_NPARMS][2] = {
    { 0.0, 0.0},                              /* CONFIG_METHOD_SELECT_PARM   */
    { 0.0, 0.0},                              /* CONFIG_METHOD_DEALLOC_PARM  */
    { 0.0, 0.0},                              /* CONFIG_METHOD_UNCROSS_PARM  */
    { 0,   0.0},                              /* CONFIG_METHOD_FREESKY_PARM  */
    { 0,   0.0},                              /* CONFIG_METHOD_RECOVER_PARM  */
    { 0,   9}};                               /* CONFIG_METHOD_BONUS_PARM    */

static int CONFIG_METHOD_ParmNValues[CONFIG_METHOD_NPARMS] ={
    9,                                        /* CONFIG_METHOD_SELECT_PARM   */
    3,                                        /* CONFIG_METHOD_DEALLOC_PARM  */
    5,                                        /* CONFIG_METHOD_UNCROSS_PARM  */
    1,                                        /* CONFIG_METHOD_FREESKY_PARM  */
    1,                                        /* CONFIG_METHOD_RECOVER_PARM  */
    1};                                       /* CONFIG_METHOD_BONUS_PARM    */

static char **CONFIG_METHOD_ParmValues[CONFIG_METHOD_NPARMS] = {
    CONFIG_METHOD_SelectChoices ,             /* CONFIG_METHOD_SELECT_PARM   */
    CONFIG_METHOD_DeallocChoices,             /* CONFIG_METHOD_DEALLOC_PARM  */
    CONFIG_METHOD_UncrossChoices,             /* CONFIG_METHOD_UNCROSS_PARM  */
    CONFIG_METHOD_FreeSkyChoices ,            /* CONFIG_METHOD_FREESKY_PARM  */
    CONFIG_METHOD_FibreToRecover,             /* CONFIG_METHOD_RECOVER_PARM  */
    CONFIG_METHOD_BonusPrio};                 /* CONFIG_METHOD_BONUS_PARM    */

/*  The following defines the global variables used by the method-specific
 *  code.
 */

typedef struct CONFIG_METHOD_GlobalType {
   CONFIG_METHOD_MatrixElement **Matrix;       
                                 /* Address of memory allocated to hold addrs
                                  * of starts of rows of allocation matrix. */
   CONFIG_METHOD_MatrixElement *MatrixAddr;
                                 /* Address of memory allocated for matrix */
   CONFIG_METHOD_TargetElement *TargetMatrix;
   float *TargetCrowding;        /* Address of target crowding array */
   CONFIG_Boolean **Cross;       /* Address of crossing matrix rows*/
   CONFIG_Boolean *CrossAddr;    /* Address of crossing matric */
   CONFIG_Boolean SkyFlag;       /* Have we been asked for sky only */
   CONFIG_Boolean RecoverFlag;   /* Have we been asked for recovery */
   CONFIG_Boolean SwapPromote;   /* Have we been asked for recovery */
   int FibresForSky;             /* Number of fibres to be left for sky */
   int Field;                    /* The field being configured */
   int NTargets;                 /* Number of possible targets */
   int NPiv;                     /* Number of pivots (fibres) */
   int MaxSwapLevel;             /* How many levels of iteration */
   int MaxMoves;                 /* How many moves to make before quitting */ 
   int SwapLevel;                /* The Current Swap Depth */
   CONFIG_Boolean SwapAbort;     /* Abort flag used for iterative swapping */
   int Moves;                    /* How many moves have we made */
   int RecoverPiv;               /* Which Pivot are we trying to recover */
   int BonusPrio;                /* Priority below which we don't care */
   int *ConeIndex;               /* Pointer to Index array for cones */
   int *IBad;                    /* Pointer to Index array for cones */
   int *Tried;                   /* Pointer to list of attempted moves */
   double *VPivX;                /* Pointer to list of Vpiv x coords */
   double *VPivY;                /* Pointer to list of Vpiv y coords */
   double PcDone;                /* Status indicator for swaps */
   int SwapMatrix[4][20];        /* List of successful moves in this tree */
   CONFIG_METHOD_ParamValue ParameterValues[CONFIG_METHOD_NPARMS];
                                 /* Current parameter values */
   char ImportFile[80];          /* Name of Ascii file to Import*/
   int FibreTypesForSky[20];     /* Fibre types for which a specified number
                                    are to be allocated to sky targets. */
   int FibreSkyCounts[20];       /* Sky counts for FibreTypesForSky entries */
   int FibreSkyEntries;          /* Number of Fibre type/sky entries */
   int MeritValue;               /* Weighted merit value for configuration */
} CONFIG_METHOD_GlobalType;

 
CONFIG_METHOD_GlobalType CONFIG_METHOD_Global;
                                 /* Actual global structure used */
                                 
/* -------------------------------------------------------------------------- */

static void ConfigMethodDealloc (
   const CONFIG_MethodInfo *MethodInfo,
   int Pivot,                     /* (>) The number of the deallocated pivot */
   int Target,                    /* (>) Target allocated to this pivot */
   CONFIG_ReportFunction
            ReportFunction,       /* (>) Feedback routine to call */
   void *FunctionArg,             /* (>) User-supplied argument */
   StatusType *Status);           /* (!) Inherited status */

static void ConfigMethodNewAlloc (
   const CONFIG_MethodInfo *MethodInfo,
   int Pivot,                     /* (>) The number of the allocated pivot */
   int Target,                    /* (>) The number of the allocated target */
   double Theta);                 /* (>) The twist angle for the button */

static void ConfigMethodSetAlloc (
   const CONFIG_MethodInfo *MethodInfo,
   int Pivot,                     /* (>) The newly allocated pivot */
   int Target,                    /* (>) The newly allocated target */
   double Theta,                  /* (>) The theta angle for the button */
   StatusType *Status);           /* (!) Inherited status variable */

static CONFIG_Boolean ConfigMethodPickNext (
   const CONFIG_MethodInfo *MethodInfo,
   int *Pivot,                    /* (<) The newly allocated pivot */
   int *Target,                   /* (<) The newly allocated target */
   double *Theta);                /* (<) The theta angle for the button */

static void ConfigMethodUpdateMatrix (
   const CONFIG_MethodInfo *MethodInfo,
   CONFIG_ReportFunction
            ReportFunction,          /* (>) Feedback routine to call */
   void *FunctionArg,                /* (>) Caller-supplied argument */
   StatusType *Status);              /* (!) Inherited status */
   
static int ConfigMethodDisableSky (
   const CONFIG_MethodInfo *MethodInfo);

static int ConfigMethodReEnable (
   const CONFIG_MethodInfo *MethodInfo);

static int ConfigMethodMaxAlloc (void);

static int ConfigMethodReassignToSky (
   const CONFIG_MethodInfo *MethodInfo,
   int FibreType,                  /* (>) Type code for fibres to reallocate */
   int TotalToReallocate,          /* (>) Total # fibres to be reallocated */
   int PivotsToReallocate,         /* (>) Number of fibres to reallocate */
   CONFIG_ReportFunction
            ReportFunction,        /* (>) Feedback routine to call */
   void *FunctionArg);             /* (>) Caller-supplied argument */
   
static void ConfigMethodSortPivots (
   const CONFIG_MethodInfo *MethodInfo,
   StatusType *Status);            /* (!) Inherited status */
   
static int ConfigMethodAllocCount (
   const CONFIG_MethodInfo *MethodInfo);

/* All routines added after this point belong to GBD's extensions */

static void indexi(int n, int *arr, int *indx); /* NR-ish indexing routine */

static CONFIG_Boolean ConfigMethodDefineCones(
   const CONFIG_MethodInfo *MethodInfo,
   int LPrio,                     /* (>) Lower numerical priority to use */
   int HPrio);                    /* (>) Upper numerical priority to use */

static void ConfigMethodConeAllocate(
   const CONFIG_MethodInfo *MethodInfo,
   CONFIG_ReportFunction ReportFunction,  /* Function used to report progress */
   void *FunctionArg);                     /* Arg to pass to report function */

static void ConfigMethodBlockSwapAllocate(
   const CONFIG_MethodInfo *MethodInfo,
   CONFIG_ReportFunction ReportFunction,  /* Function used to report progress */
   void *FunctionArg);                    /* Arg to pass to report function */

static int ConfigMethodDoSwap(
   const CONFIG_MethodInfo *MethodInfo,
   int Target,                          /* The Target to try and allocate */ 
   int LastPivot,                       /* The last pivot to be moved */
   long *ReportTime,                    /* Time in secs count at last report */
   CONFIG_ReportFunction ReportFunction,/* Function used to report progress */
   void *FunctionArg);                  /* Arg to pass to report function */

static CONFIG_Boolean ConfigMethodSwapAllocate(
   const CONFIG_MethodInfo *MethodInfo,
   int  IPiv,                        /* (>) Pivot to Allocate */
   int  ITarget);                    /* (>) Target to Allocate */

static void ConfigMethodSwapDeallocate(
   const CONFIG_MethodInfo *MethodInfo,
   int  IPiv,                        /* (>) Pivot to Deallocate */
   int  ITarget);                    /* (>) Target to Deallocate */

static void ConfigMethodSwapDeallocateR(
   const CONFIG_MethodInfo *MethodInfo,
   CONFIG_ReportFunction
            ReportFunction,          /* (>) Feedback routine to call */
   void *FunctionArg,                /* (>) Caller-supplied argument */
   int  ILev);                       /* (>) Swap Level to Deallocate */

static void ConfigMethodSwapAllocateR(
   const CONFIG_MethodInfo *MethodInfo,
   CONFIG_ReportFunction
            ReportFunction,          /* (>) Feedback routine to call */
   void *FunctionArg,                /* (>) Caller-supplied argument */
   int  ILev);                       /* (> Swap Level to Allocate */

static CONFIG_Boolean ConfigMethodCrashTest (
   const CONFIG_MethodInfo *MethodInfo,
   int Pivot,                       /* (>) The pivot we just moved */
   int Target,                      /* (>) The target just allocated */
   double Theta);                   /* (>) The relevant theta */

static CONFIG_Boolean ConfigMethodFibreCross(
   const CONFIG_MethodInfo *MethodInfo,
   int IPiv,                       /*(>) 1st Fibre */
   int JPiv);                      /*(>  2nd Fibre */


static void ConfigMethodUncrossFibres(
   const CONFIG_MethodInfo *MethodInfo,
   CONFIG_ReportFunction  RF,          /* (>) Feedback routine to call */
   void *FA);                          /* (>) Caller-supplied argument */

static void ConfigMethodReportUncrossing(
   const CONFIG_MethodInfo *MethodInfo,
   int IPiv,int ITarg,
   int JPiv,int JTarg,
   CONFIG_ReportFunction  RF,          /* (>) Feedback routine to call */
   void *FA);                          /* (>) Caller-supplied argument */

static void ConfigMethodAutoDeallocate(
   const CONFIG_MethodInfo *MethodInfo,
   CONFIG_ReportFunction
            ReportFunction,            /* (>) Feedback routine to call */
   void *FunctionArg,                  /* (>) Caller-supplied argument */
   StatusType *Status);                /* (!) Inherited status */

static void ConfigMethodImportAlloc(
   const CONFIG_MethodInfo *MethodInfo,
   CONFIG_ReportFunction RF,
   void *FA);
   
static void ConfigMethodRandomSort (
   int Array[],                        /* (!) Array to be jumbled */
   int Entries);                       /* (>) Number of entries in array */

static int ConfigMethodNthLowest (
   int Array[],                        /* (>) Array in question */
   int Entries,                        /* (>) Number of entries in array */
   int N);                             /* (>) The value of N (from 1 up) */
   
/* -------------------------------------------------------------------------- */

/*              C o n f i g  M e t h o d  G l o b a l  I n i t
 *
 *  Function:
 *     Performs any global initialisation reqired for allocation.
 *
 *  Description:
 *     This routine, which is allocation algorithm specific, is called
 *     just once, during program startup, to allow the algorithm to perform 
 *     any necessary initialisation of its global structures.
 */

static void ConfigMethodGlobalInit (const CONFIG_MethodInfo *MethodInfo DUNUSED)
{
   /*  Local variables  */

   int Index;                     /* Index through method parameters */
   short UseOxfordAlgorithm;      /* True if Oxford algorithm is default */

   /*  We have two main configurations for running the allocation algorithm.
    *  In one, the Oxford algorithm due to Gavin Dalton is the default, and
    *  in the other the 'hardest' algorithm due to Keith Taylor is the
    *  default. In general, the Oxford algorithm (iterative allocation) is
    *  the most effective, but it has problems in some cases, particularly
    *  those with large numbers of targets and relatively few fibres.
    *
    *  For 2dF the default algorithm is the Oxford, but if this code is being
    *  built for OzPoz, the number of possible targets renders the Oxford 
    *  algorithm unsuitable. Work is being done to try to fix this. For the
    *  moment, we use the Oxford algorithm unless we determine that the
    *  instrument is FLAMES. We don't have a proper enquiry function for this
    *  test, so we fall back on an effective but unsatisfactory test on the
    *  number of pivots. (2dF has 404, 6dF has 154, FLAMES has 264 at the
    *  time of writing - Aug 2002).
    */
   
   UseOxfordAlgorithm = CONFIG_TRUE;
   if ((MethodInfo->NumberPivots > 200) && (MethodInfo->NumberPivots < 300)) {
       UseOxfordAlgorithm = CONFIG_FALSE;
   }
    
   /*  We set the various addresses etc to null or zero values - all of 
    *  these will be set by ConfigMethodInit(), but since this can be called 
    *  more than once, it must have the .MatrixAddr and .Matrix fields set 
    *  here or it may try to deallocate what it thinks is already allocated
    *  memory. The other fields are initialised here more for completeness 
    *  than anything.
    */

   CONFIG_METHOD_Global.MatrixAddr = NULL;
   CONFIG_METHOD_Global.Matrix = NULL;
   CONFIG_METHOD_Global.CrossAddr = NULL;
   CONFIG_METHOD_Global.Cross = NULL;
   CONFIG_METHOD_Global.TargetMatrix = NULL;
   CONFIG_METHOD_Global.TargetCrowding = NULL;
   CONFIG_METHOD_Global.Field = 0;
   CONFIG_METHOD_Global.ConeIndex = NULL;
   
   /*  These global values control how deeply the iterative 'Oxford' algorithm
    *  is prepared to go in its atttempts to gain additional allocations by
    *  swapping fibres. Both values are essentially arbitrary, but these
    *  values seem to give reasonable results. 
    *
    *  MaxSwapLevel is the maximum call depth to which the recursive swap
    *  phase of the algorithm will go. MaxMoves is the maximum number of
    *  swaps the algorithm will attempt for each unallocated target. In
    *  both cases, increasing the value increases both the amount of time
    *  taken by the algorithm, and the chances of getting an improved
    *  allocation. In both cases, its a matter of diminishing returns.
    */
    
   CONFIG_METHOD_Global.MaxSwapLevel = 10;
   CONFIG_METHOD_Global.MaxMoves = MethodInfo->NumberPivots;
   
   /*  The MeritValue field gives a 'figure of merit for the configuration -
    *  the sum of the squares of the priority values for all the allocated 
    *  targets. This can be used to compare two possible allocations, for 
    *  example two that differ only in that one has one high priority target
    *  allocated and the other a pair of lower priority targets. This field
    *  is a recent addition and is only used in some of 'iterative' algorithm's
    *  swap phase at present.
    */
    
   CONFIG_METHOD_Global.MeritValue = 0;            

   /*  If there are any 'any value' character parameters, set their addresses
    *  to null. Because if any new values are set, any memory already allocated
    *  to such a parameter will be freed.
    */

   for (Index = 0; Index < CONFIG_METHOD_NPARMS; Index++) {
      if (CONFIG_METHOD_ParmTypes[Index] == 'C') {
         if (CONFIG_METHOD_ParmNValues[Index] == 0) {
            CONFIG_METHOD_Global.ParameterValues[Index].StringAddress = NULL;
         }
      }
   }

   /*  The parameter values have to be initialised here, since they may never
    *  be changed at all and so need sensible default values. This section
    *  has to be modified if a new parameter is introduced. Note that 
    *  limited range character parameters have to have both their string
    *  addresses and integer values set. Logical and integer parameters need
    *  their integer values set, floating point values need their double
    *  value set, and character parameters that can take any value need to
    *  have their string addresses either left at NULL (from the previous loop)
    *  or set to point to a static string used as a default value.
    */

   if (UseOxfordAlgorithm) {
   
      /*  For the Oxford algorithm, we use iterative allocation, don't
       *  attempt tentative deallocation, and uncross at the end of the
       *  allocation.
       */
       
      CONFIG_METHOD_Global.ParameterValues[CONFIG_METHOD_SELECT_PARM]
                                       .IntegerValue = CONFIG_METHOD_SEL_GBD;
      CONFIG_METHOD_Global.ParameterValues[CONFIG_METHOD_SELECT_PARM]
          .StringAddress = CONFIG_METHOD_SelectChoices[CONFIG_METHOD_SEL_GBD];
          
      CONFIG_METHOD_Global.ParameterValues[CONFIG_METHOD_DEALLOC_PARM]
                                    .IntegerValue = CONFIG_METHOD_DEALLOC_NEVER;
      CONFIG_METHOD_Global.ParameterValues[CONFIG_METHOD_DEALLOC_PARM]
          .StringAddress = 
                     CONFIG_METHOD_DeallocChoices[CONFIG_METHOD_DEALLOC_NEVER];

      CONFIG_METHOD_Global.ParameterValues[CONFIG_METHOD_UNCROSS_PARM]
           .IntegerValue = CONFIG_METHOD_UNCROSS_ATEND;
      CONFIG_METHOD_Global.ParameterValues[CONFIG_METHOD_UNCROSS_PARM]
          .StringAddress = 
                    CONFIG_METHOD_UncrossChoices[CONFIG_METHOD_UNCROSS_ATEND];
   } else {
   
      /*  For the Taylor algorithm, we use the 'hardest' allocation option,
       *  try tentative deallocation if there are fibres unallocated, and
       *  we don't try an uncrossing pass, since this is only supported
       *  if the Oxford algorithm is being used. (It would be nice to
       *  get this to work, by the way.)
       */
       
      CONFIG_METHOD_Global.ParameterValues[CONFIG_METHOD_SELECT_PARM]
                                       .IntegerValue = CONFIG_METHOD_SEL_HARD;
      CONFIG_METHOD_Global.ParameterValues[CONFIG_METHOD_SELECT_PARM]
          .StringAddress = CONFIG_METHOD_SelectChoices[CONFIG_METHOD_SEL_HARD];
          
      CONFIG_METHOD_Global.ParameterValues[CONFIG_METHOD_DEALLOC_PARM]
                                    .IntegerValue = CONFIG_METHOD_DEALLOC_ONLY;
      CONFIG_METHOD_Global.ParameterValues[CONFIG_METHOD_DEALLOC_PARM]
          .StringAddress = 
                     CONFIG_METHOD_DeallocChoices[CONFIG_METHOD_DEALLOC_ONLY];

      CONFIG_METHOD_Global.ParameterValues[CONFIG_METHOD_UNCROSS_PARM]
           .IntegerValue = CONFIG_METHOD_UNCROSS_NEVER;
      CONFIG_METHOD_Global.ParameterValues[CONFIG_METHOD_UNCROSS_PARM]
          .StringAddress = 
                    CONFIG_METHOD_UncrossChoices[CONFIG_METHOD_UNCROSS_NEVER];
   }


   CONFIG_METHOD_Global.
             ParameterValues[CONFIG_METHOD_FREESKY_PARM].IntegerValue = 
                                                    FREESKY_PARAM_DEFAULT;

   CONFIG_METHOD_Global.
             ParameterValues[CONFIG_METHOD_FREESKY_PARM].StringAddress = NULL;

   CONFIG_METHOD_Global.
             ParameterValues[CONFIG_METHOD_RECOVER_PARM].IntegerValue = 1;

   CONFIG_METHOD_Global.
             ParameterValues[CONFIG_METHOD_RECOVER_PARM].StringAddress = NULL;

   CONFIG_METHOD_Global.
             ParameterValues[CONFIG_METHOD_BONUS_PARM].IntegerValue = 4;

   CONFIG_METHOD_Global.
             ParameterValues[CONFIG_METHOD_BONUS_PARM].StringAddress = NULL;
 

/*
 * We the maximum number for the FREESKY parameter and RECOVER parameter.
 *
 * FREESKY max was originally 400 and should probably be the number 
 * of pivots minus the number of guide, but we don't have access to the
 * number of guide easily.
 *
 * RECOVER was originally 403, so I have set it to the number of pivots
 * minus one, but I suspect this should have been 1 to Number of pivots.
 * It is possible the RECOVER mode does not work as yet.
 */
 
   CONFIG_METHOD_ParmLimits[CONFIG_METHOD_FREESKY_PARM][1] = 
       MethodInfo->NumberPivots-1;
   CONFIG_METHOD_ParmLimits[CONFIG_METHOD_RECOVER_PARM][1] = 
       MethodInfo->NumberPivots-1;
      
}
   
/* -------------------------------------------------------------------------- */

/*                    C o n f i g  M e t h o d  I n i t
 *
 *  Function:
 *     Performs any necessary initialisation reqired for allocation.
 *
 *  Description:
 *     This routine, which is allocation algorithm specific, is called
 *     to allow the algorithm to perform any necessary initialisation when
 *     a new target field is set up.
 *
 *     It is passed a pointer to the array of Pivot information structures,
 *     which give the details of the pivots (home position, whether broken
 *     or not, which spectrograph they are associated with, etc), a pointer to
 *     the array of Target information structures which give the details of
 *     the possible targets (position, associated spectrograph, etc.), and
 *     a pointer to the array of structures describing the current allocations,
 *     if any. The first two should not be changed by the allocation
 *     algorithm, but the last - the allocation array - should be updated
 *     by the allocation algorithm to reflect the allocations it makes.
 */

static void ConfigMethodInit (
   const CONFIG_MethodInfo *MethodInfo,
   CONFIG_ReportFunction
            ReportFunction,          /* (>) Feedback routine to call */
   void *FunctionArg,                /* (>) Caller-supplied argument */
   StatusType *Status)               /* (!) Inherited status */
{
   /*  Local variables  */

   int AllocCount;                   /* Number of pre-allocated fibres */
   double Alpha;                     /* Non-radial pivot angle for fibre */
   int Bytes;                        /* Bytes of dynamic memory required */
   double DeltaX;                    /* X displacement of fibre */
   double DeltaY;                    /* Y displacement of fibre */
   int FibreType;                    /* Type of fibre - guide or otherwise */
   double *VPivX;                    /* X VPiv position array*/
   double *VPivY;                    /* Y VPiv position array*/
   double DistanceSquared;           /* Square of required fibre length */
   int IC;                           /* Value of the uncrossing parameter */
   int ID;                           /* Value of the deallocation parameter */
   int IT;                           /* Value of the method select parameter */
   int IPiv,JPiv;                    /* Index through pivots */
   int ITarget;                      /* Index through targets */
   int IType;                        /* Index through pivot types */
   CONFIG_METHOD_MatrixElement **Matrix;
                                     /* Local synonym for .Matrix in globals */
   CONFIG_METHOD_MatrixElement *MatrixAddr;
                                     /* Start of memory allocated for matrix */
   CONFIG_METHOD_MatrixElement **MatrixLineAddr;   
                                     /* Addr of memory allocated to hold 
                                      * addresses of lines of matrix */
   CONFIG_METHOD_MatrixElement *MatrixPtr;
                                     /* Points to matrix line addresses */
   int NumberTypes;                  /* Number of types in sky count array */
   CONFIG_Boolean *CrossAddr;
   CONFIG_Boolean **CrossLineAddr;
   CONFIG_Boolean *CrossPtr;
   CONFIG_Boolean SkyFlag,CrossFlag,RecoverFlag;

   CONFIG_METHOD_TargetElement *TargetMatrix;  /*Local synonym */
   float *TargetCrowding;            /* Local synonym */
   int *Tried;
   int *Index,*IBad;
   double MaxExtensionSquared;       /* Square of maximum fibre extension */
   double ParkDistanceSquared;       /* Square of fibre length when parked */
   double PercentDone;               /* Percentage of current task completed */
   long PivotX;                      /* Pivot home position in X */
   long PivotY;                      /* Pivot home position in Y */
   int PossibleAllocations;          /* Number of possible pivot/target pairs */
   int ReportPivs;                   /* Report after processing this many */
   char TargetSpect;                 /* Spectrograph code for target */
   char TargetType;                  /* Type of target - guide or otherwise */
   double TargetX;                   /* Target position in X */
   double TargetY;                   /* Target position in Y */
   double Theta;                     /* Theta value for possible allocation */
   double ThetaFib;                  /* Theta for the fibre - from the pivot */
   int NTargets;                     /* Number of targets */
   int *Ind;                         /* Address of ConeIndex global array */
   int *IPtr;
   int fibreType;

   /*  Look at the allocation options and see if any of them are incompatible.
    */
   IT = CONFIG_METHOD_Global.ParameterValues[CONFIG_METHOD_SELECT_PARM]
     .IntegerValue;
   IC = CONFIG_METHOD_Global.ParameterValues[CONFIG_METHOD_UNCROSS_PARM]
     .IntegerValue;
   ID = CONFIG_METHOD_Global.ParameterValues[CONFIG_METHOD_DEALLOC_PARM]
     .IntegerValue;
     
   CONFIG_METHOD_Global.SkyFlag = CONFIG_FALSE;
   CONFIG_METHOD_Global.RecoverFlag = CONFIG_FALSE;
   CONFIG_METHOD_Global.BonusPrio =  CONFIG_METHOD_Global.
                       ParameterValues[CONFIG_METHOD_BONUS_PARM].IntegerValue;
   CONFIG_METHOD_Global.FibresForSky = CONFIG_METHOD_Global.
     ParameterValues[CONFIG_METHOD_FREESKY_PARM].IntegerValue;
   RecoverFlag = CONFIG_METHOD_Global.RecoverFlag;
   if (IT == CONFIG_METHOD_SEL_SKYONLY) {
     /*  It used to be that only the 'Oxford' algorithm handled the 'only
      *  allocate sky' option. However, with the introduction of the explicit
      *  sky allocation code it turns out that in the general case the Taylor
      *  algorithm does better. I (KS) am not sure why this is now the case -
      *  I may have broken something, and need to discuss this with GBD. 
      *  In any case, at this point there used to be code that set
      *  CONFIG_METHOD_Global.ParameterValues[CONFIG_METHOD_SELECT_PARM]
      *  .IntegerValue to CONFIG_METHOD_SEL_GBD. Instead, we now set it to
      *  use the 'hardest' algorithm instead. This needs to be revisited.
      */
     CONFIG_METHOD_Global.ParameterValues[CONFIG_METHOD_SELECT_PARM]
          .IntegerValue = CONFIG_METHOD_SEL_HARD;
     CONFIG_METHOD_Global.SkyFlag = CONFIG_TRUE;
   }
   SkyFlag = CONFIG_METHOD_Global.SkyFlag;
   
   /*  CrossFlag is true if one of the uncrossing options has been selected */
   
   CrossFlag = (IC == CONFIG_METHOD_UNCROSS_ONLY ||  
                         IC == CONFIG_METHOD_UNCROSS_THEN_ALLOCATE );
   
   /*  If we are not using the Oxford algorithm, we can't handle uncrossing.
    *  If uncrossing has been requested, we should issue a warning and then
    *  disable it.
    */
   
   if (CrossFlag) {
      if (IT != CONFIG_METHOD_SEL_GBD) {
         MethodInfo->InfoRoutine(MethodInfo->InfoArgument,
          "Fibre uncrossing cannot be run with this target selection algorithm"
                                                                             );
         MethodInfo->InfoRoutine(MethodInfo->InfoArgument,
          "Fibre uncrossing has been disabled.");
         CONFIG_METHOD_Global.ParameterValues[CONFIG_METHOD_UNCROSS_PARM]
                                .IntegerValue = CONFIG_METHOD_UNCROSS_NEVER;
         CONFIG_METHOD_Global.ParameterValues[CONFIG_METHOD_UNCROSS_PARM]
            .StringAddress = 
                    CONFIG_METHOD_UncrossChoices[CONFIG_METHOD_UNCROSS_NEVER];
         CrossFlag = CONFIG_FALSE;
      }
   }
   
   
   /*  If auto-reallocation (target selection set to 'broken') is
    *  being used, tentative deallocation cannot be used.
    */
    
   if ((IT == CONFIG_METHOD_SEL_BROKEN) 
              && (ID != CONFIG_METHOD_DEALLOC_NEVER)) {
      CONFIG_METHOD_Global.ParameterValues[CONFIG_METHOD_DEALLOC_PARM]
                                  .IntegerValue=CONFIG_METHOD_DEALLOC_NEVER;
      CONFIG_METHOD_Global.ParameterValues[CONFIG_METHOD_DEALLOC_PARM]
          .StringAddress = 
                     CONFIG_METHOD_DeallocChoices[CONFIG_METHOD_DEALLOC_NEVER];
      MethodInfo->InfoRoutine(MethodInfo->InfoArgument,
          "Tentative Deallocation is incompatible with Auto-Reallocation.");
      MethodInfo->InfoRoutine(MethodInfo->InfoArgument,
          "Tentative deallocation has been disabled.");
   }
   
   if (*Status != STATUS__OK) return;
    
   /*  Release any memory that might have been allocated for previous
    *  invocations of the algorithm.
    */
    
   if (CONFIG_METHOD_Global.MatrixAddr != NULL) {
      free (CONFIG_METHOD_Global.MatrixAddr);
      CONFIG_METHOD_Global.MatrixAddr = NULL;
   }
   if (CONFIG_METHOD_Global.Matrix != NULL) {
      free (CONFIG_METHOD_Global.Matrix);
      CONFIG_METHOD_Global.Matrix = NULL;
   }
   if (CONFIG_METHOD_Global.CrossAddr != NULL) {
      free (CONFIG_METHOD_Global.CrossAddr);
      CONFIG_METHOD_Global.CrossAddr = NULL;
   }
   if (CONFIG_METHOD_Global.Cross != NULL) {
      free (CONFIG_METHOD_Global.Cross);
      CONFIG_METHOD_Global.Cross = NULL;
   }
   if (CONFIG_METHOD_Global.VPivX != NULL) {
      free (CONFIG_METHOD_Global.VPivX);
      CONFIG_METHOD_Global.VPivX = NULL;
   }
   if (CONFIG_METHOD_Global.VPivY != NULL) {
      free (CONFIG_METHOD_Global.VPivY);
      CONFIG_METHOD_Global.VPivY = NULL;
   }
   if (CONFIG_METHOD_Global.TargetCrowding != NULL) {
      free (CONFIG_METHOD_Global.TargetCrowding);
      CONFIG_METHOD_Global.TargetCrowding = NULL;
   }
   if (CONFIG_METHOD_Global.Tried != NULL) {
      free (CONFIG_METHOD_Global.Tried);
      CONFIG_METHOD_Global.Tried = NULL;
   }
   Ind = CONFIG_METHOD_Global.ConeIndex;
   Ind--;
   if (CONFIG_METHOD_Global.ConeIndex != NULL) {
     free (Ind);
     CONFIG_METHOD_Global.ConeIndex = NULL;
   }
   IBad = CONFIG_METHOD_Global.IBad;
   IBad--;
   if (CONFIG_METHOD_Global.IBad != NULL) {
     free (IBad);
     CONFIG_METHOD_Global.IBad = NULL;
   }

   if (CONFIG_METHOD_Global.TargetMatrix != NULL) {
     NTargets = CONFIG_METHOD_Global.NTargets;
     for (IT=0;IT<NTargets;IT++){
       if (CONFIG_METHOD_Global.TargetMatrix[IT].Pivot != NULL) {
         free (CONFIG_METHOD_Global.TargetMatrix[IT].Pivot);
         CONFIG_METHOD_Global.TargetMatrix[IT].Pivot = NULL;
       }
       if (CONFIG_METHOD_Global.TargetMatrix[IT].ConeTargets != NULL) {
         free (CONFIG_METHOD_Global.TargetMatrix[IT].ConeTargets);
         CONFIG_METHOD_Global.TargetMatrix[IT].ConeTargets = NULL;
       }
     }
     free(CONFIG_METHOD_Global.TargetMatrix);
     CONFIG_METHOD_Global.TargetMatrix = NULL;
   }

   /*  For the purposes of reporting progress, we pick a suitable number
    *  of pivots. Here we report every 10% of changes, since this should
    *  be a pretty rapid operation.
    */

   ReportPivs = (int) ((float) MethodInfo->NumberPivots / 10.0);
   if (ReportPivs < 0) ReportPivs = 1;  

   /*  We allocate the matrix that holds all the possible allocations,
    *  there being one entry for each pivot-target combination - ie the
    *  matrix is an array NPiv by NTargets, each element simply indicating
    *  whether or not the corresponding allocation is possible. So we
    *  start by setting it up on the basis simply of 'can the fibre from
    *  this pivot reach this target?'. Then as we start to make allocations
    *  each new allocation will invalidate other possible allocations, and
    *  these have to be worked out as we go.

    *  ****GBD**** For efficiency reasons start by making a copy of this 
    *  matrix that will just hold what's possible given no allocations,
    *  this form of the matrix should never change... Actually the simplest
    *  way to do this is just to add another status variable to the struct.
    */

   Bytes = MethodInfo->NumberPivots * 
           MethodInfo->NumberTargets * 
           sizeof(CONFIG_METHOD_MatrixElement);

   if ((MatrixAddr = (CONFIG_METHOD_MatrixElement *) malloc (Bytes)) == NULL) {
      char s[100];
      sprintf(s,
              "Unable to allocate memory for allocation matrix of %d bytes",
              Bytes);
      MethodInfo->ErrRepRoutine(0,s);

      *Status = CONFIG__NO_MEMORY;
      goto Exit;
   }
   Bytes = MethodInfo->NumberPivots * sizeof(CONFIG_METHOD_MatrixElement *);
   if ((MatrixLineAddr = 
                   (CONFIG_METHOD_MatrixElement **) malloc (Bytes)) == NULL) {

       char s[100];
       sprintf(s,
               "Unable to allocate memory for matrix addresses (%d bytes)",
               Bytes);
       MethodInfo->ErrRepRoutine(0,s);
       *Status = CONFIG__NO_MEMORY;
       goto Exit;
   }
   Bytes = MethodInfo->NumberTargets * sizeof(CONFIG_METHOD_TargetElement);
   if ((TargetMatrix = (CONFIG_METHOD_TargetElement *) malloc (Bytes))==NULL) {
       char s[100];
       sprintf(s,
         "Unable to allocate memory for target allocation matrix of %d bytes",
               Bytes);
       MethodInfo->ErrRepRoutine(0,s);
      *Status = CONFIG__NO_MEMORY;
      goto Exit;
   }
   Bytes = MethodInfo->NumberTargets * sizeof(float);
   if ((TargetCrowding = (float *) malloc (Bytes))==NULL) {
       char s[100];
       sprintf(s,
         "Unable to allocate memory for target crowding array of %d bytes",
               Bytes);
       MethodInfo->ErrRepRoutine(0,s);
      *Status = CONFIG__NO_MEMORY;
      goto Exit;
   }
   Bytes = MethodInfo->NumberPivots * MethodInfo->NumberPivots * 
           sizeof(CONFIG_Boolean);
   if ((CrossAddr = (CONFIG_Boolean *) malloc (Bytes)) == NULL) {
       char s[100];
       sprintf(s,
               "Unable to allocate memory for crossing matrix of %d bytes",
               Bytes);
       MethodInfo->ErrRepRoutine(0,s);
       *Status = CONFIG__NO_MEMORY;
       goto Exit;
   }
   Bytes = MethodInfo->NumberPivots * sizeof(CONFIG_Boolean *);
   if ((CrossLineAddr = (CONFIG_Boolean **) malloc (Bytes)) == NULL) {
       char s[100];
       sprintf(s,
         "Unable to allocate memory for crossing matrix addresses (%d bytes)",
               Bytes);
               MethodInfo->ErrRepRoutine(0,s);
      *Status = CONFIG__NO_MEMORY;
      goto Exit;
   }
     
   Bytes = MethodInfo->NumberTargets * sizeof(int);
   if ((Tried = (int *) malloc (Bytes))==NULL) {
       char s[100];
       sprintf(s,
             "Unable to allocate memory for attempted moves array of %d bytes",
               Bytes);
       MethodInfo->ErrRepRoutine(0,s);
       *Status = CONFIG__NO_MEMORY;
       goto Exit;
   }

   Bytes = MethodInfo->NumberPivots * sizeof(double);
   if ((VPivX = (double *) malloc (Bytes))==NULL) {
       char s[100];
       sprintf(s,
               "Unable to allocate memory for vpivots array of %d bytes",
               Bytes);
       MethodInfo->ErrRepRoutine(0,s);
      *Status = CONFIG__NO_MEMORY;
      goto Exit;
   }
   Bytes = MethodInfo->NumberPivots * sizeof(double);
   if ((VPivY = (double *) malloc (Bytes))==NULL) {
       char s[100];
       sprintf(s,
               "Unable to allocate memory for vpivots array of %d bytes",
               Bytes);
       MethodInfo->ErrRepRoutine(0,s);
      *Status = CONFIG__NO_MEMORY;
      goto Exit;
   }

   CONFIG_METHOD_Global.ConeIndex = NULL;
   CONFIG_METHOD_Global.IBad = NULL;
   IBad = NULL;
   Index = NULL;
   Bytes = (MethodInfo->NumberTargets+1)* sizeof(int);
   if ((IBad = (int *) malloc (Bytes)) == NULL) {
     printf("Failed to find memory for IBad array...\n");
     goto Exit;
   }
   if ((Index = (int *) malloc (Bytes)) == NULL) {
     printf("Failed to find memory for Index array...\n");
     goto Exit;
   }
   /* Index +1 here to get around fortran numbering in indexi */
   CONFIG_METHOD_Global.ConeIndex = &Index[1];
   CONFIG_METHOD_Global.IBad = &IBad[1];

   /*  Set up the matrix line addresses in the global array used for this
    *  purpose. Note that the effect is that .Matrix can be accessed as if
    *  it were a 2D array, ie .Matrix[IPiv][ITarget] and this generates
    *  references to the appropriate elements of the allocation matrix.
    *  Similarly for the theta matrix.
    */

   CONFIG_METHOD_Global.Matrix = MatrixLineAddr;
   CONFIG_METHOD_Global.Cross = CrossLineAddr;
   CONFIG_METHOD_Global.NTargets = MethodInfo->NumberTargets;
   CONFIG_METHOD_Global.NPiv = MethodInfo->NumberPivots;
   CONFIG_METHOD_Global.VPivX = VPivX;
   CONFIG_METHOD_Global.VPivY = VPivY;
   CONFIG_METHOD_Global.Field = MethodInfo->Field;
   CONFIG_METHOD_Global.MatrixAddr = MatrixAddr;
   CONFIG_METHOD_Global.CrossAddr = CrossAddr;
   CONFIG_METHOD_Global.TargetMatrix = TargetMatrix;
   CONFIG_METHOD_Global.TargetCrowding = TargetCrowding;
   CONFIG_METHOD_Global.Tried = Tried;
   
   for (ITarget=0;ITarget<MethodInfo->NumberTargets;ITarget++){
     IPtr = NULL;
     Bytes = MethodInfo->NumberPivots*sizeof(int);
     if ((IPtr = (int *) malloc (Bytes)) == NULL){
         char s[100];
         sprintf(s,
               "Unable to allocate memory for Pivot sub-array of %d bytes, %d",
                 Bytes,ITarget);
         MethodInfo->ErrRepRoutine(0,s);
         *Status = CONFIG__NO_MEMORY;
         goto Exit;
     }
     CONFIG_METHOD_Global.TargetMatrix[ITarget].Pivot = IPtr;
   }
   for (ITarget=0;ITarget<MethodInfo->NumberTargets;ITarget++){
     IPtr = NULL;
     Bytes = MethodInfo->NumberTargets*sizeof(int);
     if ((IPtr = (int *) malloc (Bytes)) == NULL){
         char s[100];
         sprintf(s,
              "Unable to allocate memory for Target sub-array of %d bytes, %d",
                 Bytes,ITarget);
         MethodInfo->ErrRepRoutine(0,s);
         *Status = CONFIG__NO_MEMORY;
         goto Exit;
     }
     CONFIG_METHOD_Global.TargetMatrix[ITarget].ConeTargets = IPtr;
   }

   MatrixPtr = MatrixAddr;
   CrossPtr = CrossAddr;
   for (IPiv = 0; IPiv < MethodInfo->NumberPivots; IPiv++) {
      CONFIG_METHOD_Global.Matrix[IPiv] = MatrixPtr;
      CONFIG_METHOD_Global.Cross[IPiv] = CrossPtr;
      MatrixPtr += MethodInfo->NumberTargets;
      CrossPtr += MethodInfo->NumberPivots;
   }
   for (IPiv = 0; IPiv < MethodInfo->NumberPivots ; IPiv++){
     for (JPiv = 0; JPiv < MethodInfo->NumberPivots ; JPiv++){
       CONFIG_METHOD_Global.Cross[IPiv][JPiv] = CONFIG_FALSE;
     }
   }
   for (ITarget = 0; ITarget < MethodInfo->NumberTargets; ITarget++){
      CONFIG_METHOD_Global.TargetMatrix[ITarget].TargetPivot = -1;
      TargetX = (double) MethodInfo->TargetInfo[ITarget].X;
      TargetY = (double) MethodInfo->TargetInfo[ITarget].Y;
      CONFIG_METHOD_Global.TargetMatrix[ITarget].TargetR = 
                    sqrt(TargetX*TargetX + TargetY*TargetY + 0.0000001);
      CONFIG_METHOD_Global.TargetMatrix[ITarget].TargetT = 
                                                 atan2(TargetY,TargetX);
   }

   /*  This section attempts to calculate a 'crowding' metric for each
    *  target, in order to allow allocation methods that attempt to
    *  allocate on the basis of how crowded the targets are. The figure
    *  stored in the TargetCrowding global array for each target is the
    *  average distance squared between this target and all the other
    *  targets. The lower this is, the more crowded the target.
    *  (Actually, this makes targets on the edge of the field look 
    *  unnaturally uncrowded, so this really needs a bit more tweaking.)
    */
    
   for (ITarget=0; ITarget < MethodInfo->NumberTargets; ITarget++){
      double TotDistSq = 0.0;
      int JTarget;
      TargetX = (double) MethodInfo->TargetInfo[ITarget].X;
      TargetY = (double) MethodInfo->TargetInfo[ITarget].Y;
      for (JTarget=0; JTarget < MethodInfo->NumberTargets; JTarget++){
         double JTargetX;
         double JTargetY;
         double DistSq;
         JTargetX = (double) MethodInfo->TargetInfo[JTarget].X;
         JTargetY = (double) MethodInfo->TargetInfo[JTarget].Y;
         DistSq = ((JTargetX - TargetX) * (JTargetX - TargetX)) +
                      ((JTargetY - TargetY) * (JTargetY - TargetY));
         TotDistSq += DistSq;
      }
      CONFIG_METHOD_Global.TargetCrowding[ITarget] = TotDistSq;
   }
   
   /*  For convenience, we access the Matrix from now on through the
    *  'Matrix' variable, which operates syntatically just like a 
    *  2D array (although that isn't what it is). 
    */

   Matrix = CONFIG_METHOD_Global.Matrix;
   TargetMatrix = CONFIG_METHOD_Global.TargetMatrix;

   /*  Initially, we run through all the elements of the array calculating
    *  whether or not an allocation is possible purely on the basis of fibre
    *  length and on whether or not a target and fibre are incompatible on
    *  the grounds that one is a fiducial and the other isn't.
    *
    *  All the next bit should be executed whenever Method is called, 
    *  irrespective of any  previous allocations ***GBD*** */


   for (ITarget = 0; ITarget < MethodInfo->NumberTargets; ITarget++) {
     TargetMatrix[ITarget].NumPivots = 0;
     TargetMatrix[ITarget].NConeTargets = 0;
   }
   for (IPiv = 0; IPiv < MethodInfo->NumberPivots; IPiv++) {
      if (ReportFunction != NULL) {
         if ((IPiv % ReportPivs) == 0) {
            PercentDone = ((float) IPiv / 
                                 (float) MethodInfo->NumberPivots) * 100.0;
            if ((*ReportFunction)(FunctionArg,'P',"Setting up path matrix",
                 0,0,0,0,0,0.0,PercentDone) != 0) {
                MethodInfo->InfoRoutine(MethodInfo->InfoArgument,
                                        "Cancel request received");
               *Status = CONFIG__CANCELLED;
               goto Exit;
            }
         }
      } 
      PivotX = MethodInfo->PivotInfo[IPiv].X;
      PivotY = MethodInfo->PivotInfo[IPiv].Y;
      MaxExtensionSquared = (double)MethodInfo->PivotInfo[IPiv].FibreLength;
      MaxExtensionSquared = MaxExtensionSquared * MaxExtensionSquared;
      FibreType = MethodInfo->PivotInfo[IPiv].FibreType;
      for (ITarget = 0; ITarget < MethodInfo->NumberTargets; ITarget++) {
         Matrix[IPiv][ITarget].CollidingPivot = 0;
         TargetX = (double) MethodInfo->TargetInfo[ITarget].X;
         TargetY = (double) MethodInfo->TargetInfo[ITarget].Y;
         DeltaX = (double) PivotX - TargetX;
         DeltaY = (double) PivotY - TargetY;
         DistanceSquared = (DeltaX * DeltaX) + (DeltaY * DeltaY);
         DeltaX = (double) PivotX - MethodInfo->PivotInfo[IPiv].ParkX;
         DeltaY = (double) PivotY - MethodInfo->PivotInfo[IPiv].ParkY;
         ParkDistanceSquared = (DeltaX * DeltaX) + (DeltaY * DeltaY);
         if ((DistanceSquared < MaxExtensionSquared) && 
                      (DistanceSquared > ParkDistanceSquared)) {
            TargetType = MethodInfo->TargetInfo[ITarget].Type;
            TargetSpect = MethodInfo->TargetInfo[ITarget].Spect;
            if (!MethodInfo->PivotInfo[IPiv].AllowAllocChange) {
               /*
                *  We are not allowed to change the allocation of this pivot.
                */
               Matrix[IPiv][ITarget].PathStatus = CONFIG_METHOD_IGNORE;

            } else if (!MethodInfo->FibreTargetCompatible(MethodInfo,
                                           TargetType,TargetSpect,FibreType)) {
               /* 
                *  See if he pivot and the target are of incompatible types 
                */
               Matrix[IPiv][ITarget].PathStatus = CONFIG_METHOD_INCOMPATIBLE;

            } else {
               /*
                * Check if the bend angle would allow the allocation
                */
               if (MethodInfo->CheckAlpha(
                      MethodInfo,
                      &(MethodInfo->TargetInfo[ITarget]),
                      &(MethodInfo->PivotInfo[IPiv]),&Alpha)) {
                  /*
                   * Bend angle would allow the allocation.  Caculate the
                   * theta angle we would get and then check against invalid
                   * positions
                   */
                  ThetaFib = MethodInfo->ThetaFib (
                      MethodInfo,
                      &(MethodInfo->TargetInfo[ITarget]),
                      &(MethodInfo->PivotInfo[IPiv]));
                  Theta = (ThetaFib > PI) ? ThetaFib - PI : ThetaFib + PI;
		  fibreType = MethodInfo->PivotInfo[IPiv].FibreType;
                  if (!MethodInfo->ColInvPos(MethodInfo, fibreType,
                                            TargetX,TargetY,Theta)){
                                            
                      /*  Almost there. We need to see if the allocation would
                       *  result in a fibre crossing an adjacent parked pivot
                       *  position. Note that for some systems (2dF) this isn't
                       *  a problem - and the test routine always returns OK.
                       *  For others (OzPoz, 6dF) we don't want to cover a
                       *  parked button, or EVEN to cover the position where a
                       *  pivot goes when parked, since this can make things
                       *  almost impossible for the delta task, even if the
                       *  configuration is technically feasible.
                       */
                      
                      if (MethodInfo->AdjPark(MethodInfo,IPiv,TargetX,
                                                       TargetY,Theta)) {
                         Matrix[IPiv][ITarget].PathStatus = 
                                                   CONFIG_METHOD_PARK_CLASH;
                      } else {
                      
                          /*
                           * Position is ok.  But ensure we don't do this if 
                           * the fibre is broken;
                           */
                          Matrix[IPiv][ITarget].PathStatus = 
                                                     CONFIG_METHOD_POSSIBLE;
                          Matrix[IPiv][ITarget].Alpha = Alpha;
                          /* Don't do this if broken */
                          FibreType = MethodInfo->PivotInfo[IPiv].FibreType;
                          if (!MethodInfo->FibreIsBroken(MethodInfo,
                                                             FibreType)) {
                             TargetMatrix[ITarget].Pivot[TargetMatrix[ITarget].
                                                              NumPivots]=IPiv;
                             TargetMatrix[ITarget].NumPivots++;
                          }
                      }
                   } else {
                      /* 
                       * Position would collide with an invalid positon
                       */
                      Matrix[IPiv][ITarget].PathStatus = CONFIG_METHOD_SCREWED;
                      if (IT == CONFIG_METHOD_SEL_IMPORTONLY) {
                         Matrix[IPiv][ITarget].PathStatus = 
                                                   CONFIG_METHOD_POSSIBLE;
                      }
                      /*
                       * If already allocated, output a warning.
                       */
                      if ((MethodInfo->AllocInfo[IPiv].State == 'A') && 
                         (MethodInfo->AllocInfo[IPiv].TargetIndex == ITarget)){
                         /*  This printf() shouldn't be here - either don't do
                          *  the test or produce a proper error message! *** */
                         printf("Invalid position (screw ?)");
                         printf("  Collision set for allocation: %d %d\n",
                                                              IPiv,ITarget);
                      }
                   }
               } else {
                  /*
                   *  Bend angle would prevent allocation.
                   */
                  Matrix[IPiv][ITarget].PathStatus = CONFIG_METHOD_BAD_ALPHA;
                  if (IT == CONFIG_METHOD_SEL_IMPORTONLY)
                    Matrix[IPiv][ITarget].PathStatus = CONFIG_METHOD_POSSIBLE;
                  /*
                   *  If already allocated, output a warning.
                   */
                  if ((MethodInfo->AllocInfo[IPiv].State == 'A') && 
                      (MethodInfo->AllocInfo[IPiv].
                                      TargetIndex == ITarget)){
                    printf("Bad Alpha Angle set for allocation: %d %d\n"
                           ,IPiv,ITarget);
                  }
                }
             }
          } else {
            /*
             * Pivot won't reach target.
             */
            Matrix[IPiv][ITarget].PathStatus = CONFIG_METHOD_WONT_REACH;
            if (IT == CONFIG_METHOD_SEL_IMPORTONLY) 
              Matrix[IPiv][ITarget].PathStatus = CONFIG_METHOD_POSSIBLE;
            /*
             *  If already allocated, output a warning.
             */
            if ((MethodInfo->AllocInfo[IPiv].State == 'A') && 
                (MethodInfo->AllocInfo[IPiv].
                 TargetIndex == ITarget)){
                printf("Bad Extension set for allocation: %d %d\n"
                       ,IPiv,ITarget);
            }

          }
       }
    }
    
   /*  Sort the entries in the TargetMatrix[].Pivot[] arrays. These list
    *  the pivots that can reach each target. We organise them into increasing
    *  order of bend angle, so that pivot representing the straightest 
    *  allocation comes first. This will encourage the iterative code to 
    *  allocate the straightest fibres first.
    */
    
   ConfigMethodSortPivots (MethodInfo,Status);
   if (*Status != 0) goto Exit;
   
   /* Reset all the sky target counts for each fibre type. Setting them to
    * -1 means that no attempt will be made to distinguish between sky and
    * non-sky types for each type. Setting the types to -1 means that these
    * are not valid type codes anyway.
    */
    
   NumberTypes = sizeof(CONFIG_METHOD_Global.FibreTypesForSky)/sizeof(int);
   for (IType = 0; IType < NumberTypes; IType++) {
       CONFIG_METHOD_Global.FibreTypesForSky[IType] = -1;
       CONFIG_METHOD_Global.FibreSkyCounts[IType] = -1;
   }
   CONFIG_METHOD_Global.FibreSkyEntries = 0;
   
   /*
    * Check to see if the nutty astronomer is trying to place any objects
    * beyond the useable edges of the field.  This should now be indicated
    * by a target with a priority of CONFIG_PRIO_INVALID 
    */

   for (ITarget = 0; ITarget < MethodInfo->NumberTargets; ITarget++){
       if (MethodInfo->TargetInfo[ITarget].Priority == CONFIG_PRIO_INVALID) {
           for (IPiv = 0 ; IPiv < MethodInfo->NumberPivots; IPiv++){
               Matrix[IPiv][ITarget].PathStatus = CONFIG_METHOD_OUTSIDE;
           }
       }
   }

   /* 
    * If we're not on sky allocations then zap all the sky objects. This 
    * scheme has really been replaced now by the explicit control over
    * sky allocations on a per-fibre type basis, so we don't expect to 
    * see that used. (We expect to see the 'fibres for sky' parameter 
    * always zero.)
    */
    
   if (!SkyFlag && !CrossFlag) {
     if (CONFIG_METHOD_Global.FibresForSky != 0){
       for (ITarget = 0; ITarget < MethodInfo->NumberTargets ; ITarget++){
         if (MethodInfo->TargetInfo[ITarget].Type == 'S'){
           for (IPiv = 0 ; IPiv < MethodInfo->NumberPivots ; IPiv++){
             if (Matrix[IPiv][ITarget].PathStatus == CONFIG_METHOD_POSSIBLE){
               Matrix[IPiv][ITarget].PathStatus = CONFIG_METHOD_IGNORE;
             }
           }
         }
       }
     }
   }

   /*  The allocation information we have already been passed can be
    *  used to rule out a range of possible allocations. Any fibres that
    *  are broken can no longer be allocated, and we handle that first. At the
    *  same time, we count the number of allocated fibres that we've been given.
    */

   AllocCount = 0;
   for (IPiv = 0; IPiv < MethodInfo->NumberPivots; IPiv++) {
      if (ReportFunction != NULL) {
         if ((IPiv % 5) == 0) {
            PercentDone = ((float) IPiv / 
                                   (float) MethodInfo->NumberPivots) * 100.0;
            if ((*ReportFunction)(FunctionArg,'P',"Checking for broken fibres",
                 0,0,0,0,0,0.0,PercentDone) != 0) {
                MethodInfo->InfoRoutine(MethodInfo->InfoArgument,
                                        "Cancel request received");
                *Status = CONFIG__CANCELLED;
                goto Exit;
            }
         }
      }
      FibreType = MethodInfo->PivotInfo[IPiv].FibreType;
      if (MethodInfo->FibreIsBroken(MethodInfo,FibreType)) {
         for (ITarget = 0; ITarget < MethodInfo->NumberTargets; ITarget++) {
            Matrix[IPiv][ITarget].PathStatus = CONFIG_METHOD_BROKEN;
            if (IT == CONFIG_METHOD_SEL_IMPORTONLY){ 
              Matrix[IPiv][ITarget].PathStatus = CONFIG_METHOD_POSSIBLE;
            }
         }
      } else if (MethodInfo->AllocInfo[IPiv].State == 'A') {
         AllocCount++;
      }
   }
   
   /*  We now do a pass through the matrix, setting the theta values for
    *  those allocations that are possible. In each case, we assume no twist
    *  of the pivot, and so simply calculate the theta value that corresponds
    *  to a straight button, in line with the fibre.
    */

   PossibleAllocations = 0;
   for (IPiv = 0; IPiv < MethodInfo->NumberPivots; IPiv++) {
      if (ReportFunction != NULL) {
         if ((IPiv % ReportPivs) == 0) {
            PercentDone = ((float) IPiv / 
                                 (float) MethodInfo->NumberPivots) * 100.0;
            if ((*ReportFunction)(FunctionArg,'P',
                       "Setting radial button angles",0,0,0,0,0,0.0,
                                                     PercentDone) != 0) {
                MethodInfo->InfoRoutine(MethodInfo->InfoArgument,
                                        "Cancel request received");
                *Status = CONFIG__CANCELLED;
                goto Exit;
            }
         }
      } 
      for (ITarget = 0; ITarget < MethodInfo->NumberTargets; ITarget++) {
         Theta = 0.0;
         if (Matrix[IPiv][ITarget].PathStatus == CONFIG_METHOD_POSSIBLE) {
            PossibleAllocations++;
            ThetaFib = MethodInfo->ThetaFib (
                         MethodInfo,
                         &(MethodInfo->TargetInfo[ITarget]),
                                       &(MethodInfo->PivotInfo[IPiv]));
            Theta = (ThetaFib > PI) ? ThetaFib - PI : ThetaFib + PI;
         }
         Matrix[IPiv][ITarget].Theta = Theta;           
      }
   }

   /*  Now we do a pass through the already allocated pivots. First we set the
    *  row and column for the pivot and fibre in question in the possible
    *  path matrix to 'in use' (we can't use ConfigMethodSetAlloc() for this,
    *  since it will perform checks that we don't want performed here). Then
    *  we simply re-evaluate all the possible paths in the light of the 
    *  known allocations.
    *  ****GBD**** Since this routine is essentially initialisation only I
    *  guess this bit can stay in, since I don't think I'm expecting things to 
    *  be pre-allocated...apart from for the uncrossing case, which means that
    *  we need to check here to see if that's what we were asked for!
    */
  
   
   if (CONFIG_METHOD_Global.ParameterValues[CONFIG_METHOD_SELECT_PARM]
       .IntegerValue == CONFIG_METHOD_SEL_GBD || 
       CONFIG_METHOD_Global.ParameterValues[CONFIG_METHOD_SELECT_PARM]
       .IntegerValue == CONFIG_METHOD_SEL_BROKEN || 
       CONFIG_METHOD_Global.ParameterValues[CONFIG_METHOD_UNCROSS_PARM]
       .IntegerValue == CONFIG_METHOD_UNCROSS_ONLY ||
       CONFIG_METHOD_Global.ParameterValues[CONFIG_METHOD_UNCROSS_PARM]
       .IntegerValue == CONFIG_METHOD_UNCROSS_THEN_ALLOCATE) {
     if (AllocCount > 0) {
       for (IPiv = 0 ; IPiv < MethodInfo->NumberPivots ; IPiv++){
         if (MethodInfo->AllocInfo[IPiv].State == 'A') {
           ITarget = MethodInfo->AllocInfo[IPiv].TargetIndex;
           TargetMatrix[ITarget].TargetPivot = IPiv;
           Theta = MethodInfo->AllocInfo[IPiv].Theta;
           Matrix[IPiv][ITarget].Theta = Theta;
           TargetX = MethodInfo->TargetInfo[ITarget].X;
           TargetY = MethodInfo->TargetInfo[ITarget].Y;
           MethodInfo->GetVirtPiv(MethodInfo,TargetX,TargetY,Theta,
                                  &DeltaX,&DeltaY);
           VPivX[IPiv] = DeltaX;
           VPivY[IPiv] = DeltaY;
         }
         if (ReportFunction != NULL) {
           if ((IPiv % ReportPivs) == 0) {
             PercentDone = ((float) IPiv / 
                                (float) MethodInfo->NumberPivots) * 100.0;
             if ((*ReportFunction)(FunctionArg,'P',
                                   "Processing allocated fibres",
                                   0,0,0,0,0,0.0,PercentDone) != 0) {
                 MethodInfo->InfoRoutine(MethodInfo->InfoArgument,
                                        "Cancel request received");
                 *Status = CONFIG__CANCELLED;
                 goto Exit;
             }
           }
         }
       }
     }
   } else { /* do whatever you would have done before */

     if (AllocCount > 0) {
        ConfigMethodUpdateMatrix (MethodInfo,ReportFunction,FunctionArg,Status);
     }
   }      
Exit:;

   /*  On the way out, if there were any errors, release any memory
    *  obtained by this routine.
    */

   if (*Status != STATUS__OK) {
      if (CONFIG_METHOD_Global.MatrixAddr != NULL) {
         free (CONFIG_METHOD_Global.MatrixAddr);
         CONFIG_METHOD_Global.MatrixAddr = NULL;
      }
      if (CONFIG_METHOD_Global.Matrix != NULL) {
         free (CONFIG_METHOD_Global.Matrix);
         CONFIG_METHOD_Global.Matrix = NULL;
      }
      if (CONFIG_METHOD_Global.CrossAddr != NULL) {
         free (CONFIG_METHOD_Global.CrossAddr);
         CONFIG_METHOD_Global.CrossAddr = NULL;
      }
      if (CONFIG_METHOD_Global.Cross != NULL) {
         free (CONFIG_METHOD_Global.Cross);
         CONFIG_METHOD_Global.Cross = NULL;
      }
      if (CONFIG_METHOD_Global.TargetMatrix != NULL) {
        for (IT=0;IT<MethodInfo->NumberTargets;IT++){
          if (CONFIG_METHOD_Global.TargetMatrix[IT].Pivot != NULL) {
            free (CONFIG_METHOD_Global.TargetMatrix[IT].Pivot);
            CONFIG_METHOD_Global.TargetMatrix[IT].Pivot = NULL;
          }
          if (CONFIG_METHOD_Global.TargetMatrix[IT].ConeTargets != NULL) {
            free (CONFIG_METHOD_Global.TargetMatrix[IT].ConeTargets);
            CONFIG_METHOD_Global.TargetMatrix[IT].ConeTargets = NULL;
          }
        }
        free(CONFIG_METHOD_Global.TargetMatrix);
        CONFIG_METHOD_Global.TargetMatrix = NULL;
      }
   }
   
   /*  Just a dummy call to prevent compilers complaining about a diagnostic
    *  routine that isn't actually used at present.
    */
    
   if (0) (void) ConfigMethodAllocCount(MethodInfo);

}

/* -------------------------------------------------------------------------- */

/*              C o n f i g  M e t h o d  S e t  S k y
 *
 *  Function:
 *     Sets the number of fibres of a given type to be allocated to sky.
 *
 *  Description:
 *     This routine allows the higher levels to exercise precise control
 *     over the number of fibres of each type that ought to be allocated
 *     to sky targets. This is a requirement that all allocation algorithms
 *     need to support, so it is more appropriate to handle it in this way
 *     than through the method-specific parameters. The settings laid down
 *     by a call to this routine should be taken into account by the next
 *     call to ConfigMethodAllocate(). The settings made by this routine 
 *     are only relevant for fibre types that can be allocated to both
 *     sky and non-sky targets. ConfigMethodInit() resets the sky allocation
 *     settings to a mode where no distinction is made between sky and non-
 *     sky targets.
 */

static void ConfigMethodSetSky (
   const CONFIG_MethodInfo *MethodInfo,
   int FibreType,                    /* (>) Fibre type code */
   int NumberSkyAllocations,         /* (>) Number of fibres for sky targets */
   StatusType *Status)               /* (!) Inherited status */
{
   /*  Local variables */
   
   CONFIG_Boolean Found;             /* True if entry found for this type */
   int IType;                        /* Index through pivot types */
   int MaxTypes;                     /* Size of sky count array */
   int NumberTypes;                  /* Number of types in sky count array */
   
   if (*Status != STATUS__OK) return;
   
   /*  See if we already have an entry for this type in the global array
    *  used to return these settings.
    */
    
   Found = CONFIG_FALSE;
   MaxTypes = sizeof(CONFIG_METHOD_Global.FibreTypesForSky)/sizeof(int);
   NumberTypes = CONFIG_METHOD_Global.FibreSkyEntries;
   for (IType = 0; IType < NumberTypes; IType++) {
      if (CONFIG_METHOD_Global.FibreTypesForSky[IType] == FibreType) {
         CONFIG_METHOD_Global.FibreSkyCounts[IType] = NumberSkyAllocations;
         Found = CONFIG_TRUE;
         break;
      }
   }
   
   /*  If not, we create a new entry */
   
   if (NumberTypes >= MaxTypes) {
      *Status = CONFIG__PARM_OUT_RANGE;
   } else {
      CONFIG_METHOD_Global.FibreTypesForSky[NumberTypes] = FibreType;
      CONFIG_METHOD_Global.FibreSkyCounts[NumberTypes] = NumberSkyAllocations;
      CONFIG_METHOD_Global.FibreSkyEntries++;
   }
   
}   

/* -------------------------------------------------------------------------- */

/*              C o n f i g  M e t h o d  S o r t  P i v o t s
 *
 *  Function:
 *     Sorts pivot arrays in order of bend angle.
 *
 *  Description:
 *     This routine sorts the the entries in the TargetMatrix[].Pivot[] arrays.
 *     These list the pivots that can reach each target. We organise them into
 *     increasing order of bend angle, so that pivot representing the 
 *     straightest allocation comes first. This will encourage the iterative
 *     code to allocate the straightest fibres first. This assumes that
 *     these arrays have already been filled by the initialisation code.
 */

static void ConfigMethodSortPivots (
   const CONFIG_MethodInfo *MethodInfo,     /* (>) Method information */
   StatusType *Status)                      /* (!) Inherited status */
{
   /*  Local variables */
   
   double* Alphas;                   /* Malloc'd array of bend angles */
   int IPiv;                         /* Loop index through pivots */
   int ITarget;                      /* Loop index through targets */
   int JPiv;                         /* Second loop index through pivots */
   CONFIG_METHOD_MatrixElement **Matrix;
                                     /* Synonym for .Matrix in globals */
   int NPivots;                      /* Number of pivots for a given target */
   int NumberPivots;                 /* Total number of pivots */
   int NumberTargets;                /* Total number of targets */
   int* Pivots;                      /* Malloc'd array of pivot numbers */
   CONFIG_METHOD_TargetElement *TargetMatrix;
                                     /* Synonym for .TargetMatrix in globals */
   
   
   if (*Status != 0) return;
   
   Pivots = (int*) NULL;
   Alphas = (double*) NULL;
   
   TargetMatrix = CONFIG_METHOD_Global.TargetMatrix;
   Matrix = CONFIG_METHOD_Global.Matrix;
   
   NumberPivots = MethodInfo->NumberPivots;
   NumberTargets = MethodInfo->NumberTargets;
  
   /*  Allocate two work arrays for the sort. One holds the list of pivot
    *  numbers, and the other holds the corresponding bend angle for each
    *  if allocated to the target in question.
    */
   
   Pivots = (int*) malloc (NumberPivots * sizeof(int));
   Alphas = (double*) malloc (NumberPivots * sizeof(double));
   if ((Pivots == (int*) NULL) || (Alphas == (double*) NULL)) {
      char Error[100];
      sprintf(Error,
          "Unable to allocate memory for sort work arrays for %d pivots",
                                                             NumberPivots);
      MethodInfo->ErrRepRoutine(0,Error);
      *Status = CONFIG__NO_MEMORY;
      goto Exit;
   }
   
   /*  Now work through each target. For each target, collect the list of
    *  pivots that can reach it, and the corresponding fibre bend angles,
    *  into the arrays Pivots and Alphas. Then sort these on the basis of
    *  the alpha values, and copy the sorted pivot list into the pivot
    *  array held in TargetMatrix for that target.
    */
   
   for (ITarget = 0; ITarget < NumberTargets; ITarget++){
      NPivots = TargetMatrix[ITarget].NumPivots;
      if (NPivots > 0) {
         for (IPiv = 0; IPiv < NPivots; IPiv++) {
            Pivots[IPiv] = TargetMatrix[ITarget].Pivot[IPiv];
            Alphas[IPiv] = Matrix[Pivots[IPiv]][ITarget].Alpha;
         }
         
         /*  The sort is a crude bubble sort - good enough for something
          *  only done once for each target and which doesn't involve a
          *  large number of entries.
          */
         
         for (IPiv = 0; IPiv < NPivots; IPiv++) {
            for (JPiv = IPiv + 1; JPiv < NPivots; JPiv++) {
               if (Alphas[IPiv] > Alphas[JPiv]) {
                  double TempAlpha = Alphas[IPiv];
                  int TempPivot = Pivots[IPiv];
                  Alphas[IPiv] = Alphas[JPiv];
                  Alphas[JPiv] = TempAlpha;
                  Pivots[IPiv] = Pivots[JPiv];
                  Pivots[JPiv] = TempPivot;
               }
            }
         }
         
         /*  Copy back sorted array of pivot numbers */
         
         for (IPiv = 0; IPiv < NPivots; IPiv++) {
            TargetMatrix[ITarget].Pivot[IPiv] = Pivots[IPiv];
         }
      }
   }
   
Exit:;

   /*  Release any allocated workspace */
   
   if (Pivots) free(Pivots);
   if (Alphas) free(Alphas);
        
}

/* -------------------------------------------------------------------------- */

/*              C o n f i g  M e t h o d  A l l o c a t e
 *
 *  Function:
 *     Performs a full allocation of fibres to targets.
 *
 *  Description:
 *     This is the method-specific layer routine that actually performs
 *     an allocation.
 */

static void ConfigMethodAllocate (
   const CONFIG_MethodInfo *MethodInfo,
   CONFIG_ReportFunction
            ReportFunction,          /* (>) Feedback routine to call */
   void *FunctionArg,                /* (>) Caller-supplied argument */
   StatusType *Status)               /* (!) Inherited status */
{
   /*  Local variables */
   
   int AllocCount;                   /* Number of pivots allocated so far */
   CONFIG_Boolean CancelFlag;        /* True if cancel request made */
   CONFIG_Boolean Candidate;         /* True if a candidate for deallocation */
   int *CandidateArray;              /* List of pivots for deallocation */
   int CandidatePivots;              /* Number pivots that can be deallocated */
   int DeallocCode;                  /* Value of deallocation parameter */
   CONFIG_Boolean DeAllocSkyAtEnd;   /* True if sky deallocation deferred */
   int ICand;                        /* Index through candidate pivots */
   int IPiv;                         /* Pivot index number */
   int NPiv;                         /* Total number of pivots */
   int ITarget;                      /* Target index number */ 
   int IType;                        /* Index through fibre types */ 
   int NTargets;                     /* Number of targets */
   int MaxAllocs;                    /* Maximum possible allocations */
   double PercentDone;               /* Percentage of current task completed */
   int Pivot;                        /* Loop index through pivots */
   CONFIG_Boolean SkyControl;        /* True if sky allocations controlled */
   int SelectParm;                   /* Code for allocation method */
   int Target;                       /* Target allocated to Pivot */
   double Theta;                     /* Theta value for possible allocation */
   const CONFIG_AllocInfo *AllocInfo;      /* Address of allocation structure */
   const CONFIG_TargetInfo *TargetInfo;    /* Address of target information */
   const CONFIG_PivotInfo *PivotInfo;     /* Address of pivot information */
   int IPrio;                        /* Priority level to work on */
   int UnallocatedFibres;            /* Number of fibres unallocated */
   CONFIG_METHOD_MatrixElement **Matrix;
   CONFIG_METHOD_TargetElement *TargetMatrix;
   if (*Status != STATUS__OK) return;

   AllocInfo  = MethodInfo->AllocInfo;
   TargetInfo = MethodInfo->TargetInfo;
   PivotInfo  =  MethodInfo->PivotInfo;
   NPiv = CONFIG_METHOD_Global.NPiv;
   NTargets = CONFIG_METHOD_Global.NTargets;
   Matrix = CONFIG_METHOD_Global.Matrix;
   TargetMatrix = CONFIG_METHOD_Global.TargetMatrix;
   CandidateArray = (int*) NULL;
   
   /*  If there have been calls to ConfigMethodSetSky() to set fibre counts
    *  to be explicitly left for sky targets, then we disable sky targets
    *  for all the fibre types in question. We will then allocate to 
    *  non-sky targets, and towards the end of this routine we will
    *  re-enable the sky targets and explicitly choose fibres to be deallocated
    *  and reallocated to sky targets.
    */
   
   SkyControl = CONFIG_FALSE;
   for (IType = 0; IType < CONFIG_METHOD_Global.FibreSkyEntries; IType++) {
      if (CONFIG_METHOD_Global.FibreSkyCounts[IType] >= 0) {
         SkyControl = CONFIG_TRUE;
         break;
      }
   }
   if (SkyControl) ConfigMethodDisableSky(MethodInfo);
      
   /*  Work out the most allocations we can expect, given the state of the
    *  allocation matrix.
    */
    
   MaxAllocs = ConfigMethodMaxAlloc();

   /* ****GBD**** test which config method was selected to see if we want to do
    * the standard routines or not. It's easier to just write a different
    * subroutine tree from here instead of changing PickNext! ****GBD****
    */
    
   CancelFlag = CONFIG_FALSE;

   DeAllocSkyAtEnd = CONFIG_FALSE;
   DeallocCode = CONFIG_METHOD_Global.
     ParameterValues[CONFIG_METHOD_SELECT_PARM].IntegerValue;

   if (DeallocCode == CONFIG_METHOD_SEL_IMPORT 
                            || DeallocCode == CONFIG_METHOD_SEL_IMPORTONLY) {
     ConfigMethodImportAlloc(MethodInfo,ReportFunction,FunctionArg);
     return;
   }
   if (DeallocCode == CONFIG_METHOD_SEL_BROKEN){
     if (ConfigMethodDefineCones(MethodInfo,0,9)){
       ConfigMethodAutoDeallocate(MethodInfo,ReportFunction,FunctionArg,Status);
     }
     /* Might as well see if we can uncross here as well, since we may
      * have made a lot of reallocations...*/
     DeallocCode = CONFIG_METHOD_Global.
       ParameterValues[CONFIG_METHOD_UNCROSS_PARM].IntegerValue;
     if (DeallocCode == CONFIG_METHOD_UNCROSS_ATEND) {
       if (ConfigMethodDefineCones(MethodInfo,0,9)) {
         ConfigMethodUncrossFibres(MethodInfo,ReportFunction,FunctionArg);
         ConfigMethodConeAllocate(MethodInfo,ReportFunction,FunctionArg);
       }
     }
     return; /* Since we don't really want to do anything else here just yet */
   }
         

   /* see if we were asked to just uncross the existing configuration */
   DeallocCode = CONFIG_METHOD_Global.
     ParameterValues[CONFIG_METHOD_UNCROSS_PARM].IntegerValue;
   if (DeallocCode == CONFIG_METHOD_UNCROSS_ONLY ||
       DeallocCode == CONFIG_METHOD_UNCROSS_THEN_ALLOCATE) {
     if (ConfigMethodDefineCones(MethodInfo,0,9)) {
       ConfigMethodUncrossFibres(MethodInfo,ReportFunction,FunctionArg);
       if (DeallocCode == CONFIG_METHOD_UNCROSS_THEN_ALLOCATE){
            ConfigMethodConeAllocate(MethodInfo,ReportFunction,FunctionArg); 
                                                                  /* Extras */
       }
     }
     /*  Just to be on the safe side, we now call FieldCheck() to make
      *  sure the configuration we've produced is valid. */
     
     MethodInfo->FieldCheck(MethodInfo,ReportFunction,FunctionArg,1,Status);
     if (*Status != STATUS__OK) goto Exit;
     AllocCount = 0;
     for (IPiv = 0; IPiv < NPiv; IPiv++) {
       if (AllocInfo[IPiv].State == 'A') AllocCount++;
     }
     return;  /*Since only asked to uncross*/
   }
   
   /*  We are now at the point where a normal allocation of fibres is
    *  handled.  The 'Oxford' iterative algorithm is treated separately
    *  from the original 'Taylor' algorithm and its variants ('select
    *  hardest/closest/straightest fibre' etc.)  Note that in honour of
    *  its author, the code for the Oxford algorithm is CONFIG_METHOD_SEL_GBD.
    */

   UnallocatedFibres = 0;
   if (CONFIG_METHOD_Global.ParameterValues[CONFIG_METHOD_SELECT_PARM]
       .IntegerValue != CONFIG_METHOD_SEL_GBD) {

     
      /*  'Taylor' case. The basic allocation process is quite simple. We use 
       *  ConfigMethodPickNext() to select the next allocation to be made,
       *  based on the matrix of possible allocations of fibres to targets.
       *  We then record that allocation, using ConfigMethodSetAlloc(). The
       *  new allocation will have invalidated some of the possible allocations,
       *  and we call ConfigMethodNewAlloc() to update the matrix to reflect
       *  this. Then we repeat the process until there are no more allocations
       *  that can be made.
       */ 
   
     DeAllocSkyAtEnd = CONFIG_TRUE;
     AllocCount = 0;
     while (ConfigMethodPickNext(MethodInfo,&IPiv,&ITarget,&Theta)) {
       ConfigMethodSetAlloc(MethodInfo,IPiv,ITarget,Theta,Status);
       if (*Status != STATUS__OK) break;
       ConfigMethodNewAlloc(MethodInfo,IPiv,ITarget,Theta);
       if (ReportFunction != NULL) {
         AllocCount++;
         PercentDone = (float) AllocCount / (float) NPiv * 100.0; 
         if ((*ReportFunction)(FunctionArg,'A',
             "Allocating fibres",IPiv,(TargetInfo[ITarget].Type == 'F'),
              TargetInfo[ITarget].OriginalIndex,
              TargetInfo[ITarget].X,TargetInfo[ITarget].Y,Theta,PercentDone)) {
            CancelFlag = CONFIG_TRUE;
            goto Exit;
         }
       }
     }
   } else {

      /*  This is the 'Oxford' algorithm. The big problem with this code is
       *  that only one person understands it, and that understanding hasn't
       *  yet been converted into code comments! Fixing this is an ongoing
       *  exercise.
       */

     DeallocCode = CONFIG_METHOD_Global.
       ParameterValues[CONFIG_METHOD_UNCROSS_PARM].IntegerValue;
     Theta = 0.;
     PercentDone=0.;
     for (IPrio = 9; IPrio > 0 ; IPrio--) {
       if ((*ReportFunction)(FunctionArg,'P',"Generating Cone Tree",
                                              0,0,0,0,0,Theta,PercentDone)) {
         CancelFlag = CONFIG_TRUE;
         goto Exit;
       }
       if (ConfigMethodDefineCones(MethodInfo,IPrio,9)) {
         ConfigMethodConeAllocate(MethodInfo,ReportFunction,FunctionArg);
         if (!CONFIG_METHOD_Global.SkyFlag){
           /*  Removed KS 23/8/02. A final swap phase seems to work properly
            *  now and is faster.
           ConfigMethodBlockSwapAllocate(MethodInfo,ReportFunction,
                                                          FunctionArg);
           */
         }
         PercentDone+=5;
         if ((DeallocCode == CONFIG_METHOD_UNCROSS_EACH)){
           if ((*ReportFunction)(FunctionArg,'P',"Generating Cone Tree",
                                              0,0,0,0,0,Theta,PercentDone)) {
             CancelFlag = CONFIG_TRUE;
             goto Exit;
           }
           if (!CONFIG_METHOD_Global.SkyFlag){
             ConfigMethodDefineCones(MethodInfo,0,9);
             ConfigMethodUncrossFibres(MethodInfo,ReportFunction,FunctionArg);
           }
         }
       }
       PercentDone=10.*(10.-IPrio);
     }
     
     /*  This is a try to see if one final swap pass can be as good as one
      *  each priority level. KS 23/8/02  The define cones call isn't strictly
      *  needed, since it merely duplicates the last such call made as we 
      *  go through the priority levels. However, I've put it in mainly as
      *  as reminder that it's a necessary repliminary to a call to 
      *  ConfigMethodBlockSwapAllocate(), should we try to rejig the code to
      *  make the swap phase a general purpose option for different algorithms.
      */
      
     if (!CONFIG_METHOD_Global.SkyFlag){
        if (ConfigMethodDefineCones(MethodInfo,0,9)) {
            ConfigMethodBlockSwapAllocate(MethodInfo,ReportFunction,
                                                            FunctionArg);
        }
     }
     /* For Now just try a single uncrossing pass */
     if ((*ReportFunction)(FunctionArg,'P',"Generating Cone Tree",
                                                0,0,0,0,0,Theta,PercentDone)) {
       CancelFlag = CONFIG_TRUE;
       goto Exit;
     }
     if (!CONFIG_METHOD_Global.SkyFlag){
       if (ConfigMethodDefineCones(MethodInfo,0,9)) {
         if (CONFIG_METHOD_Global.
            ParameterValues[CONFIG_METHOD_UNCROSS_PARM].IntegerValue 
                                               != CONFIG_METHOD_UNCROSS_NEVER)
            ConfigMethodUncrossFibres(MethodInfo,ReportFunction,FunctionArg);
       } 
     }
     
     /*  The 'Oxford' algorithm does not keep the allocation matrix updated,
      *  so if any subsequent stages are to make use of it, this needs to
      *  to corrected now.
      */
   
     ConfigMethodUpdateMatrix (MethodInfo,ReportFunction,FunctionArg,Status);
   }

   /*  Just to be on the safe side, we now call FieldCheck() to make
    *  sure the configuration we've produced is valid.
    */

   /* MethodInfo->FieldCheck(MethodInfo,ReportFunction,FunctionArg,1,Status); */
   if (*Status != STATUS__OK) goto Exit;
   AllocCount = 0;
   for (IPiv = 0; IPiv < NPiv; IPiv++) {
      if (AllocInfo[IPiv].State == 'A') AllocCount++;
   }

   /*  Now, the question is whether we ought to attempt to improve on
    *  this configuration by trying a tentative deallocation sequence. Here,
    *  we deallocate each pivot in turn and then, with the freedom that
    *  gives us, we again allocate fibres until there are no more we can
    *  allocate. We can't loose here - when we deallocate a fibre there will
    *  always be at least one allocation we can make (the one we just
    *  deallocated) and often that's the one that does get made first and 
    *  we're back where we started. However, sometimes some other allocation 
    *  is made, and occasionally we find we can make more than one allocation,
    *  in which case we've gained. This is very much a 'final tweak' option -
    *  it takes a while and usually won't allocate more than the odd one or 
    *  two additional fibres.
    *
    *  It may be that we already have all the targets or pivots allocated.
    *  Only if there are both unallocated pivots and unallocated targets
    *  is it usually worth trying this pass, although we allow the option
    *  of trying it anyway even then, because the different allocations it
    *  produces may have other advantages. (Better radial distribution, for
    *  example.)
    */

   DeallocCode = CONFIG_METHOD_Global.
              ParameterValues[CONFIG_METHOD_DEALLOC_PARM].IntegerValue;
   if ((DeallocCode == CONFIG_METHOD_DEALLOC_ALWAYS) || 
                 ((DeallocCode == CONFIG_METHOD_DEALLOC_ONLY) && 
                                          (AllocCount < MaxAllocs))) {
                                          
      /*  The tentative deallocation code was originally designed to work
       *  only in the Taylor algorithm case. There is an explicit test for
       *  the algorithm used in ConfigMethodDealloc() which means that the
       *  deallocation is treated as tentative if the main algorithm
       *  used is the Oxford algorithm. We defeat that here, but in a 
       *  very inelegant way. Ideally ConfigMethodDealloc() should have a
       *  'tentative' argment, but it is part of an externally defined
       *  interface, so that's tricky.
       */
      
      SelectParm = CONFIG_METHOD_Global.
                       ParameterValues[CONFIG_METHOD_SELECT_PARM].IntegerValue;
      if (SelectParm == CONFIG_METHOD_SEL_GBD) {
         CONFIG_METHOD_Global.
           ParameterValues[CONFIG_METHOD_SELECT_PARM].IntegerValue =
                                                      CONFIG_METHOD_SEL_HARD;
      }  
       
      /*  This is the tentative deallocation loop. Most of the code here
       *  is taken up with the calls to the report function. The actual
       *  process is very simple. For each pivot, unless it is one that
       *  was pre-allocated (we assume we shouldn't touch them), we call
       *  ConfigMethodDealloc() to deallocate it and update the possible
       *  path matrix accordingly. We then go back to the normal allocation
       *  loop, calling ConfigMethodPickNext(), ConfigMethodSetAlloc() and
       *  ConfigMethodNewAlloc() until there are no new allocations possible.
       *  Note that if we get a 'cancel' request, we don't actually cancel
       *  with fibres deallocated and not replaced - we defer the actual
       *  cancellation until the processing for the deallocated fibre is
       *  complete.
       */
       
      for (Pivot = 0; Pivot < NPiv; Pivot++) {
         if (AllocInfo[Pivot].State == 'A') {
            Target = AllocInfo[Pivot].TargetIndex;
            if ((!TargetInfo[Target].PreAllocated) && 
                                  (TargetInfo[Target].Type != 'F')) {
               ConfigMethodDealloc (MethodInfo,Pivot,Target,NULL,0,Status);
               PercentDone = (float) Pivot / (float) NPiv * 100.0; 
               if (ReportFunction != NULL) {
                  if ((*ReportFunction)(FunctionArg,'D',
                           "Re-allocating fibres",Pivot,
                              (TargetInfo[Target].Type == 'F'),
                                TargetInfo[Target].OriginalIndex,
                                  TargetInfo[Target].X,TargetInfo[Target].Y,
                                     AllocInfo[Pivot].Theta,PercentDone)) {
                      CancelFlag = CONFIG_TRUE;                    
                  }
               }
               while (ConfigMethodPickNext(
                                      MethodInfo,&IPiv,&ITarget,&Theta)) {
                  ConfigMethodSetAlloc(MethodInfo,IPiv,ITarget,Theta,Status);
                  if (*Status != STATUS__OK) break;
                  ConfigMethodNewAlloc(MethodInfo,IPiv,ITarget,Theta);
                  if (ReportFunction != NULL) {
                     if ((*ReportFunction)(FunctionArg,'A',
                           "Re-allocating fibres",IPiv,
                             (TargetInfo[ITarget].Type == 'F'),
                                TargetInfo[ITarget].OriginalIndex,
                                   TargetInfo[ITarget].X,TargetInfo[ITarget].Y,
                                     Theta,PercentDone)) {
                        CancelFlag = CONFIG_TRUE;
                     }
                  }
               }
            }
         }
         if (CancelFlag) goto Exit;
      }
      AllocCount = 0;
      for (IPiv = 0; IPiv < NPiv; IPiv++) {
         if (AllocInfo[IPiv].State == 'A') AllocCount++;
      }
      
      /*  Re-instate the previous global value for the selection algorithm */
      
      CONFIG_METHOD_Global.
         ParameterValues[CONFIG_METHOD_SELECT_PARM].IntegerValue = SelectParm;
   }

   /*  The Oxford algorithm has built in to it the code required to leave a
    *  specified number of fibres for sky targets. The Taylor algorithm does
    *  not, so in this case we need to do a final pass de-allocating a
    *  number of fibres for use as sky fibres. We start with the lowest
    *  priority fibres and keep going until we've deallocated enough.
    */
   
   if (DeAllocSkyAtEnd) {
      /*  We need an array to record the candidate pivot numbers. */
      
      int* CandidateArray = (int*) malloc (NPiv * sizeof(int));
      if (CandidateArray == (int*) NULL) {
         char s[100];
         sprintf(s,
            "Unable to allocate memory for candidate pivot array of %ld bytes",
                                                  (long)(NPiv * sizeof(int)));
         MethodInfo->ErrRepRoutine(0,s);
         *Status = CONFIG__NO_MEMORY;
         goto Exit;
      }
   
      /*  First we see how many unallocated fibres we have. Note that we
       *  don't count guide fibres. We count those that are shown as
       *  unallocated, as able to have their allocation changed, as not
       *  broken and as non-guide fibres.
       */
       
      UnallocatedFibres = 0;
      for (IPiv = 0 ; IPiv < NPiv ; IPiv++) {
         if ((!MethodInfo->FibreIsBroken(MethodInfo,PivotInfo[IPiv].FibreType))
            && (!MethodInfo->FibreIsGuide(MethodInfo,PivotInfo[IPiv].FibreType))
               && (PivotInfo[IPiv].AllowAllocChange)
                  && (AllocInfo[IPiv].State == 'U')) {
            UnallocatedFibres++;
         }
      }
      
      /*  Now we work through each priority band of targets, deallocating
       *  until we have sufficient fibres free. If deallocating all the
       *  fibres at a given priority is insufficient, then we obviously
       *  deallocate the lot and move on to the next level. When we find
       *  that we only need to delete some at a given level, then we have
       *  to try to select them in an unbiassed way - taking the first N
       *  starting at pivot number 0 and going up will just deallocate a
       *  block of pivots in the same area of the plate. So we build up an
       *  array of candidate pivots for deallocation, jumble them up, and
       *  then deallocate as many as are needed.
       */
       
      for (IPrio = 0; IPrio < 10; IPrio++) {
         if (UnallocatedFibres > CONFIG_METHOD_Global.FibresForSky) break;
         
         /*  See which pivots at this priority level are candidates for
          *  deallocation. We want only allocated, non-guide, pivots at the
          *  correct priority level that we are allowed to change.
          */
          
         CandidatePivots = 0;
         for (Pivot = 0; Pivot < NPiv; Pivot++) {
            Candidate = (AllocInfo[Pivot].State == 'A');
            if (Candidate) {
               Target = AllocInfo[Pivot].TargetIndex;
               if (TargetInfo[Target].Priority != IPrio) {
                  Candidate = CONFIG_FALSE;
               }
               if (!PivotInfo[Pivot].AllowAllocChange) Candidate = CONFIG_FALSE;
               if (TargetInfo[Target].PreAllocated) Candidate = CONFIG_FALSE;
               if (MethodInfo->FibreIsGuide(
                                  MethodInfo,PivotInfo[Pivot].FibreType)) {
                  Candidate = CONFIG_FALSE;
               }
            }
            if (Candidate) {
               CandidateArray[CandidatePivots] = Pivot;
               CandidatePivots++;
            }
         }
         
         /*  Now we jumble up the list of candidate pivots and deallocate
          *  until we have enough spare fibres. If we have fewer candidates
          *  than we need we don't need to jumble them up since we'll end
          *  up deallocating all of them anyway, but it doesn't matter. 
          */
          
         ConfigMethodRandomSort(CandidateArray,CandidatePivots);
         for (ICand = 0; ICand < CandidatePivots; ICand++) {
            if (UnallocatedFibres >= CONFIG_METHOD_Global.FibresForSky) break;
            Pivot = CandidateArray[ICand];
            Target = AllocInfo[Pivot].TargetIndex;
            ConfigMethodDealloc (MethodInfo,Pivot,Target,NULL,0,Status);
            UnallocatedFibres++;
            PercentDone = (float) UnallocatedFibres / 
                     (float) CONFIG_METHOD_Global.FibresForSky * 100.0; 
            if (ReportFunction != NULL) {
               if ((*ReportFunction)(FunctionArg,'D',
                           "Deallocating fibres for sky",Pivot,
                              (TargetInfo[Target].Type == 'F'),
                                TargetInfo[Target].OriginalIndex,
                                  TargetInfo[Target].X,TargetInfo[Target].Y,
                                     AllocInfo[Pivot].Theta,PercentDone)) {
                   CancelFlag = CONFIG_TRUE;
                   break;                    
               }
            }
         } 
      }
   }
   
   /*  If we disabled sky targets earlier in order to provide a more explicit
    *  control over the allocation to sky targets, this is the point where
    *  we re-enable the sky targets. We then use ConfigMethodReassignToSky()
    *  to choose the right number of pivots to deallocate and (if sky
    *  targets are available) to re-allocate them to sky targets. We do this
    *  differently for FLAMES and 2dF/6dF. FLAMES has a number of different
    *  fibre types with different sky requirements. 2dF merely has a
    *  requirement that the total number of sky targets be divided evenly
    *  between the non-guide fibre types. The trick is to see if all types
    *  have the same sky count. 6dF only has one fibre type, so we treat this
    *  the same as FLAMES and do it all in one hit.
    */
   
   if (SkyControl) {
   
      short Even = CONFIG_TRUE;
      
      ConfigMethodReEnable(MethodInfo);
      if (CONFIG_METHOD_Global.FibreSkyEntries == 1) {
         Even = CONFIG_FALSE;
      } else {
         for (IType = 1; IType < CONFIG_METHOD_Global.FibreSkyEntries; 
                                                                IType++) {
            if (CONFIG_METHOD_Global.FibreSkyCounts[IType - 1] !=
                            CONFIG_METHOD_Global.FibreSkyCounts[IType]) {
               Even = CONFIG_FALSE;
               break;
            }
         }
      }
      
      if (!Even) {
      
         /*  This is the normal FLAMES case, where we do each fibre type in
          *  turn, going first to the types that need the fewest skies.
          */
          
         for (IType = 0; IType < CONFIG_METHOD_Global.FibreSkyEntries; 
                                                                   IType++) {
            int TypeIndex = ConfigMethodNthLowest(
                    CONFIG_METHOD_Global.FibreSkyCounts,
                    CONFIG_METHOD_Global.FibreSkyEntries,IType + 1);
            if (CONFIG_METHOD_Global.FibreSkyCounts[TypeIndex] > 0) {
               ConfigMethodReassignToSky(MethodInfo,
                      CONFIG_METHOD_Global.FibreTypesForSky[TypeIndex],
                        CONFIG_METHOD_Global.FibreSkyCounts[TypeIndex],
                          CONFIG_METHOD_Global.FibreSkyCounts[TypeIndex],
                                             ReportFunction,FunctionArg);
            }
         }
         
      } else {
      
         /*  This is the 2dF/6dF case, where we loop round giving one
          *  sky to each type in turn. Note, that this is slower, but in the
          *  case where there aren't enough skies to go around, it splits
          *  the available skies more evenly.
          */
      
         int ISky;
         for (ISky = 0; ISky < CONFIG_METHOD_Global.FibreSkyCounts[0]; ISky++) {
            for (IType = 0; IType < CONFIG_METHOD_Global.FibreSkyEntries; 
                                                                   IType++) {
               ConfigMethodReassignToSky(MethodInfo,
                     CONFIG_METHOD_Global.FibreTypesForSky[IType],
                        CONFIG_METHOD_Global.FibreSkyCounts[IType],1,
                                              ReportFunction,FunctionArg);
            }
         }
      }
            
   }
               
   /* Now do the decrossing pass if we were asked to run it only
    * at the end of the show */

   DeallocCode = CONFIG_METHOD_Global.
              ParameterValues[CONFIG_METHOD_UNCROSS_PARM].IntegerValue;
   if ((DeallocCode == CONFIG_METHOD_UNCROSS_ATEND)){
     if (!CONFIG_METHOD_Global.SkyFlag){
       ConfigMethodDefineCones(MethodInfo,0,9);
       ConfigMethodUncrossFibres(MethodInfo,ReportFunction,FunctionArg);
     }
     AllocCount = 0;
     for (IPiv = 0; IPiv < NPiv; IPiv++) {
       if (AllocInfo[IPiv].State == 'A') AllocCount++;
     }
   }
  
   /*  And at the end, we do a full field check to make sure the configuration
    *  is valid.
    */
    
   MethodInfo->FieldCheck(MethodInfo,ReportFunction,FunctionArg,1,Status);
   
Exit:;
   
   if (CandidateArray) free (CandidateArray);
   if (CancelFlag) {
       MethodInfo->InfoRoutine(MethodInfo->InfoArgument,
                               "Cancel request received");
      *Status = CONFIG__CANCELLED;
   }
   
}

 
/* -------------------------------------------------------------------------- */

/*              C o n f i g  M e t h o d  N e w  A l l o c
 *
 *  Function:
 *     Revises the allocation matrix in the light of a new allocation.
 *
 *  Description:
 *     This routine is called whenever a new allocation is made to update
 *     the set of possible allocations maintained in the matrix of all
 *     possible paths between pivots and targets. When a new fibre is 
 *     allocated, this may invalidate some of the possible allocations,
 *     since the fibre path represented by a potential allocation may now
 *     conflict with the path represented by the newly made allocation.
 *
 *     NOTE, CURRENTLY, MOST COMMENTS REFER TO THE CASE WHERE THE
 *     CHECK_FPIL MACRO IS DEFINED.  THEY MUST BE MODIFIED FOR
 *     USE OF MethodInfo->Collision().    
 *
 *
 *  Note:
 *     This routine is almost identical to ConfigCheckAllocation(), and it
 *     should really be possible to extract most of the checking code from
 *     both into one utility routine that both could call. However, it is
 *     harder to see how to do this efficiently - bear in mind that this
 *     is one of the critical parts of the allocation code, and will be
 *     slowed noticeably by having to collect all the information it has
 *     accumulated about the new pivot and pass it to some utility routine.
 *     Although there is a lot of lines of code here, most of them will
 *     execute very quickly. Still, some experimentation could usefully be
 *     done here.
 */

void ConfigMethodNewAlloc (
   const CONFIG_MethodInfo *MethodInfo,
   int Pivot,                     /* (>) The number of the allocated pivot */
   int Target,                    /* (>) The number of the allocated target */
   double Theta)                  /* (>) The twist angle for the button */
{

   /*  A note on variable names.  A new allocation of a fibre to a pivot has 
    *  been made.  So there is now a new, unalterable, position for a button
    *  and there is a button running from the newly allocated pivot point to
    *  the virtual pivot point of that fibre. All the variables describing
    *  this new allocation contain 'New'. All the variables describing 
    *  the potential allocations represented by the matrix do not have 'New'
    *  but are otherwise the same names. In many cases, these qualtities are
    *  coordinates in microns, and as such are held as integers. However, we
    *  sometimes need these values in floating point as well, and in these
    *  cases we prefix the name with a 'D'.
    */ 
 
   /*  Local variables  */
   int Collision;   /* True if there is a collision */


#ifdef CHECK_FPIL
   long ButtonX1;              /* X-start of safe rectangle around old target */
   long ButtonX2;              /* X-end of safe rectangle around old target */
   long ButtonY1;              /* Y-start of safe rectangle around old target */
   long ButtonY2;              /* Y-end of safe rectangle around old target */
   CONFIG_Boolean CheckCollision;   /* True if there is a collision */
   double DFibrePivotX;        /* Virtual pivot position in X for new fibre */
   double DFibrePivotY;        /* Virtual pivot position in Y for new fibre */
   double DNewFibrePivotX;     /* Virtual pivot position in X for fibre */
   double DNewFibrePivotY;     /* Virtual pivot position in Y for fibre */
   double DNewPivotX;          /* Floating point value of NewPivotX */
   double DNewPivotY;          /* Floating point value of NewPivotY */
   double DNewTargetX;         /* Floating point value of NewTargetX */
   double DNewTargetY;         /* Floating point value of NewTargetY */
   double DPivotX;             /* Floating point value of PivotX */
   double DPivotY;             /* Floating point value of PivotY */
   double DTargetX;            /* Floating point value of TargetX */
   double DTargetY;            /* Floating point value of TargetY */
   long FibreX1;               /* X-start of safe rectangle around old fibre */
   long FibreX2;               /* X-end of safe rectangle around old fibre */
   long FibreY1;               /* Y-start of safe rectangle around old fibre */
   long FibreY2;               /* Y-end of safe rectangle around old fibre */
   long NewButtonX1;           /* X-start of safe rectangle around new target */
   long NewButtonX2;           /* X-end of safe rectangle around new target */
   long NewButtonY1;           /* Y-start of safe rectangle around new target */
   long NewButtonY2;           /* Y-end of safe rectangle around new target */
   long NewFibreX1;            /* X-start of safe rectangle around fibre posn */
   long NewFibreX2;            /* X-end of safe rectangle around fibre posn */
   long NewFibreY1;            /* Y-start of safe rectangle around fibre posn */
   long NewFibreY2;            /* Y-end of safe rectangle around fibre posn */
#endif
   int IPiv;                   /* Loop index through pivots */
   int ITarget;                /* Loop index through targets */
   CONFIG_METHOD_MatrixElement **Matrix;
                               /* Local synonym for .Matrix in globals */
   int NPiv;                   /* Number of pivots */

   long NewTargetX;            /* X-coordinate of newly allocated target */
   long NewTargetY;            /* Y-coordinate of newly allocated target */
   long NewPivotX;             /* X-coordinate of newly allocated pivot */
   long NewPivotY;             /* Y-coordinate of newly allocated pivot */
   int NTargets;               /* Number of possible targets */
   const CONFIG_PivotInfo *PivotInfo;/* Pivot info array of structures */
   long PivotX;                /* X-coordinate of previously allocated pivot */
   long PivotY;                /* Y-coordinate of previously allocated pivot */
   int  fibreType;             /* fibre type of previously allocated pivot */
#ifdef CHECK_FPIL
   CONFIG_Boolean Safe;        /* Indicates rects have no overlap regions */
   long SafeClearance;         /* Safe clearance (microns) about button */
#endif
   const CONFIG_TargetInfo *TargetInfo; /* Target info array of structures */
   long TargetX;               /* X-coordinate of previously allocated target */
   long TargetY;               /* Y-coordinate of previously allocated target */

   /*  Note that this code could be simplified enormously if we were game
    *  just to make one call to tdFcollisionButBut() and two calls to
    *  tdFcollisionButFib() for each of the possible allocations. However,
    *  these are very precise, careful, slow routines, and we want to avoid
    *  using them if possible, simply because we expect this routine to be
    *  called a great many times. Hence the business with 'safe' rectangles,
    *  and the huge number of variables needed to describe them.
    */

   /*  Useful constants. We need to know the maximum offset of a button
    *  from the target point, so we know, crudely, how much clearance we
    *  need to be really safe when looking at overlapping rectangles later
    *  in the code. We get this from the housekeeping routines rather than
    *  from the dimensions .h file, to allow larger clearances to be 
    *  experimented with easily.
    */
#  ifdef CHECK_FPIL
   SafeClearance = MethodInfo->SafeClearance(MethodInfo);
#  endif
   /*  Pick up local pointers to the various information structures whose
    *  addresses are held in the method-specific global structure. Also
    *  pick up the number of possible targets and pivots.
    */

   Matrix = CONFIG_METHOD_Global.Matrix;
   NTargets = CONFIG_METHOD_Global.NTargets;
   NPiv = CONFIG_METHOD_Global.NPiv;
   PivotInfo = MethodInfo->PivotInfo;
   TargetInfo = MethodInfo->TargetInfo;

   /*  Calculate the rectangles surrounding the newly allocated fibre and
    *  its button. The NewButtonxxx values give the rectangle that safely
    *  clears the newly allocated button position.  The NewFibrexxx values
    *  give the rectangle that safely clears the whole new allocated fibre.
    */

   NewTargetX = TargetInfo[Target].X;
   NewTargetY = TargetInfo[Target].Y;
#ifdef CHECK_FPIL
   DNewTargetX = (double) NewTargetX;
   DNewTargetY = (double) NewTargetY;
   NewButtonX1 = NewTargetX - SafeClearance;
   NewButtonX2 = NewTargetX + SafeClearance;
   NewButtonY1 = NewTargetY - SafeClearance;
   NewButtonY2 = NewTargetY + SafeClearance;
#endif
   NewPivotX = PivotInfo[Pivot].X;
   NewPivotY = PivotInfo[Pivot].Y;
#ifdef CHECK_FPIL
   DNewPivotX = (double) NewPivotX;
   DNewPivotY = (double) NewPivotY;
   if (NewPivotX > NewTargetX) {
      NewFibreX1 = NewTargetX - SafeClearance;
      NewFibreX2 = NewPivotX + SafeClearance;
   } else {
      NewFibreX1 = NewPivotX - SafeClearance;
      NewFibreX2 = NewTargetX + SafeClearance;
   }
   if (NewPivotY > NewTargetY) {
      NewFibreY1 = NewTargetY - SafeClearance;
      NewFibreY2 = NewPivotY + SafeClearance;
   } else {
      NewFibreY1 = NewPivotY - SafeClearance;
      NewFibreY2 = NewTargetY + SafeClearance;
   }
#endif

   /*  We we also need to know the position of the virtual pivot point
    *  of the newly allocated button, since that is where the fibre is
    *  regarded as joining the button.
    */

#ifdef CHECK_FPIL
   MethodInfo->GetVirtPiv (MethodInfo,DNewTargetX,DNewTargetY,Theta,
                           &DNewFibrePivotX,
                           &DNewFibrePivotY);
#endif

   /*  We need to pass through every element of the matrix. In every case
    *  where an allocation is flagged as being possible, we have to consider
    *  whether this new allocation has invalidated it.
    *
    *  Note: I believe this routine doesn't need to consider the case where
    *  a fibre doesn't really exist. Non-existent fibres can't be allocated,
    *  so no allocation that has a CONFIG_METHOD_POSSIBLE code in the matrix
    *  can involve a non-existent fibre. 
    */

   for (IPiv = 0; IPiv < NPiv; IPiv++) {
      PivotX = PivotInfo[IPiv].X;
      PivotY = PivotInfo[IPiv].Y;
      fibreType = PivotInfo[IPiv].FibreType;
#     ifdef CHECK_FPIL
      DPivotX = (double) PivotX;
      DPivotY = (double) PivotY;
#     endif
      for (ITarget = 0; ITarget < NTargets; ITarget++) {
         if (Matrix[IPiv][ITarget].PathStatus == CONFIG_METHOD_POSSIBLE) {

            /*  This allocation is flagged as possible. We need to check
             *  1) whether there is a collision between the newly allocated
             *  button and the potential button position, 2) whether there is
             *  a collision between the newly allocated button and the 
             *  the potential fibre path, 3) whether there is a collision
             *  between the newly allocated fibre path and the potential
             *  button position. The standard routines for doing the formal
             *  checks for these are quite time-consuming, so we try to do 
             *  some quick and dirty checks first to see if we need to call
             *  them.  We crudely define 'rectangles of interest' around
             *  the two button positions and the two fibres - these are just
             *  the smallest rectangles in the standard coordinate system 
             *  that fully enclose (and allow for clearance) each button and
             *  each fibre. Only if these overlap do we need to call the
             *  more detailed check routines.
             */

            Collision = CONFIG_FALSE;
#ifdef      CHECK_FPIL
            CheckCollision = CONFIG_FALSE;
#endif

            TargetX = TargetInfo[ITarget].X;
            TargetY = TargetInfo[ITarget].Y;

#ifdef      CHECK_FPIL
            DTargetX = (double) TargetX;
            DTargetY = (double) TargetY;


            ButtonX1 = TargetX - SafeClearance;
            ButtonX2 = TargetX + SafeClearance;
            ButtonY1 = TargetY - SafeClearance;
            ButtonY2 = TargetY + SafeClearance;
            
            /*  So, do we need to call the button-button collision routine?  */

            Safe = CONFIG_FALSE;
            if (ButtonX2 < NewButtonX1) {
               Safe = CONFIG_TRUE;
            } else if (NewButtonX2 < ButtonX1) {
               Safe = CONFIG_TRUE;
            } else if (ButtonY2 < NewButtonY1) {
               Safe = CONFIG_TRUE;
            } else if (NewButtonY2 < ButtonY1) {
               Safe = CONFIG_TRUE;
            }

            if (!Safe) {
               if (MethodInfo->ColButBut (MethodInfo, DTargetX, DTargetY, 
                        Matrix[IPiv][ITarget].Theta, DNewTargetX, DNewTargetY,
                                                        Theta)) {
                  CheckCollision = CONFIG_TRUE;
               }
            }

            /*  Now, what about the possibility of the newly allocated fibre
             *  colliding with the potentially allocated button? The potentially
             *  allocated button is at (TargetX,TargetY), and its 'safe'
             *  rectangle is delimited by (ButtonX1,ButtonX2,ButtonY1,ButtonY2).
             *  The fibre leading to the newly allocated button goes
             *  from its pivot (NewPivotX,NewPivotY) to the virtual pivot
             *  point of its button (NewFibrePivotX,NewFibrePivotY). Its 'safe'
             *  rectangle is given by (NewFibreX1,NewFibreX2,NewFibreY1,
             *  NewFibreY2).
             */

            if (!CheckCollision) {
               Safe = CONFIG_FALSE;
               if (ButtonX2 < NewFibreX1) {
                  Safe = CONFIG_TRUE;
               } else if (NewFibreX2 < ButtonX1) {
                  Safe = CONFIG_TRUE;
               } else if (ButtonY2 < NewFibreY1) {
                  Safe = CONFIG_TRUE;
               } else if (NewFibreY2 < ButtonY1) {
                  Safe = CONFIG_TRUE;
               }

               if (!Safe) {
                  if (MethodInfo->ColButFib (MethodInfo,DTargetX, DTargetY, 
                          Matrix[IPiv][ITarget].Theta,DNewFibrePivotX,
                             DNewFibrePivotY,DNewPivotX,DNewPivotY)) {
                     CheckCollision = CONFIG_TRUE;
                  }
               }
            }

            /*  And, finally, what about the possibility of the potentially
             *  allocated fibre colliding with the newly allocated button?
             *  The newly allocated button is at (NewTargetX,NewTargetY), and
             *  its 'safe' rectangle is given by (NewButtonX1,NewButtonX2,
             *  NewButtonY1,NewButtonY2).  The fibre leading to the potentially
             *  allocated button goes from its pivot (PivotX,PivotY) to the
             *  virtual pivot point of its button, which we still have to
             *  calculate as (FibrePivotX,FibrePivotY). The 'safe' rectangle
             *  for the fibre is (FibreX1,FibreX2,FibreY1,FibreY2), which we
             *  also still have to calculate.
             */

            if (!CheckCollision) {

               if (PivotX > TargetX) {
                  FibreX1 = TargetX - SafeClearance;
                  FibreX2 = PivotX + SafeClearance;
               } else {
                  FibreX1 = PivotX - SafeClearance;
                  FibreX2 = TargetX + SafeClearance;
               }
               if (PivotY > TargetY) {
                  FibreY1 = TargetY - SafeClearance;
                  FibreY2 = PivotY + SafeClearance;
               } else {
                  FibreY1 = PivotY - SafeClearance;
                  FibreY2 = TargetY + SafeClearance;
               }

               Safe = CONFIG_FALSE;
               if (NewButtonX2 < FibreX1) {
                  Safe = CONFIG_TRUE;
               } else if (FibreX2 < NewButtonX1) {
                  Safe = CONFIG_TRUE;
               } else if (NewButtonY2 < FibreY1) {
                  Safe = CONFIG_TRUE;
               } else if (FibreY2 < NewButtonY1) {
                  Safe = CONFIG_TRUE;
               }

               if (!Safe) {
                  MethodInfo->GetVirtPiv (MethodInfo,DTargetX,DTargetY,
                        Matrix[IPiv][ITarget].Theta,&DFibrePivotX,
                                                         &DFibrePivotY);

                  if (MethodInfo->ColButFib (MethodInfo,
                                             DNewTargetX, DNewTargetY, Theta, 
                                             DFibrePivotX,DFibrePivotY,
                                             DPivotX,DPivotY)) {
                     CheckCollision = CONFIG_TRUE;
                  }
               }
            }
#           endif

            Collision = MethodInfo->Collision(MethodInfo, 
                                      NewTargetX, NewTargetY,
                                      Theta,
                                      NewPivotX, NewPivotY,
                                      TargetX, TargetY, 
                                      Matrix[IPiv][ITarget].Theta, 
                                      PivotX, PivotY);

#           ifdef CHECK_FPIL
            if ((CheckCollision)&&(!Collision))
            {
                fprintf(stderr,
               "ConfigMethodCheckNewAlloc:CHECK_FPIL failure, %d %d %d %d\n",
                        (int)CheckCollision, (int)Collision, Pivot, ITarget);
                Collision = 1;
            }
#           endif

            if (Collision) {
               Matrix[IPiv][ITarget].PathStatus = CONFIG_METHOD_COLLISION;
               Matrix[IPiv][ITarget].CollidingPivot = Pivot;
            }

            if (!Collision) {
              if (MethodInfo->ColInvPos(MethodInfo,fibreType,
	      				NewTargetX,NewTargetY,Theta)) {
                Matrix[IPiv][ITarget].PathStatus = CONFIG_METHOD_SCREWED;
                Collision = CONFIG_TRUE;
              }
            }
              

         }
      }
   }

}


/* -------------------------------------------------------------------------- */

/*              C o n f i g  M e t h o d  D e a l l o c
 *
 *  Function:
 *     Revises the possible path matrix in the light of a deallocation.
 *
 *  Description:
 *     This routine is called whenever a deallocation is made to update
 *     the set of possible allocations maintained in the matrix of all
 *     possible paths between pivots and targets. When an allocation is
 *     cancelled, this may make valid some of the allocation paths, previously
 *     flagged as impossible because of collision with the now deallocated
 *     fibre.  This is a routine required by the interface between the 
 *     'handshaking and external interface' layer and the method-specific
 *     code, but it happens to be one that is also used directly by the
 *     method-specific code (in the 'trial deallocation' procedure).
 */

void ConfigMethodDealloc (
   const CONFIG_MethodInfo *MethodInfo,
   int Pivot,                     /* (>) The number of the deallocated pivot */
   int Target,                    /* (>) Target allocated to this pivot */
   CONFIG_ReportFunction
            ReportFunction,       /* (>) Feedback routine to call */
   void *FunctionArg,             /* (>) User-supplied argument */
   StatusType *Status)            /* (!) Inherited status */
{

   /*  Local variables  */

   int CollidingPivot;           /* Pivot causing collision - ignored */
   int Field;                    /* The field being configured */
   int ITarget;                  /* Loop index through targets */
   int IPiv;                     /* Loop index through pivots */
   CONFIG_METHOD_MatrixElement **Matrix;
                                 /* Local synonym for .Matrix in globals */
   int NPiv;                     /* Number of pivots */
   int NTargets;                 /* Number of targets */
   double PercentDone;           /* Percentage of current task completed */
   int ReportPivs;               /* Report after processing this many */

   if (*Status != STATUS__OK) return;

   /*  Pick up local pointers to the various information structures whose
    *  addresses are held in the method-specific global structure. Also
    *  pick up the number of possible targets and pivots.
    */

   Matrix = CONFIG_METHOD_Global.Matrix;
   NTargets = CONFIG_METHOD_Global.NTargets;
   NPiv = CONFIG_METHOD_Global.NPiv;
   Field = CONFIG_METHOD_Global.Field;

   /*  For the purposes of reporting progress, we pick a suitable number
    *  of pivots. Here we report every 10% of changes, since this should
    *  be a pretty rapid operation.
    */

   ReportPivs = (int) ((float) NPiv *0.1);
   if (ReportPivs < 0) ReportPivs = 1; 

   /* Do a little housekeeping of our own here */
   ITarget = MethodInfo->AllocInfo[Pivot].TargetIndex;
   CONFIG_METHOD_Global.TargetMatrix[ITarget].TargetPivot=-1;

   /*  Call the housekeeping utility routine to record this deallocation
    *  in the allocation information structure array.    */

   MethodInfo->NoteDeallocation (MethodInfo,Pivot,Status);


   if (*Status != STATUS__OK) goto Exit;

   /*  We reset the matrix elements for the pivot and the target that it
    *  was assigned to.  Any that were flagged as 'in use' now may be
    *  possible again - we have to check them. The element for the allocation
    *  that has just been deallocated must be possible.
    *  This bit wont execute if we're using Swaps, as it's not used and
    *  takes hours...
    */

   if (CONFIG_METHOD_Global.ParameterValues[CONFIG_METHOD_SELECT_PARM]
       .IntegerValue != CONFIG_METHOD_SEL_GBD){
     Matrix[Pivot][Target].PathStatus = CONFIG_METHOD_POSSIBLE;
     for (IPiv = 0; IPiv < NPiv; IPiv++) {
       if (Matrix[IPiv][Target].PathStatus == CONFIG_METHOD_TARGET_IN_USE) {
         if (MethodInfo->CheckAllocation(MethodInfo,IPiv,Target,
                 Matrix[IPiv][Target].Theta,&CollidingPivot)) {
           Matrix[IPiv][Target].PathStatus = CONFIG_METHOD_POSSIBLE;
         }
       }
     }
     for (ITarget = 0; ITarget < NTargets; ITarget++) {
       if (Matrix[Pivot][ITarget].PathStatus == CONFIG_METHOD_FIBRE_IN_USE) {
         if (MethodInfo->CheckAllocation(MethodInfo,Pivot,ITarget,
                            Matrix[Pivot][ITarget].Theta,&CollidingPivot)) {
           Matrix[Pivot][ITarget].PathStatus = CONFIG_METHOD_POSSIBLE;
         }
       }
     }

   /*  We run through all the possible paths given in the matrix. Some of
    *  these will be flagged as impossible because of a collision, and these
    *  are the ones we're interested in. For each of these, the number of
    *  the fibre for which the collision was detected is recorded in the
    *  matrix. If this colliding fibre is not the one we have just deallocated,
    *  then clearly the allocation is still invalid. If the colliding fibre
    *  recorded in the matrix is the one we've just deallocated, then this
    *  doesn't automatically mean that the path is now valid, since there
    *  may be other allocated fibres that it will still collide with (a path
    *  may collide with many fibres, but only one is recorded in the matrix).
    *  So in this case we need to do a full validity check on the allocation.
    */

     for (IPiv = 0; IPiv < NPiv; IPiv++) {
       if (ReportFunction != NULL) {
         if ((IPiv % ReportPivs) == 0) {
           PercentDone = ((float) IPiv / (float) NPiv) * 100.0;
           if ((*ReportFunction)(FunctionArg,'P',
                          "Updating fibre paths after deallocation",
                                 0,0,0,0,0,0.0,PercentDone) != 0) {
               MethodInfo->InfoRoutine(MethodInfo->InfoArgument,
                                        "Cancel request received");
                *Status = CONFIG__CANCELLED;
                goto Exit;
           }
         }
       } 
       for (ITarget = 0; ITarget < NTargets; ITarget++) {
         if (Matrix[IPiv][ITarget].PathStatus == CONFIG_METHOD_COLLISION) {
           if (Matrix[IPiv][ITarget].CollidingPivot == Pivot) {
             if (MethodInfo->CheckAllocation(MethodInfo,IPiv,ITarget,
                         Matrix[IPiv][ITarget].Theta,&CollidingPivot)) {
               Matrix[IPiv][ITarget].PathStatus = CONFIG_METHOD_POSSIBLE;
             }
           }
         }
       }
     }
   }


Exit:;

}

/* -------------------------------------------------------------------------- */

/*             C o n f i g  M e t h o d  S e t  A l l o c
 *
 *  Function:
 *     Updates the method-specific global structures after a new allocation.
 *
 *  Description:
 *     After a new allocation of pivot and target is made, this routine 
 *     should be called to update the various method-specific global 
 *     structures. It sets the possible path matrix elements
 *     for the actual pivot and fibre involved (ie one row and one column),
 *     but does not check for the effects of the new allocation upon the
 *     other possible paths - that is left to ConfigMethodNewAlloc().
 *     It calls ConfigNoteAllocation() so that the global allocation
 *     structure is updated with the new allocation.
 */

void ConfigMethodSetAlloc (
   const CONFIG_MethodInfo *MethodInfo,
   int Pivot,                     /* (>) The newly allocated pivot */
   int Target,                    /* (>) The newly allocated target */
   double Theta,                  /* (>) The theta angle for the button */
   StatusType *Status)            /* (!) Inherited status variable */
{

   /*  Local variables  */

   int ITarget;                  /* Loop index through targets */
   int IPiv,IOPiv;                     /* Loop index through pivots */
   CONFIG_METHOD_MatrixElement **Matrix;
                                 /* Local synonym for .Matrix in globals */
   int NPiv;                     /* Number of pivots */
   int NTargets;                 /* Number of targets */

   if (*Status != STATUS__OK) return;
   /*  Pick up local pointers to the various information structures whose
    *  addresses are held in the method-specific global structure. Also
    *  pick up the number of possible targets and pivots.
    */

   Matrix = CONFIG_METHOD_Global.Matrix;
   NTargets = CONFIG_METHOD_Global.NTargets;
   NPiv = CONFIG_METHOD_Global.NPiv;

   /*  Call the general non-method specific utility routine to update the
    *  global allocation information structure array. (This also checks that
    *  Pivot and Target are valid, which is a useful internal check.)
    *
    *  Note, allocation is tentative at this stage (Second last argument )
    */
   IOPiv = CONFIG_METHOD_Global.TargetMatrix[Target].TargetPivot;

   MethodInfo->NoteAllocation (MethodInfo,Pivot,Target,Theta,1,Status);
   if (CONFIG_METHOD_Global.ParameterValues[CONFIG_METHOD_SELECT_PARM].
                            IntegerValue == CONFIG_METHOD_SEL_IMPORTONLY) { 
     *Status = STATUS__OK;
   }
   if (*Status != STATUS__OK) goto Exit;

   /*  We make the internal check that the allocation shown is one that the
    *  possible path matrix regards as possible.  If it doesn't, then there's
    *  a problem here.
    */

   if (Matrix[Pivot][Target].PathStatus != CONFIG_METHOD_POSSIBLE) {
       char s[100];
       MethodInfo->TentativeFix(MethodInfo, Pivot, 0, Status);
       sprintf(s,"Invalid allocation, pivot %d to target %d attempted",
               Pivot + 1, Target + 1);
       MethodInfo->ErrRepRoutine(0,s);
       *Status = CONFIG__INVAL_ALLOC;
       goto Exit;
   }

   /*  Since this pivot is now allocated, we have to modify all the entries
    *  in the possible allocation matrix to reflect this.
    *  Don't do this for Oxford Code as it's an overhead    */
   
   if ((CONFIG_METHOD_Global.ParameterValues[CONFIG_METHOD_SELECT_PARM]
        .IntegerValue != CONFIG_METHOD_SEL_GBD) &&
       (CONFIG_METHOD_Global.ParameterValues[CONFIG_METHOD_SELECT_PARM]
        .IntegerValue != CONFIG_METHOD_SEL_BROKEN)){

       /* Confirm the tenative allocation */
       MethodInfo->TentativeFix(MethodInfo, Pivot, 1, Status);
       
       for (ITarget = 0; ITarget < NTargets; ITarget++) {
           if (Matrix[Pivot][ITarget].PathStatus == CONFIG_METHOD_POSSIBLE) {
               Matrix[Pivot][ITarget].PathStatus = CONFIG_METHOD_FIBRE_IN_USE;
           }
       }
       for (IPiv = 0; IPiv < NPiv; IPiv++) {
           if (Matrix[IPiv][Target].PathStatus == CONFIG_METHOD_POSSIBLE) {
               Matrix[IPiv][Target].PathStatus = CONFIG_METHOD_TARGET_IN_USE;
           }
       }
       Matrix[Pivot][Target].PathStatus = CONFIG_METHOD_ALLOCATED;
       Matrix[Pivot][Target].Theta = Theta;
   } else { /* Oxford Algorithm...*/
     
     /* OK, so because we didn't do that last bit for this algorithm the
      * last time we were here, and therefore we didn't call NewAlloc either
      * we're acutally going to have to call a new version of NewAlloc to
      * see if we actually hit anything with this move... */
     
     /*CrashTest will now update the VPivX,VPivY coords in the global struct*/

     if (ConfigMethodCrashTest(MethodInfo,Pivot,Target,Theta) == CONFIG_TRUE) {
         /* Reject the tentative allocation */
         MethodInfo->TentativeFix(MethodInfo, Pivot, 0, Status);
         *Status = CONFIG__INVAL_ALLOC;
     } else {
         /* Confirm the tentative allocation */
         MethodInfo->TentativeFix(MethodInfo, Pivot, 1, Status);
     }
   }
   
   /*  This section used to be before the crash test was performed, so was
    *  setting the pivot even if the allocation was invalid - KS 23/8/02
    */
   
   if (*Status == 0) {
      CONFIG_METHOD_Global.TargetMatrix[Target].TargetPivot=Pivot;
   }
       
 
Exit:;
   
}
                                                                      
/* -------------------------------------------------------------------------- */

/*             C o n f i g  M e t h o d  U p d a t e  M a t r i x
 *
 *  Function:
 *     Updates the allocation matrix on the basis of the allocation data.
 *
 *  Description:
 *     This routine goes through the allocations recorded in the allocation
 *     information and uses this to update the allocation matrix to reflect
 *     those allocations that are now invalid in the light of the allocations
 *     in the allocation data. This is needed in two places in the code: at
 *     the initialisation stage, if prior allocations had been made, and at
 *     the end of an allocation using the 'Oxford' algorithm, which for
 *     efficiency and algorithmic reasons does not keep the allocation matrix
 *     updated.
 *
 */
static void ConfigMethodUpdateMatrix (
   const CONFIG_MethodInfo *MethodInfo,
   CONFIG_ReportFunction
            ReportFunction,          /* (>) Feedback routine to call */
   void *FunctionArg,                /* (>) Caller-supplied argument */
   StatusType *Status)               /* (!) Inherited status */
{
   /*  Local variables */
   
   int CollidingPivot;               /* Pivot causing collision */
   int IPiv;                         /* Pivot index number */
   int ITarget;                      /* Target index number */ 
   int TargetIndex;                  /* Target index for allocated pivot */
   CONFIG_METHOD_MatrixElement **Matrix;
                                     /* Local synonym for .Matrix in globals */
   int NPiv;                         /* Number of pivots */
   double PercentDone;               /* Percentage of current task completed */
   int PivotIndex;                   /* Second loop index through all pivots */
   int ReportPivs;                   /* Report after processing this many */
   CONFIG_METHOD_TargetElement *TargetMatrix;
                                     /*Local synonym for global .TargetMatrix */
   
   if (*Status != STATUS__OK) return;
   
   Matrix = CONFIG_METHOD_Global.Matrix;
   TargetMatrix = CONFIG_METHOD_Global.TargetMatrix;
   NPiv = CONFIG_METHOD_Global.NPiv;
   ReportPivs = (int) ((float) NPiv *0.1);
   
   for (IPiv = 0; IPiv < MethodInfo->NumberPivots; IPiv++) {
      if (MethodInfo->AllocInfo[IPiv].State == 'A') {
         for (ITarget = 0; ITarget < MethodInfo->NumberTargets; ITarget++) {
            if (Matrix[IPiv][ITarget].PathStatus == CONFIG_METHOD_POSSIBLE) {
               Matrix[IPiv][ITarget].PathStatus = CONFIG_METHOD_FIBRE_IN_USE;
            }
         }
         TargetIndex = MethodInfo->AllocInfo[IPiv].TargetIndex;
         for (PivotIndex = 0; PivotIndex < MethodInfo->NumberPivots; 
                                                             PivotIndex++) {
            if (Matrix[PivotIndex][TargetIndex].PathStatus 
                 == CONFIG_METHOD_POSSIBLE) {
               Matrix[PivotIndex][TargetIndex].PathStatus = 
                 CONFIG_METHOD_TARGET_IN_USE;
            }
         }
         TargetMatrix[TargetIndex].TargetPivot = IPiv;
         Matrix[IPiv][MethodInfo->AllocInfo[IPiv].TargetIndex].PathStatus =
             CONFIG_METHOD_ALLOCATED;
         Matrix[IPiv][MethodInfo->AllocInfo[IPiv].TargetIndex].Theta =
             MethodInfo->AllocInfo[IPiv].Theta;
      }
   }
   for (IPiv = 0; IPiv < MethodInfo->NumberPivots; IPiv++) {
      if (ReportFunction != NULL) {
         if ((IPiv % ReportPivs) == 0) {
            PercentDone = ((float) IPiv / 
                                (float) MethodInfo->NumberPivots) * 100.0;
            if ((*ReportFunction)(FunctionArg,'P',
                                   "Processing allocated fibres",
                                   0,0,0,0,0,0.0,PercentDone) != 0) {
               MethodInfo->InfoRoutine(MethodInfo->InfoArgument,
                                        "Cancel request received");
               *Status = CONFIG__CANCELLED;
               goto Exit;
            }
         }
      }
      for (ITarget = 0; ITarget < MethodInfo->NumberTargets; ITarget++) {
         if (Matrix[IPiv][ITarget].PathStatus == CONFIG_METHOD_POSSIBLE) {
            if (!MethodInfo->CheckAllocation(MethodInfo,IPiv,ITarget,
                           Matrix[IPiv][ITarget].Theta,&CollidingPivot)) {
               Matrix[IPiv][ITarget].PathStatus = CONFIG_METHOD_COLLISION;
               Matrix[IPiv][ITarget].CollidingPivot = CollidingPivot;
             
            }
         }
       
      }
   }

Exit:;

}

/* -------------------------------------------------------------------------- */

/*             C o n f i g  M e t h o d  P i c k  N e x t
 *
 *  Function:
 *     Identifies the hardest pivot to allocate and allocates it a target.
 *
 *  Description:
 *     This routine looks at the allocation matrix and identifies the
 *     pivot that is the hardest to allocate, in the sense of being the
 *     pivot with the fewest possible targets. The algorithm says that
 *     this is the pivot that should be allocated next, and so it then
 *     picks out the most desireable target for it - the highest priority
 *     target it can reach. If unable to make any allocation, this routine
 *     returns a false value.
 *
 */

CONFIG_Boolean ConfigMethodPickNext (
   const CONFIG_MethodInfo *MethodInfo,
   int *Pivot,                    /* (<) The newly allocated pivot */
   int *Target,                   /* (<) The newly allocated target */
   double *Theta)                 /* (<) The theta angle for the button */
{

   /*  Local variables  */

   const CONFIG_AllocInfo *AllocInfo;  /* Allocation info array of structures */
   int BestTarget;               /* The target we should choose */
   CONFIG_Boolean CanAllocate;   /* True if there are possible allocations */
   int ClosestTarget;            /* The target closest to the pivot */
   double DeltaX;                /* Distance in X from target to pivot */
   double DeltaY;                /* Distance in Y from target to pivot */
   int FurthestTarget;           /* The target furthest from the pivot */
   int HardestPivot;             /* Pivot with the least possible targets */
   int HardestTarget;            /* The target with the least possible pivots */
   int IPiv;                     /* Loop index through pivots */
   int ITarget;                  /* Loop index through targets */
   double LengthSquared;         /* Square of fibre length */
   CONFIG_METHOD_MatrixElement **Matrix;
                                 /* Local synonym for .Matrix in globals */
   float MaxCrowding;            /* Maximum crowding of all the targets */
   int MaxPriority;              /* Maximum priority of possible targets */
   double MaxLengthSquared;      /* Square of longest fibre length */
   double MinAlpha;              /* Least non-radial pivot angle */
   float MinCrowding;            /* Minimum crowding of the possible targets */
   double MinLengthSquared;      /* Square of shortest fibre length */
   int MinPivotCount;            /* Least pivot count for any target */
   int MinTargets;               /* Least number of targets for any pivot */
   int MostCrowdedTarget;        /* Most crowded of the possible targets */
   int NPiv;                     /* Number of pivots */
   int NTargets;                 /* Number of targets */
   int PivotCount;               /* Number of pivots that can reach a target */
   const CONFIG_PivotInfo *PivotInfo;  /* Pivot info array of structures */
   long PivotX;                  /* X-coordinate of pivot */
   long PivotY;                  /* Y-coordinate of pivot */
   int Priority;                 /* Priority of possible target */
   int SelectionCode;            /* Parameterised selection criterion */
   int StraightestTarget;        /* Target closest to the radial pivot line */
   int TargetCount;              /* Number of possible targets */
   const CONFIG_TargetInfo *TargetInfo;/* Target info array of structures */
   long TargetX;                 /* X-coordinate of target */
   long TargetY;                 /* Y-coordinate of target */

   /*  Pick up local pointers to the various information structures whose
    *  addresses are held in the method-specific global structure. Also
    *  pick up the number of possible targets and pivots.
    */

   Matrix = CONFIG_METHOD_Global.Matrix;
   NTargets = CONFIG_METHOD_Global.NTargets;
   NPiv = CONFIG_METHOD_Global.NPiv;
   AllocInfo = MethodInfo->AllocInfo;
   PivotInfo = MethodInfo->PivotInfo;
   TargetInfo = MethodInfo->TargetInfo;

   MinTargets = NTargets + 1;
   HardestPivot = 0;
   CanAllocate = CONFIG_FALSE;

   /*  We go through each pivot in turn  */

   for (IPiv = 0; IPiv < NPiv; IPiv++) {

      /*  If the pivot is already allocated, or broken, we ignore it.  An
       *  eligible pivot will have a state given as 'U' for 'unallocated'
       */

      if (AllocInfo[IPiv].State == 'U') {

         /*  Look through the matrix for this pivot, counting the number of
          *  possible targets it has.
          */

         TargetCount = 0;
         for (ITarget = 0; ITarget < NTargets; ITarget++) {
            if (Matrix[IPiv][ITarget].PathStatus == CONFIG_METHOD_POSSIBLE) {
               TargetCount++;
            }
         }

         /*  Is this the hardest pivot so far?  */

         if (TargetCount > 0) {
            CanAllocate = CONFIG_TRUE;
            if (TargetCount < MinTargets) {
               MinTargets = TargetCount;
               HardestPivot = IPiv;
            }
         }
      }
   }

   /*  If there were any possible allocations, then we already know the
    *  pivot to use - the hardest one. We then look through the possible
    *  targets for that pivot and select the highest priority one. If there
    *  are a number at that priority, we pick on the basis of distance
    *  between fibre and pivot or on the basis of the non-radial pivot angle,
    *  or on the basis of 'hardest object' - the object that can be reached
    *  by the fewest fibres.
    *
    *  (At present, which to use is unclear - we calculate the lot!)
    */

   if (CanAllocate) {
      MaxPriority = -1;
      TargetCount = 0;
      BestTarget = 0;
      for (ITarget = 0; ITarget < NTargets; ITarget++) {
         if (Matrix[HardestPivot][ITarget].PathStatus 
                                               == CONFIG_METHOD_POSSIBLE) {
            Priority = TargetInfo[ITarget].Priority;
            if (Priority > MaxPriority) {
               MaxPriority = Priority;
               TargetCount = 1;
               BestTarget = ITarget;
            } else if (Priority == MaxPriority) {
               TargetCount++;
            }
         }
      }

      /*  Having got the highest priority in the range of possible
       *  targets, we may find that we've got more than one at that
       *  priority, in which case we have to search again to find the
       *  best one to use.
       */

      if (TargetCount > 1) {
         PivotX = PivotInfo[HardestPivot].X;
         PivotY = PivotInfo[HardestPivot].Y;
         MinAlpha = PI;
         MaxLengthSquared = 0.0;
         MinPivotCount = NPiv + 1;
         MinLengthSquared = (double) MethodInfo->MaxExtension(MethodInfo) + 
                            1000.0;
         MinLengthSquared = MinLengthSquared * MinLengthSquared;
         HardestTarget = 0;
         ClosestTarget = 0;
         StraightestTarget = 0;
         MostCrowdedTarget = 0;
         FurthestTarget = 0;
         BestTarget = 0;
         MaxCrowding = 0.0;
         for (ITarget = 0; ITarget < NTargets; ITarget++) {
            if (CONFIG_METHOD_Global.TargetCrowding[ITarget] > MaxCrowding) {
               MaxCrowding = CONFIG_METHOD_Global.TargetCrowding[ITarget];
            }
         }
         MinCrowding = MaxCrowding;
         for (ITarget = 0; ITarget < NTargets; ITarget++) {
            if (Matrix[HardestPivot][ITarget].PathStatus 
                                      == CONFIG_METHOD_POSSIBLE) {
               Priority = TargetInfo[ITarget].Priority;
               if (Priority == MaxPriority) {
                  TargetX = TargetInfo[ITarget].X;
                  TargetY = TargetInfo[ITarget].Y;
                  DeltaX = (double) TargetX - PivotX;
                  DeltaY = (double) TargetY - PivotY;
                  LengthSquared = (DeltaX * DeltaX) + (DeltaY * DeltaY);
                  PivotCount = 0;
                  for (IPiv = 0; IPiv < NPiv; IPiv++) {
                     if (Matrix[IPiv][ITarget].PathStatus
                                             == CONFIG_METHOD_POSSIBLE) {
                        PivotCount++;
                     }
                  }

                  if (LengthSquared > MaxLengthSquared) {
                     MaxLengthSquared = LengthSquared;
                     FurthestTarget = ITarget;
                  }
                  if (LengthSquared < MinLengthSquared) {
                     MinLengthSquared = LengthSquared;
                     ClosestTarget = ITarget;
                  }
                  if (CONFIG_METHOD_Global.TargetCrowding[ITarget] 
                                                             < MinCrowding) {
                     MinCrowding = 
                               CONFIG_METHOD_Global.TargetCrowding[ITarget];
                     MostCrowdedTarget = ITarget;
                  }
                  if (MinAlpha > Matrix[HardestPivot][ITarget].Alpha) {
                     MinAlpha = Matrix[HardestPivot][ITarget].Alpha;
                     StraightestTarget = ITarget;
                  }
                  if (PivotCount < MinPivotCount) {
                     MinPivotCount = PivotCount;
                     HardestTarget = ITarget;
                  }

                  /*  Whether we use Best, MostCrowded, Hardest or Straightest
                   *  target is something controlled by a method parameter.
                   *  Here, we always try to make guide fibres the closest
                   *  choice, and use the parameter for ordinary targets.
                   */

                  SelectionCode = CONFIG_METHOD_Global.
                     ParameterValues[CONFIG_METHOD_SELECT_PARM].IntegerValue;
                  if (MethodInfo->FibreIsGuide(MethodInfo,
                                       PivotInfo[HardestPivot].FibreType))
                  {
                     BestTarget = ClosestTarget;
                  } else {
                     switch (SelectionCode) {
                        case CONFIG_METHOD_SEL_CROWD : {
                           BestTarget = MostCrowdedTarget; break;
                        }
                        case CONFIG_METHOD_SEL_FAR : {
                           BestTarget = FurthestTarget; break;
                        }
                        case CONFIG_METHOD_SEL_HARD : {
                           BestTarget = HardestTarget; break;
                        }
                        case CONFIG_METHOD_SEL_STRAIGHT : {
                           BestTarget = StraightestTarget; break;
                        }
                     }
                  }
               }
            } 
         }
      }
      *Pivot = HardestPivot;
      *Target = BestTarget;
      *Theta = Matrix[HardestPivot][BestTarget].Theta;
   }

   return (CanAllocate);

}



/* -------------------------------------------------------------------------- */

/*             C o n f i g  M e t h o d  R e l e a s e  M e m
 *
 *  Function:
 *     Releases any memory allocated by the allocation-specific code.
 *
 *  Description:
 *     This routine is called at the end of the program in order to give 
 *     it a chance to release any dynamic memory that may have been allocated.
 *     Any such memory should have its address recorded in the global data
 *     structures.
 */

static void ConfigMethodReleaseMem (
   const CONFIG_MethodInfo *MethodInfo DUNUSED)
{
   /*  Local variables  */

   int Index,IT;                /* Index through method parameters */
   int *ind,*ibad;

   /*  Check all the dynamically allocated memory addresses, and free
    *  any that are still allocated.
    */
   if (CONFIG_METHOD_Global.Matrix != NULL) {
      free (CONFIG_METHOD_Global.Matrix);
      CONFIG_METHOD_Global.Matrix = NULL;
   }
   if (CONFIG_METHOD_Global.MatrixAddr != NULL) {
      free (CONFIG_METHOD_Global.MatrixAddr);
      CONFIG_METHOD_Global.MatrixAddr = NULL;
   }
   if (CONFIG_METHOD_Global.Cross != NULL) {
      free (CONFIG_METHOD_Global.Cross);
      CONFIG_METHOD_Global.Cross = NULL;
   }
   if (CONFIG_METHOD_Global.VPivX != NULL) {
      free (CONFIG_METHOD_Global.VPivX);
      CONFIG_METHOD_Global.VPivX = NULL;
   }
   if (CONFIG_METHOD_Global.VPivY != NULL) {
      free (CONFIG_METHOD_Global.VPivY);
      CONFIG_METHOD_Global.VPivY = NULL;
   }
   if (CONFIG_METHOD_Global.CrossAddr != NULL) {
      free (CONFIG_METHOD_Global.CrossAddr);
      CONFIG_METHOD_Global.CrossAddr = NULL;
   }
   ind = CONFIG_METHOD_Global.ConeIndex;
   ind--;
   if (CONFIG_METHOD_Global.ConeIndex != NULL) {
     free (ind);
     CONFIG_METHOD_Global.ConeIndex = NULL;
   }
   ibad = CONFIG_METHOD_Global.IBad;
   ibad--;
   if (CONFIG_METHOD_Global.IBad != NULL) {
     free (ibad);
     CONFIG_METHOD_Global.IBad = NULL;
   }

   Index = CONFIG_METHOD_Global.NTargets;

   if (CONFIG_METHOD_Global.TargetMatrix != NULL) {
     for (IT=0;IT<Index;IT++){
       if (CONFIG_METHOD_Global.TargetMatrix[IT].Pivot != NULL) {
         free (CONFIG_METHOD_Global.TargetMatrix[IT].Pivot);
         CONFIG_METHOD_Global.TargetMatrix[IT].Pivot = NULL;
       }
       if (CONFIG_METHOD_Global.TargetMatrix[IT].ConeTargets != NULL) {
         free (CONFIG_METHOD_Global.TargetMatrix[IT].ConeTargets);
         CONFIG_METHOD_Global.TargetMatrix[IT].ConeTargets = NULL;
       }
     }
     free(CONFIG_METHOD_Global.TargetMatrix);
     CONFIG_METHOD_Global.TargetMatrix = NULL;
   }
   /*  If there are any 'any value' character parameters that have been set,
    *  free any memory allocated to them.
    */

   for (Index = 0; Index < CONFIG_METHOD_NPARMS; Index++) {
      if (CONFIG_METHOD_ParmTypes[Index] == 'C') {
         if (CONFIG_METHOD_ParmNValues[Index] == 0) {
            if (CONFIG_METHOD_Global.ParameterValues[Index].StringAddress
                                                                    != NULL) {
               free (
                 CONFIG_METHOD_Global.ParameterValues[Index].StringAddress);
            }
         }
      }
   }
   
}

/* -------------------------------------------------------------------------- */

/*         C o n f i g  M e t h o d  N u m b e r  P a r a m s
 *
 *  Function:
 *     Returns the number of method-specific parameters.
 *
 *  Description:
 *     This routine returns the number of method-specific parameters that
 *     can be used to control the details of the allocation algorithm.
 */

static int ConfigMethodNumberParams (
    const CONFIG_MethodInfo *MethodInfo DUNUSED)
{
   return (CONFIG_METHOD_NPARMS);
}

/* -------------------------------------------------------------------------- */

/*         C o n f i g  M e t h o d  P a r a m  D e t a i l s
 *
 *  Function:
 *     Returns details about the method-specific parameters.
 *
 *  Description:
 *     This routine returns details about the various method-specific 
 *     parameters that can be used to control the details of the 
 *     allocation algorithm. This includes their names, types, and
 *     possible values.
 */

static void ConfigMethodParamDetails (
   const CONFIG_MethodInfo *MethodInfo DUNUSED,
   int ParameterNumber,
   char **Name,
   char *Type,
   double Limits[],
   int *NValues,
   char ***Values,
   StatusType *Status)
{
   /*  Local variables  */

   int Index;                         /* Index into parameter arrays */

   if (*Status != STATUS__OK) return;

   if ((ParameterNumber < 1) || (ParameterNumber > CONFIG_METHOD_NPARMS)) {
       MethodInfo->ErrRepRoutine(0,
        "Attempt to get details about non-existent method-specific parameter");
      *Status = CONFIG__INVAL_PARM_NO;
   } else {
      Index = ParameterNumber - 1;
      *Name = CONFIG_METHOD_ParmNames[Index];
      *Type = CONFIG_METHOD_ParmTypes[Index];
      Limits[0] = CONFIG_METHOD_ParmLimits[Index][0];
      Limits[1] = CONFIG_METHOD_ParmLimits[Index][1];
      *NValues = CONFIG_METHOD_ParmNValues[Index];
      *Values = CONFIG_METHOD_ParmValues[Index];
   }

}

/* -------------------------------------------------------------------------- */

/*              C o n f i g  M e t h o d  S e t  P a r a m
 *
 *  Function:
 *     Sets the value of one of the method-specific parameters.
 *
 *  Description:
 *     This routine sets the value of one of the various method-specific 
 *     parameters that can be used to control the details of the 
 *     allocation algorithm. The parameter is specified by name, and
 *     the value is specified as a character string.
 */

static void ConfigMethodSetParam (
   const CONFIG_MethodInfo *MethodInfo DUNUSED,
   char *Name,                        /* (>) Name of parameter */
   char *Value,                       /* (>) Value of parameter */
   StatusType *Status)                /* (!) Inherited status value */
{
   /*  Local variables  */

   int Bytes;                         /* Number of bytes for character param */
   char *ExpertStr;                   /* Pointer to "(expert)" string */
   double FloatValue;                 /* Value of floating point param */
   int Index;                         /* Index into parameter arrays */
   long IntegerValue;                 /* Value of integer parameter */
   int Length;                        /* Chars in string to compare */
   double MaxLimit;                   /* Maximum value for parameter */
   double MinLimit;                   /* Minimum value for parameter */
   CONFIG_Boolean NameMatched;        /* True if Name is a valid name */
   int NValues;                       /* Number of values for char param */
   CONFIG_Boolean ParamOK;            /* True if char param is valid string */
   char **ParmValues;                 /* Address of possible char values */
   char *StringAddress;               /* Address of parameter value string */
   int StringIndex;                   /* Index through possible parm values */
   char Type;                         /* PArameter type 'I','F','C','L' */

   if (*Status != STATUS__OK) return;

   /*  Work through the names of the various parameters looking for
    *  a match with that specified.
    */

   NameMatched = CONFIG_FALSE;
   for (Index = 0; Index < CONFIG_METHOD_NPARMS; Index++) {
      if (!strcmp(Name,CONFIG_METHOD_ParmNames[Index])) {

         /*  Name matches, so pick up the type, limits (for numrical values)
          *  and number of possible values (for character parameters).
          */

         NameMatched = CONFIG_TRUE;
         Type = CONFIG_METHOD_ParmTypes[Index];
         MinLimit = CONFIG_METHOD_ParmLimits[Index][0];
         MaxLimit = CONFIG_METHOD_ParmLimits[Index][1];
         NValues = CONFIG_METHOD_ParmNValues[Index];

         /*  Now handle the parameter according to type  */

         if (Type == 'L') {

            /*  Logical parameters must be either 'T' or 'F' */

            if (!strcmp(Value,"T")) {
               CONFIG_METHOD_Global.ParameterValues[Index].IntegerValue = 
                                                                   CONFIG_TRUE;
            } else if (!strcmp(Value,"F")) {
               CONFIG_METHOD_Global.ParameterValues[Index].IntegerValue = 
                                                                  CONFIG_FALSE;
            } else {
                char s[100];
                sprintf(s,
                      "'%s' is a logical parameter and cannot be set to '%s'",
                        Name,Value);
                MethodInfo->ErrRepRoutine(0,s);
               *Status = CONFIG__INVAL_PARM_VALUE;
            }

         } else if (Type == 'I') {

            /*  Integer parameters must be valid integer values within
             *  the specified range.
             */

            if (sscanf(Value,"%ld",&IntegerValue) == 1) {
               if (((double) IntegerValue < MinLimit) ||
                    ((double) IntegerValue > MaxLimit)) {
                   char s[100];
                   sprintf(s,
                  "Parameter '%s' must be between %d and %d. %ld is invalid.",
                           Name,(int) MinLimit, (int) MaxLimit, IntegerValue);
                   MethodInfo->ErrRepRoutine(0,s);
                   *Status = CONFIG__PARM_OUT_RANGE;
               } else {
                  CONFIG_METHOD_Global.ParameterValues[Index].IntegerValue = 
                                                               IntegerValue;
               }
            } else {
                char s[100];
                sprintf(s,
                        "'%s' is not a valid value for integer parameter '%s'",
                        Value,Name);
                MethodInfo->ErrRepRoutine(0,s);
                *Status = CONFIG__INVAL_PARM_VALUE;
            }

         } else if (Type == 'F') {

            /*  Floating point parameters must be valid real values within
             *  the specified range.
             */

            if (sscanf(Value,"%lf",&FloatValue) == 1) {
               if ((FloatValue < MinLimit) || (FloatValue > MaxLimit)) {
                   char s[100];
                   sprintf(s,
                   "Parameter '%s' must be between %f and %f. %f is invalid.",
                           Name,MinLimit,MaxLimit,FloatValue);
                           MethodInfo->ErrRepRoutine(0,s);
                  *Status = CONFIG__PARM_OUT_RANGE;
               } else {
                  CONFIG_METHOD_Global.ParameterValues[Index].DoubleValue = 
                                                                 FloatValue;
               }
            } else {
                char s[100];
                sprintf(s,
                        "'%s' is not a valid value for parameter '%s'",
                        Value,Name);
                MethodInfo->ErrRepRoutine(0,s);
               *Status = CONFIG__INVAL_PARM_VALUE;
            }

         } else if (Type == 'C') {

            /*  Character values can have either any value, or one of a 
             *  specified set. 
             */

            if (NValues <= 0) {

               /*  This paramter can have any value, so it must be valid.
                *  We allocate memory to hold a copy of the new value (this
                *  allows the value to be any length at all), first freeing
                *  any previously allocated.
                */

               if (CONFIG_METHOD_Global.ParameterValues[Index].StringAddress
                                                                     != NULL) {
                  free (CONFIG_METHOD_Global.ParameterValues[Index]
                                                              .StringAddress);
               }
               Bytes = strlen(Value);
               if (Bytes == 0) {
                  CONFIG_METHOD_Global.ParameterValues[Index].StringAddress 
                                                                      = NULL;
               } else {
                  StringAddress = (char *) malloc(Bytes + 1);
                  strcpy (StringAddress,Value);
                  CONFIG_METHOD_Global.ParameterValues[Index].StringAddress
                                                             = StringAddress;
               }
            } else {

               /*  This character string must have one of a set of specified
                *  values. Check that this is one, and set the index in
                *  the global variables. Allow for the possibility that the
                *  "(expert)" may have been stripped off.
                */

               ParmValues = CONFIG_METHOD_ParmValues[Index];
               ParamOK = CONFIG_FALSE;
               for (StringIndex = 0; StringIndex < NValues; StringIndex++) {
                  if ((ExpertStr = 
                             strstr(ParmValues[StringIndex],"(expert)"))) {
                     Length = ExpertStr - ParmValues[StringIndex];
                  } else {
                     Length = strlen(Value);
                  }
                  if (!strncmp(Value,ParmValues[StringIndex],Length)) {
                     CONFIG_METHOD_Global.ParameterValues[Index].StringAddress
                                                    = ParmValues[StringIndex];
                     CONFIG_METHOD_Global.ParameterValues[Index].IntegerValue
                                                                = StringIndex;
                     ParamOK = CONFIG_TRUE;
                     break;
                  }
               }
               if (!ParamOK) {
                   char s[100];
                   sprintf(s,
                           "'%s' is not one of the possible values for '%s'",
                           Value,Name);
                   MethodInfo->ErrRepRoutine(0,s);
                   *Status = CONFIG__INVAL_PARM_VALUE;
               }
            }
         }

         /*  Finished processing a matched name, so break out of the loop
          *  through the various parameter names.
          */

         break;
      }
   }

   if (!NameMatched) {
       char s[100];
       sprintf(s,"'%s' is not a recognised parameter name",Name);
       MethodInfo->ErrRepRoutine(0,s);
      *Status = CONFIG__INVAL_PARM_NAME;
   }
}
                  
/* -------------------------------------------------------------------------- */

/*              C o n f i g  M e t h o d  G e t  P a r a m
 *
 *  Function:
 *     Gets the value of one of the method-specific parameters.
 *
 *  Description:
 *     This routine gets the value of one of the various method-specific 
 *     parameters that can be used to control the details of the 
 *     allocation algorithm. The parameter is specified by name, and
 *     the value is returned as a character string.
 */

static void ConfigMethodGetParam (
   const CONFIG_MethodInfo *MethodInfo DUNUSED,
   char *Name,                        /* (>) Name of parameter */
   char *Value,                       /* (<) Value of parameter */
   int VSize,                         /* (>) Maximum number of characters */
   StatusType *Status)                /* (!) Inherited status value */
{
   /*  Local variables  */

   int Index;                         /* Index into parameter arrays */
   CONFIG_Boolean NameMatched;        /* True if Name is a valid name */
   char Type;                         /* Parameter type 'I','F','C','L' */
   char *ValuePtr;                    /* Points to string to be returned */
   char ValueString[64];              /* Used to format numbers and logicals */

   if (*Status != STATUS__OK) return;

   /*  Work through the names of the various parameters looking for
    *  a match with that specified.
    */

   NameMatched = CONFIG_FALSE;
   ValuePtr = NULL;
   for (Index = 0; Index < CONFIG_METHOD_NPARMS; Index++) {
      if (!strcmp(Name,CONFIG_METHOD_ParmNames[Index])) {

         /*  Name matches, so pick up its type.
          */

         NameMatched = CONFIG_TRUE;
         Type = CONFIG_METHOD_ParmTypes[Index];

         /*  Now handle the parameter according to type  */

         if (Type == 'L') {

            /*  Logical parameters are returned as either 'T' or 'F' */

            if (CONFIG_METHOD_Global.ParameterValues[Index].IntegerValue) {
               ValuePtr = "T";
            } else {
               ValuePtr = "F";
            }

         } else if (Type == 'I') {

            /*  Integer parameters are formatted into a value string.
             */

            sprintf(ValueString,"%d",
                CONFIG_METHOD_Global.ParameterValues[Index].IntegerValue);
            ValuePtr = ValueString;

         } else if (Type == 'F') {

            /*  Floating point parameters are also formatted into a value
             *  string.
             */

            sprintf(ValueString,"%f",
                CONFIG_METHOD_Global.ParameterValues[Index].DoubleValue);
            ValuePtr = ValueString;

         } else if (Type == 'C') {

            /*  Character values can have either any value, or one of a 
             *  specified set. In either case, the address of the current
             *  value is held in the global variables.
             */

            ValuePtr = 
                 CONFIG_METHOD_Global.ParameterValues[Index].StringAddress;
         }

         /*  Finished processing a matched name, so break out of the loop
          *  through the various parameter names.
          */

         break;
      }
   }

   /*  After working through the various accepted parameter names, we either
    *  didn't find the name specified, in which case we have an error, or
    *  we have a pointer (ValuePtr) to a string giving the current value.
    */

   if (NameMatched) {
      if (ValuePtr == NULL) {
         Value[0] = '\0';
      } else {
         strncpy (Value,ValuePtr,VSize);
         Value[VSize - 1] = '\0';
      }
   } else {
       char s[100];
       sprintf(s,"'%s' is not a recognised parameter name",Name);
      MethodInfo->ErrRepRoutine(0,s);
      *Status = CONFIG__INVAL_PARM_NAME;
   }
}

/* -------------------------------------------------------------------------- */

/*             C o n f i g  M e t h o d  D i s a b l e  S k y
 *
 *  Function:
 *     Disables any sky targets in the allocation matrix.
 *
 *  Description:
 *     This routine looks at the allocation matrix and disables any sky
 *     targets by setting their allocation to CONFIG_METHOD_DISABLED instead
 *     of CONFIG_METHOD_POSSIBLE. This can be reversed by the routine
 *     ConfigMethodReEnable(). The routine returns the number of potential
 *     allocations that were disabled. Note that we only do this for fibre
 *     types that have to share sky and non-sky targets, and for which there
 *     is an entry in the global arrays controlling sky allocation.
 */

static int ConfigMethodDisableSky (
   const CONFIG_MethodInfo *MethodInfo)
{

   /*  Local variables  */

   int FibreType;                /* Fibre type code for a pivot */
   CONFIG_METHOD_MatrixElement **Matrix;
                                 /* Local synonym for .Matrix in globals */
   int IPiv;                     /* Loop index through pivots */
   int ITarget;                  /* Loop index through targets */
   int IType;                    /* Index through pivot types */
   int NDisabled;                /* Number of allocations disabled */
   int NPiv;                     /* Number of pivots */
   int NTargets;                 /* Number of targets */
   int NumberTypes;              /* Number of types in sky count array */
   const CONFIG_PivotInfo *PivotInfo;/* Pivot info array of structures */
   const CONFIG_TargetInfo *TargetInfo;/* Target info array of structures */
   char Type;                    /* Target type code */

   /*  Pick up local pointers to the various information structures whose
    *  addresses are held in the method-specific global structure. Also
    *  pick up the number of possible targets and pivots.
    */

   Matrix = CONFIG_METHOD_Global.Matrix;
   NTargets = CONFIG_METHOD_Global.NTargets;
   NPiv = CONFIG_METHOD_Global.NPiv;
   TargetInfo = MethodInfo->TargetInfo;
   PivotInfo = MethodInfo->PivotInfo;

   /*  We go through each target in turn . For each sky target, we go through
    *  the allocation matrix replacing any CONFIG_METHOD_POSSIBLE flag with
    *  CONFIG_METHOD_DISABLED. Since this is the only change we make, it is
    *  easily reversed with no need to save additional information - we
    *  don't need a 'TypeWas' field, for example.
    */
   
   NumberTypes = CONFIG_METHOD_Global.FibreSkyEntries;
   NDisabled = 0;
   for (ITarget = 0; ITarget < NTargets; ITarget++) {
      Type = TargetInfo[ITarget].Type;
      if (MethodInfo->TargetIsSky(MethodInfo,Type)) {
         for (IPiv = 0; IPiv < NPiv; IPiv++) {
            if (Matrix[IPiv][ITarget].PathStatus == CONFIG_METHOD_POSSIBLE) {
            
               /*  At this point we have a possible allocation for this
                *  target. If this is to a fibre of a type that shares sky
                *  and non-sky targets, and if this is a type that is being
                *  controlled explicitly, we disable the allocation. (Actually
                *  and fibre type that passes the second criterion ought to
                *  pass the first, so the first may be redundant.)
                *
                *  We don't want to disable the allocation if it is in fact
                *  a current allocation! (This can happen if we try to
                *  allocate a field with existing allocations.)
                */
               
               if (CONFIG_METHOD_Global.TargetMatrix[ITarget].TargetPivot 
                                                                     != IPiv) {
                  FibreType = PivotInfo[IPiv].FibreType;
                  if (MethodInfo->FibreSharesSky(MethodInfo,FibreType)) {
                     for (IType = 0; IType < NumberTypes; IType++) {
                        if (MethodInfo->FibreTypeMatch(MethodInfo,FibreType,
                           CONFIG_METHOD_Global.FibreTypesForSky[IType])) {
                           NDisabled++;
                           Matrix[IPiv][ITarget].PathStatus = 
                                                 CONFIG_METHOD_DISABLED;
                           break;
                        }
                     }
                  }
               }
            }
         }
      }
   }
   return NDisabled;
}

/* -------------------------------------------------------------------------- */

/*             C o n f i g  M e t h o d  R e  E n a b l e
 *
 *  Function:
 *     Reenables any disabled targets in the allocation matrix.
 *
 *  Description:
 *     This routine looks at the allocation matrix and re-enables any disabled
 *     targets by setting their allocation to CONFIG_METHOD_POSSIBLE instead
 *     of CONFIG_METHOD_DISABLED. This reverses the effect of the a routine
 *     such as ConfigMethodDisableSky(). The routine returns the number of 
 *     potential allocations that were re-enabled
 *
 */

static int ConfigMethodReEnable (
   const CONFIG_MethodInfo *MethodInfo)
{

   /*  Local variables  */

   int CollidingPivot;           /* A pivot causing a collision */
   CONFIG_METHOD_MatrixElement **Matrix;
                                 /* Local synonym for .Matrix in globals */
   int IPiv;                     /* Loop index through pivots */
   int ITarget;                  /* Loop index through targets */
   int NEnabled;                 /* Number of allocations re-enabled */
   int NPiv;                     /* Number of pivots */
   int NTargets;                 /* Number of targets */

   /*  Pick up local pointers to the various information structures whose
    *  addresses are held in the method-specific global structure. Also
    *  pick up the number of possible targets and pivots.
    */

   Matrix = CONFIG_METHOD_Global.Matrix;
   NTargets = CONFIG_METHOD_Global.NTargets;
   NPiv = CONFIG_METHOD_Global.NPiv;

   /*  We go through each target in turn . For each target, we go through
    *  the allocation matrix replacing any CONFIG_METHOD_DISABLED flag with
    *  CONFIG_METHOD_POSSIBLE. Of course, things may have changed in the
    *  interim - there has usually been an allocation performed - so each
    *  allocation has to be re-checked before we flag it as possible again.
    */
   
   NEnabled = 0;
   for (ITarget = 0; ITarget < NTargets; ITarget++) {
      for (IPiv = 0; IPiv < NPiv; IPiv++) {
         if (Matrix[IPiv][ITarget].PathStatus == CONFIG_METHOD_DISABLED) {
            if (MethodInfo->CheckAllocation(MethodInfo,IPiv,ITarget,
                         Matrix[IPiv][ITarget].Theta,&CollidingPivot)) {
                Matrix[IPiv][ITarget].PathStatus = CONFIG_METHOD_POSSIBLE;
                NEnabled++;
            } else {
                Matrix[IPiv][ITarget].CollidingPivot = CollidingPivot;
                Matrix[IPiv][ITarget].PathStatus = CONFIG_METHOD_COLLISION;
            }   
         }
      }
   }
   return NEnabled;
}

/* -------------------------------------------------------------------------- */

/*                 C o n f i g  M e t h o d  M a x  A l l o c
 *
 *  Function:
 *     Counts the maximum possible allocations from the allocation matrix.
 *
 *  Description:
 *     This routine looks at the allocation matrix and returns the maximum
 *     number of allocations on the basis of the allocations that are
 *     flagged as possible.
 */

static int ConfigMethodMaxAlloc (void)
{

   /*  Local variables  */

   CONFIG_METHOD_MatrixElement **Matrix;
                                 /* Local synonym for .Matrix in globals */
   int IPiv;                     /* Loop index through pivots */
   int ITarget;                  /* Loop index through targets */
   int MaxAllocs;                /* Maximum number of allocations */
   int MaxPivots;                /* Maximum number of allocatable pivots */
   int MaxTargets;               /* Maximum number of allocatable targets */
   int NPiv;                     /* Number of pivots */
   int NTargets;                 /* Number of targets */

   /*  Pick up local pointers to the various information structures whose
    *  addresses are held in the method-specific global structure. Also
    *  pick up the number of possible targets and pivots.
    */

   Matrix = CONFIG_METHOD_Global.Matrix;
   NTargets = CONFIG_METHOD_Global.NTargets;
   NPiv = CONFIG_METHOD_Global.NPiv;

   /*  We go through each target in turn. A target can be allocated if
    *  it has at least one possible allocation in the matrix.
    */
   
   MaxTargets = 0;
   for (ITarget = 0; ITarget < NTargets; ITarget++) {
      for (IPiv = 0; IPiv < NPiv; IPiv++) {
         if (Matrix[IPiv][ITarget].PathStatus == CONFIG_METHOD_POSSIBLE) {
            MaxTargets++;
            break;
         }
      }
   }
   
   /*  Then we go through each pivot in turn. A pivot can be allocated if
    *  it has at least one possible allocation in the matrix.
    */
   
   MaxPivots = 0;
   for (IPiv = 0; IPiv < NPiv; IPiv++) {
      for (ITarget = 0; ITarget < NTargets; ITarget++) {
         if (Matrix[IPiv][ITarget].PathStatus == CONFIG_METHOD_POSSIBLE) {
            MaxPivots++;
            break;
         }
      }
   }
   
   /*  The maximum number of possible allocations is the smaller of the
    *  two numbers we've just calculated.
    */
    
   MaxAllocs = ((MaxPivots > MaxTargets) ? MaxTargets : MaxPivots);
   
   return MaxAllocs;
}

/* -------------------------------------------------------------------------- */

/*         C o n f i g  M e t h o d  R e a s s i g n  T o  S k y
 *
 *  Function:
 *     Reassigns a fibre of the specified type to a sky target.
 *
 *  Description:
 *     This routine is intended to be called after an allocation that has
 *     been made with no fibres allocated to sky. It attempts to provide
 *     precise allocation of the specified number of fibres of the specified
 *     type to sky targets. It does so by deallocating the required number
 *     of fibres and then reallocating them to sky targets. Note that it
 *     needs the allocation matrix to have allocations to sky enabled.
 *     It attempts to be a little clever about how it selects the fibres to
 *     deallocate, looking for the lowest priority fibre that can be 
 *     reallocated to a sky target, then taking into account the number of
 *     sky fibres that each fibre can reach. Note that because this can be
 *     used in a mode where fibres are reallocated one at a time, the program
 *     also needs to know the total number of fibres to be reallocated, so
 *     it can tell if the number of free fibres is sufficient or not.
 *
 *     This routine operates in one of two modes, depending on whether or
 *     not there are any sky targets in the target list. If there are no
 *     sky targets at all, then it assumes that it has only to deallocate
 *     fibres until the specified number of the specified type are free.
 *     The assumption is that a sky grid will be allocated later, and
 *     the fibres deallocated by this routine will be allocated to these
 *     sky targets.
 */

static int ConfigMethodReassignToSky (
   const CONFIG_MethodInfo *MethodInfo,
   int FibreType,                  /* (>) Type code for fibres to reallocate */
   int TotalToReallocate,          /* (>) Total # fibres to be reallocated */
   int PivotsToReallocate,         /* (>) # of fibres to reallocate this time */
   CONFIG_ReportFunction
            ReportFunction,        /* (>) Feedback routine to call */
   void *FunctionArg)              /* (>) Caller-supplied argument */
{
   /*  A CandidatePivotInfo structure holds all the details needed about a
    *  pivot that has been determined to be a candidate for deallocation
    *  (and reallocation, if there are sky targets available).
    */
   
   typedef struct CandidatePivotInfo {
      CONFIG_Boolean Allocated;  /* True if allocated at all */
      int Priority;              /* Priority of allocated object */
      int NSkies;                /* Number of sky targets it can reach */
      int Pivot;                 /* Pivot index */
      int Target;                /* Index of allocated target, if any */
   } CandidatePivotInfo;

   /*  Local variables  */

   CONFIG_Boolean Allocated;     /* True if a pivot has been allocated */
   const CONFIG_AllocInfo *AllocInfo;  
                                 /* Allocation info array of structures */
   CONFIG_Boolean AllocToObject; /* Allocated to non-sky, non-guide object */
   CONFIG_Boolean CancelFlag;    /* True if a cancel has been requested */
   CandidatePivotInfo *CandidateInfo;
                                 /* Details of candidate pivots */
   int Counter;                  /* Loop index through re-allocations */
   char Description[128];        /* What we're doing - used for reporting */
   int HighestPrio;              /* Highest priority target allocated */
   int LowestPrio;               /* Lowest priority target allocated */
   int ICand;                    /* Index through candidate pivots */
   int IPiv;                     /* Loop index through pivots */
   CONFIG_Boolean IsCandidate;   /* True if a fibre is a candidate */
   int ITarget;                  /* Loop index through targets */
   StatusType LocalStatus;       /* Inherited status - ignored */
   CONFIG_METHOD_MatrixElement **Matrix;
                                 /* Synonym for .Matrix in globals */
   int MaxNSkies;                /* Most skies that a pivot can reach */      
   int MinFibres;                /* Fewest pivots that can reach a target */
   int MinNSkies;                /* Fewest skies that a pivot can reach */
   int NCand;                    /* Number of candidate pivots */
   int NFibres;                  /* Fibres that can be allocated to a pivot */
   int NSkies;                   /* Number of skies that a pivot can reach */
   int NotAllocated;             /* Number of candidate pivots not allocated */
   int NPiv;                     /* Number of pivots */
   int NTargets;                 /* Number of targets */
   int NumCand;                  /* Temporary number of candidates */
   int NumReallocs;              /* Number of actual reallocations performed */
   const CONFIG_TargetInfo *TargetInfo;
                                 /* Target info array of structures */
   double PercentDone;           /* Percentage complete so far */
   const CONFIG_PivotInfo *PivotInfo;
                                 /* Target info array of structures */
   int Priority;                 /* Priority of pivot's allocated target */
   int* RandomArray;             /* Used to select a random pivot */
   CONFIG_Boolean SkyTargetsExist;
                                 /* True if sky targets have been specified */
   int Target;                   /* Target to which pivot was allocated */
   int ThePivot;                 /* Pivot that gets re-allocated */
   int TheTarget;                /* Sky target to which it is re-allocated */
   double Theta;                 /* Theta angle for allocation */
   int TotalNotAllocated;        /* Number of pivots of this type unllocated */
   char Type;                    /* Target type code */

   /*  This next line is unsatisfactory. It's needed because if we're being
    *  called at the end of an allocation made using the 'Oxford' algorithm,
    *  the routine ConfigMethodSetAlloc{} tests this global parameter and
    *  only records allocations as tentative. We don't want this, and changing
    *  this parameter is the easiest way to prevent it. Really, there should be
    *  a parameter added to ConfigMethodSetAlloc{} to indicate that allocations
    *  are tentative.
    */
    
   CONFIG_METHOD_Global.ParameterValues[CONFIG_METHOD_SELECT_PARM]
                                  .IntegerValue = CONFIG_METHOD_SEL_HARD;
                                  
   /*  Pick up local pointers to the various information structures whose
    *  addresses are held in the method-specific global structure. Also
    *  pick up the number of possible targets and pivots.
    */

   CandidateInfo = (CandidatePivotInfo*) 0;
   RandomArray = (int*) 0;
   Matrix = CONFIG_METHOD_Global.Matrix;
   NTargets = CONFIG_METHOD_Global.NTargets;
   NPiv = CONFIG_METHOD_Global.NPiv;
   AllocInfo = MethodInfo->AllocInfo;
   TargetInfo = MethodInfo->TargetInfo;
   PivotInfo = MethodInfo->PivotInfo;
   NumReallocs = 0;
   CancelFlag = CONFIG_FALSE;
   
   /*  Try to provide a useful description of what's happening */
   
   sprintf (Description,"Re-assigning %s fibres for sky",
                                MethodInfo->FibreText(MethodInfo,FibreType));
                                
   /*  Allocate memory to store details of candidate pivots, and the
    *  array used to randomise them if we can't pick one on the basis of
    *  our usual criteria.
    */
   
   CandidateInfo = 
          (CandidatePivotInfo*) malloc (NPiv * sizeof(CandidatePivotInfo));
   if (CandidateInfo == (CandidatePivotInfo*) 0) goto Exit;
   RandomArray = (int*) malloc (NPiv * sizeof(int));
   if (RandomArray == (int*) 0) goto Exit;
   
   /*  See if there are sky targets specified. We do things slightly 
    *  differently if there are no sky targets at all - we assume that 
    *  in this case we're just being asked to release enough fibres to
    *  be allocated to sky later. Note - the code was modified to deal
    *  with this case, and it has become somewhat more confusing as a
    *  result. There may be something to be said for handling the two
    *  cases separately.
    */
   
   SkyTargetsExist = CONFIG_FALSE;
   for (ITarget = 0; ITarget < NTargets; ITarget++) {
      Type = TargetInfo[ITarget].Type;                  
      if (MethodInfo->TargetIsSky(MethodInfo,Type)) {
         SkyTargetsExist = CONFIG_TRUE;
         break;
      }
   }
                     
   /*  Loop, each time selecting one pivot to deallocate and one sky target
    *  to allocate it to, if sky targets are available.
    */
    
   for (Counter = 0; Counter < PivotsToReallocate; Counter++) {
   
      /*  The main loop has three stages. 1) We build up a list of all
       *  the pivots that are candidates for de/reallocation. 2) Then we
       *  choose one of these candidates and deallocate it. 3) If sky
       *  targets are available, we choose a sky target and re-allocate
       *  the pivot.
       */

      /*  Stage 1 - build up the candidate list.
       *  We go through each fibre in turn. For each fibre of the right type,
       *  we see if it is allocated to a non-sky target. If it is, we see how
       *  many sky targets it could be allocated to. (We ignore the fact for
       *  the moment, that its current allocation has some effect on where it
       *  can be allocated, ruling out, for example, a sky target very close
       *  to its current non-sky target.) If it can be allocated to one or
       *  more sky targets, we record it as a candidate for reallocation,
       *  recording the number of skies it can reach and the priority of its
       *  current target.
       */

      Target = 0;
      TheTarget = 0;
      ThePivot = 0;
      MaxNSkies = 0;
      MinNSkies = NTargets;
      LowestPrio = 10;
      HighestPrio = 0;
      NotAllocated = 0;
      TotalNotAllocated = 0;
      NCand = 0;
      for (IPiv = 0; IPiv < NPiv; IPiv++) {
         if (MethodInfo->FibreTypeMatch(
                           MethodInfo,PivotInfo[IPiv].FibreType,FibreType)) {
            if (PivotInfo[IPiv].AllowAllocChange) {

               /*  This is a fibre of the right type that's allowed to be
                *  changed. See what, if anything, it's allocated to, and
                *  see how many sky targets it can be allocated to instead.
                */

               NSkies = 0;
               Allocated = CONFIG_FALSE;
               AllocToObject = CONFIG_FALSE;
               Priority = 0;
               for (ITarget = 0; ITarget < NTargets; ITarget++) {
                  if (Matrix[IPiv][ITarget].PathStatus == 
                                                    CONFIG_METHOD_ALLOCATED) {

                     /*  If the fibre is already allocated to a guide or sky,
                      *  then it simply can't be a candidate. Ignore this pivot.
                      *  If it's allocated to an object, record the object's
                      *  priority.
                      */

                     Allocated = CONFIG_TRUE;
                     Type = TargetInfo[ITarget].Type;                  
                     if (MethodInfo->TargetIsSky(MethodInfo,Type) ||
                           MethodInfo->TargetIsGuide(MethodInfo,Type)) {
                        break;
                     } else {
                        AllocToObject = CONFIG_TRUE;
                        Priority = TargetInfo[ITarget].Priority;
                        Target = ITarget;
                     }

                  } else if (Matrix[IPiv][ITarget].PathStatus == 
                                                    CONFIG_METHOD_POSSIBLE) {

                     /*  If we have a possible allocation to a sky target,
                      *  record that.
                      */

                     Type = TargetInfo[ITarget].Type;
                     if (MethodInfo->TargetIsSky(MethodInfo,Type)) NSkies++;
                  }
               }
               if (!Allocated) TotalNotAllocated++;

               /*  If this is a candidate, record its details. A target is
                *  a candidate if it is either not allocated, or is allocated
                *  to an ordinary target object (not a sky or guide target),
                *  and can reach more than one sky (there's no point
                *  deallocating it if it can't be assigned to a sky target).
                *  In the case where we are deallocating fibres in order to 
                *  have some free when we allocate sky later, then the only
                *  fibres we want to deallocate are those that are currently
                *  allocated.
                */

               if (SkyTargetsExist) {
                  IsCandidate = ((AllocToObject || !Allocated) && (NSkies > 0));
               } else {
                  IsCandidate = AllocToObject;
               }
               if (IsCandidate) {
                  CandidateInfo[NCand].Allocated = Allocated;
                  CandidateInfo[NCand].Priority = Priority;
                  CandidateInfo[NCand].NSkies = NSkies;
                  CandidateInfo[NCand].Pivot = IPiv;
                  CandidateInfo[NCand].Target = Target;
                  if (!Allocated) NotAllocated++;
                  if (NSkies > MaxNSkies) MaxNSkies = NSkies;
                  if (NSkies < MinNSkies) MinNSkies = NSkies;
                  if (LowestPrio > Priority) LowestPrio = Priority;
                  if (HighestPrio < Priority) HighestPrio = Priority;
                  NCand++;
               }
            }
         }
      }

      /*  Now, see if we have any candidates. If so, we need to pick one.
       *  If not, that's all we can do and we exit the main loop prematurely.
       */

      if (NCand <= 0) break;
      
      /*  If we are just looking to deallocate fibres for later, and we
       *  already have enough unallocated fibres (there may have been
       *  unallocated pivots when we started) then we don't need to do
       *  any more.
       */
       
      if (!SkyTargetsExist && (TotalNotAllocated >= TotalToReallocate)) break;

      /*  Stage 2 - choose from the candidates and deallocate it.
       *
       *  What we do in this stage is work on the array of possible candidates.
       *  There are a few steps, at each of which we try to reduce the number
       *  of candidates. We try to select from the candidate list, moving
       *  any selected candidate to the start of the candidate array. If
       *  we succeed, we come out of each stage with a reduced value for
       *  NCand, and the first NCand items in the candidate array are our
       *  remaining candidates.
       *
       *  The ideal is a candidate that is not yet allocated at all. If we
       *  have any of these, copy them down to the start of the candidate
       *  info array, and modify NCand accordingly. (If there are no sky
       *  targets, and we are deallocating pivots without re-allocating
       *  them, we won't have any unallocated candidates because of the
       *  criteria used to select candidates.) Note that as we reduce the
       *  number of candidates, we modify the variables such as MaxNSkies
       *  that we will use to pick between them in later stages.
       */

      if (NotAllocated > 0) {
         NumCand = NCand;
         NCand = 0;
         MaxNSkies = 0;
         MinNSkies = NTargets;
         LowestPrio = 10;
         HighestPrio = 0;
         for (ICand = 0; ICand < NumCand; ICand++) {
            if (!CandidateInfo[ICand].Allocated) {
               Priority = CandidateInfo[ICand].Priority;
               NSkies = CandidateInfo[ICand].NSkies;
               memcpy(&CandidateInfo[NCand],&CandidateInfo[ICand],
                                         sizeof(CandidateInfo[ICand]));
               if (LowestPrio > Priority) LowestPrio = Priority;
               if (HighestPrio < Priority) HighestPrio = Priority;
               if (MaxNSkies < NSkies) MaxNSkies = NSkies;
               if (MinNSkies > NSkies) MinNSkies = NSkies;
               NCand++;
            }
         }
      }

      /*  Now, we look for the candidates at the current lowest priority. If
       *  all candidates are at the same priority, we don't have to do this
       *  step.
       */

      if (HighestPrio > LowestPrio) {
         NumCand = NCand;
         NCand = 0;
         MaxNSkies = 0;
         MinNSkies = NTargets;
         for (ICand = 0; ICand < NumCand; ICand++) {
            if (CandidateInfo[ICand].Priority == LowestPrio) {
               NSkies = CandidateInfo[ICand].NSkies;
               memcpy(&CandidateInfo[NCand],&CandidateInfo[ICand],
                                         sizeof(CandidateInfo[ICand]));
               if (MaxNSkies < NSkies) MaxNSkies = NSkies;
               if (MinNSkies > NSkies) MinNSkies = NSkies;
               NCand++;
            }
         }
      }

      /*  Now, we filter on the basis of the number of skies to which the
       *  candidate pivots can be allocated. We want the ones with the 
       *  highest number of skies. (The idea here is that this is less
       *  likely to use up skies that might be hard to get to, which might
       *  turn out to be needed for a later fibre type with fewer fibres.
       *  However, it may be that this section could be dropped.
       */

      if (MaxNSkies > MinNSkies) {
         NumCand = NCand;
         NCand = 0;
         for (ICand = 0; ICand < NumCand; ICand++) {
            if (CandidateInfo[ICand].NSkies == MaxNSkies) {
               memcpy(&CandidateInfo[NCand],&CandidateInfo[ICand],
                                         sizeof(CandidateInfo[ICand]));
               NCand++;
            }
         }
      }

      /*  Finally, we may have a number of equally suitable candidates. We
       *  pick one on a random basis, copying their pivot numbers into an
       *  array, shuffling them, and picking the first.
       */

      for (ICand = 0; ICand < NCand; ICand++) {
         RandomArray[ICand] = ICand;
      }
      ConfigMethodRandomSort (RandomArray,NCand);
      ICand = RandomArray[0];
      ThePivot = CandidateInfo[ICand].Pivot;

      /*  Now, we deallocate that pivot, if it was allocated */

      PercentDone = (NumReallocs * 100.0 / PivotsToReallocate);
      if (CandidateInfo[ICand].Allocated) {
         LocalStatus = STATUS__OK;
         Target = CandidateInfo[ICand].Target;
         ConfigMethodDealloc (MethodInfo,ThePivot,Target,NULL,0,&LocalStatus);
         if (ReportFunction != NULL) {
            if ((*ReportFunction)(FunctionArg,'D',Description,ThePivot,
                              (TargetInfo[Target].Type == 'F'),
                                TargetInfo[Target].OriginalIndex,
                                  TargetInfo[Target].X,TargetInfo[Target].Y,
                                     AllocInfo[ThePivot].Theta,PercentDone)) {
               CancelFlag = CONFIG_TRUE;
            }
         }
      }

      /*  We now have to pick a sky target to which to allocate this pivot.
       *  For the moment, we use the old 'Taylor' algorithm in its simplest
       *  form - we pick the sky target that can be reached by the fewest 
       *  fibres of this type. If we're only deallocating fibres, then we
       *  can skip all this.
       */

      if (SkyTargetsExist) {
         MinFibres = NTargets;
         for (ITarget = 0; ITarget < NTargets; ITarget++) {
            Type = TargetInfo[ITarget].Type;                  
            if (MethodInfo->TargetIsSky(MethodInfo,Type)) {
               if (Matrix[ThePivot][ITarget].PathStatus == 
                                                       CONFIG_METHOD_POSSIBLE) {

                  /*  This is a sky target that can be reached by the pivot we
                   *  are going to re-allocate. Look at all the pivots and see
                   *  how many others can reach it.
                   */

                  NFibres = 0;
                  for (IPiv = 0; IPiv < NPiv; IPiv++) {
                     if (PivotInfo[IPiv].AllowAllocChange) {
                        if (Matrix[IPiv][ITarget].PathStatus == 
                                                       CONFIG_METHOD_POSSIBLE) {
                           NFibres++;
                        }
                     }
                  }
                  if (NFibres < MinFibres) {
                     MinFibres = NFibres;
                     TheTarget = ITarget;
                  }
               }
            }
         }

         /*  We now have a pivot and a sky target. Allocate the pivot to the
          *  sky target, and that's it.
          */

         Theta = Matrix[ThePivot][TheTarget].Theta;
         LocalStatus = STATUS__OK;
         ConfigMethodSetAlloc(MethodInfo,ThePivot,TheTarget,Theta,&LocalStatus);
         ConfigMethodNewAlloc(MethodInfo,ThePivot,TheTarget,Theta);
         NumReallocs++;
         PercentDone = (NumReallocs * 100.0 / PivotsToReallocate);
         if (ReportFunction != NULL) {
            if ((*ReportFunction)(FunctionArg,'A',Description,ThePivot,
                    (TargetInfo[TheTarget].Type == 'F'),
                       TargetInfo[TheTarget].OriginalIndex,
                          TargetInfo[TheTarget].X,TargetInfo[TheTarget].Y,
                            Theta,PercentDone)) {

               CancelFlag = CONFIG_TRUE;
            }
         }
      } else {
      
         /*  If there are no sky targets, and we're just releasing fibres,
          *  all we need to do is bump the what is now strictly the
          *  fibre release count rather than the reallocation count.
          */
          
         NumReallocs++;
      }  
      if (CancelFlag) break;
   }
   
Exit:;

   if (CandidateInfo) free (CandidateInfo);
   if (RandomArray) free (RandomArray);
   
   return NumReallocs;
}

/* -------------------------------------------------------------------------- */

/*             C o n f i g  M e t h o d  D e f i n e  C o n e s
 *
 *  Function:
 *     Sets up cone trees for a given priority level.
 *
 *  Description:
 *     This routine looks at the allocation matrix generates the cone 
 *     membership trees for the Oxford algorithm. These are defined by the list
 *     of all acceptable targets that are within the cone formed by the outer-
 *     most fibres that can reach each target. Since the number of fibres 
 *     inside such a cone is fixed we want to try and use the fibres within
 *     each cone on cone members before letting them break out. This procedure
 *     has the effect of smoothing out incompleteness by making sure that
 *     difficult groups of targets are tackled first.
 *
 *     If there are no targets of the desired priority then this routine
 *     returns a false value.
 *
 *     Tn slightly more detail, the main function provided by this code is
 *     to prepate the way for a call to ConfigMethodConeAllocate(), which
 *     handles allocation by attempting to allocate the most important cones
 *     first. It's worth looking at the comments for that routine in order
 *     to understand this one. This routine sets up the .ConeIndex array in
 *     CONFIG_METHOD_Global so that it lists the targets in the current
 *     priority range in oder of importance of their cones (the concept of
 *     the 'cone' is explained in detail in ConfigMethodConeAllocate().
 *     This routine arranges things so that .ConeIndex lists the cones
 *     in the following order:  Cones with Fiducials at their apex; cones
 *     with more targets than fibres that can reach them (in decreasing order
 *     of excess targets); cones with fewer targets than fibres that can 
 *     reach them (in increasing order of excess fibres); cones whose apex
 *     targets are close to the center of the field; and finally, cones where
 *     the apex target is outside the field or outside the current priority
 *     range.
 *
 *     This routine also has the job of filling up the .TargetMatrix array
 *     also held in CONFIG_METHOD_Global. In particular, the lists of pivots
 *     and targets for each cone under consideration have to be filled up.
 *     (Note: each cone is defined by the target at its apex; similarly,
 *     each target is at the apex of a cone. This means that the terms
 *     'target' and 'cone' are sometimes used interchangably. For example,
 *     the .TargetMatrix array is really describing the cones for the targets -
 *     it lists each target in each cone, and the pivots that can reach 
 *     those targets.)
 *
 *  Prerequisites:
 *     The global .Matrix must show which allocations are possible given
 *     the various physical constraints. This routine uses the PivotInfo
 *     and TargetInfo items in the MethodInfo structure. The globals
 *     .IBad and .ConeIndex must have been allocated. The global .TargetMatrix
 *     needs to have been initialised, with the various unchanging details
 *     about the targets having been set up. This routine fills in the
 *     NConeTargets and ConeTargets arrays in .TargetMatrix.
 */

static CONFIG_Boolean ConfigMethodDefineCones (
   const CONFIG_MethodInfo *MethodInfo,
   int LPrio,                    /* (<) The desired low priority level */
   int HPrio)                    /* (<) The desired high priority level */
{

/*  A note on IBad and how it is used to sort ConeIndex:
 *
 *  The global ConeIndex and IBad pointers deserve some detailed commenting. 
 *  When the method code is initialised, memory is allocated for two arrays
 *  whose addresses are then held in ConeIndex and IBad. IBad is used only
 *  within this routine (ConfigMethodDefineCones()), which uses it to set up
 *  the ConeIndex array. (IBad could have been malloc'ed and free'd within
 *  ConfigMethodDefineCones(), but that isn't how it was coded.)
 *
 *  The code calculates an IBad value for each target. This is really a value
 *  that applies to the cone formed by the target. A cone with a higher value
 *  of IBad needs to be be allocated before a cone with a lower value of IBad.
 *  The IBad value is calculated as the number of targets in the cone minus
 *  the number of fibres in the cone.
 *
 *  There are some special cases. Cones with a fiducial target at their apex
 *  are given a very high IBad value. If these cones are handled first, this
 *  will have the effect of giving priority to that fiducial target. (The 
 *  value chosen is (total number of targets + 1), which will be higher than
 *  any value calculated normally for the cone as (#targets - #pivots).)
 *
 *  Targets that are close to the center are treated specially. They are
 *  given a fixed IBad value that is lower than any of the other targets
 *  just described. The criterion used to determine what is meant by
 *  'close to the center' in this case is calculated as follows:
 *  If you take a fibre at its maximum bend angle, you can calculate how
 *  far its tip is from the center of the field. This distance is used
 *  by the code. Targets that are closer to the center than this distance
 *  are treated as being 'close' to the center. Targets this close to the
 *  center can be reached by a lot of fibres, and it speed the algorithm
 *  up if it doesn't have to worry about them as much as the others. (The
 *  value chosen is (-#Pivots).)
 *
 *  Cones that contain no targets are given a fixed IBad value that is lower
 *  than those close to the center. This might seem silly - surely they
 *  at least contain the target at their apex? Not if that target is outside
 *  the field! Moreover, this is all done in a loop that starts by considering
 *  only high priority targets and then starts to include lower priority
 *  targets, so a cone may contain no targets in the priority range being
 *  considered in the current iteration. (The fixed IBad value used in this
 *  case is (-(#Pivots + 1).)
 *
 *  Having calculated the IBad value for each fibre, the code then sorts
 *  these IBad values and stores the result of the sort in ConeIndex, in
 *  such a way that the first element of ConeIndex holds the index number in
 *  the IBad array (which is the same as the index number allocated to the
 *  original target) of the target with the highest IBad index as
 *  calculated above.
 *
 *  (Actually, it's a trifle more confusing than that. Evidently the code
 *  wanted to make use of an existing sort routine that sorted into ascending
 *  order, but wanted the targets with the highest difficulty index to end
 *  up being referenced by first elements in the ConeIndex array. For this
 *  reason, after calculating the values as described above, the code
 *  actually negates them and then sorts them into ascending order using
 *  the index1() function. Its also made messy by the fact that index1()
 *  appears to have been automatically generated from Fortran and uses
 *  Fortran indexing, so there's a deal of +1 and -1 operations introduced
 *  to the code that uses ConeIndex in order to get around this.)
 *
 *  This routine declares a local pointer called Index and then sets
 *  Index = CONFIG_METHOD_Global.ConeIndex at the start of the routine,
 *  so it's worth remembering that 'Index' actually refers to the global
 *  array .ConeIndex.
 */
 
   /*  Local variables  */

   CONFIG_Boolean CanAllocate;   /* True if there are possible allocations */
   double DeltaX;                /* Distance in X from target to pivot */
   double DeltaY;                /* Distance in Y from target to pivot */
   double DeltaT;
   int TargetNP;                 /* Loop index through pivots */
   int ITarget,JTarget;          /* Loop index through targets */
   double Stm;                   /* sin(alpha_max */
   double Phi,TRad,TTarget;      /* cone angle */
   CONFIG_METHOD_TargetElement *TargetMatrix;
   CONFIG_METHOD_MatrixElement **Matrix;
                                 /* Local synonym for .TargetMatrix */
   int NonSkyCount;              /* Number of non-sky targets in prio range */
   int NPiv;                     /* Number of pivots */
   int NTargets;                 /* Number of targets */
   const CONFIG_TargetInfo *TargetInfo;/* Target info array of structures */
   const CONFIG_PivotInfo *PivotInfo;/* Target info array of structures */
   long TargetX;                 /* X-coordinate of target */
   long TargetY;                 /* Y-coordinate of target */
   char TargetType;
   int TargetCount,Count;
   int *IBad;
   int *Index;                   /* Local synonym for .ConeIndex */
   
   /*  Pick up local pointers to the various information structures whose
    *  addresses are held in the method-specific global structure. Also
    *  pick up the number of possible targets and pivots.
    */

   Matrix = CONFIG_METHOD_Global.Matrix;
   TargetMatrix = CONFIG_METHOD_Global.TargetMatrix;
   NTargets = CONFIG_METHOD_Global.NTargets;
   NPiv = CONFIG_METHOD_Global.NPiv;
   PivotInfo = MethodInfo->PivotInfo;
   TargetInfo = MethodInfo->TargetInfo;
   Index = CONFIG_METHOD_Global.ConeIndex-1; /* See notes to indexi() */
   IBad = CONFIG_METHOD_Global.IBad-1;
   CanAllocate = CONFIG_FALSE;

   /*  We go through targets list to see if there are any within the
    *  priority range. Set those outside this range to have cones of
    *  zero size so that we ignore them later without having to check
    *  the priorities.  Also consider targets to be ignored and which
    *  are outside the field. Note that targets outside the field will
    *  have their priority set to CONFIG_PRIO_INVALID.
    */

   TargetCount = 0;
   for (ITarget = 0; ITarget < NTargets; ITarget++) {

     if ((TargetInfo[ITarget].Priority >= LPrio) && 
         (TargetInfo[ITarget].Priority <= HPrio) &&
         (TargetInfo[ITarget].Priority != CONFIG_PRIO_INVALID)) {
         TargetCount++;
         TargetMatrix[ITarget].NConeTargets = 1;
         TargetMatrix[ITarget].ConeTargets[0] = ITarget;
     } else {
       TargetMatrix[ITarget].NConeTargets = 0;
     }
   }
                               
   /* Quit here if there was nothing doing */
   
   if (TargetCount == 0) return(CanAllocate);


   /*  If we are leaving pivots for skys (FibresForSky != 0), then we
    *  want to know if there are any non-sky targets in this priority 
    *  range. If there aren't, then we don't bother setting up the cones.
    *  It isn't clear to me (KS) that this is the best thing to do - if
    *  there are high priority sky targets, we might as well allocate them,
    *  surely? (TJF fixed an apparent bug in this loop which caused it to 
    *  return without setting up the cones if the *first* target in the
    *  priority range was a sky - this surely wan't the intention of the
    *  original code.)
    */
   
   NonSkyCount = 0;
   if (CONFIG_METHOD_Global.FibresForSky !=0){
       for (ITarget = 0 ; ITarget < NTargets; ITarget++){
           if (TargetInfo[ITarget].Priority >= LPrio && 
               TargetInfo[ITarget].Priority <= HPrio ){
               if (TargetInfo[ITarget].Type != 'S') NonSkyCount++;
           }
       }
       if (NonSkyCount == 0) return(CanAllocate);
   }
   
   /*  This code is hard to understand without a diagram. Conceptually, you 
    *  take each target in turn and determine the two extreme pivots that
    *  can access it. Ie you draw a line from the circle of pivots to the
    *  target such that this represents a fibre at the maximum bend angle.
    *  Then you draw another line from the circle of pivots to the target
    *  representing a fibre at the maximum bend angle going the other way.
    *  Fibres outside this cone can't be allocated to the target because to
    *  do so would exceed the allowed bend angle. You then look through all
    *  the other targets seeing if they are inside the cone or outside it.
    *  If a target is inside the cone, it is added to the list of targets
    *  held in the ConeTargets array associated with the original target.
    *
    *  What the code actually does is more or less as follows. Draw a circle
    *  to represent the radius at which the pivots are mounted. Pick at target 
    *  at a given position within this circle. When I did this diagram I
    *  picked a target in the upper right quadrant of the circle, and it
    *  may make it easier if you do too.  Find the two positions on
    *  the target circle at which a fibre would have its pivot if when 
    *  assigned to the target it would be at exactly the maximum bend angle.
    *  Draw one of these fibres from the pivot point through the target and
    *  project it out so that it passes the center of the pivot circle.
    *  Draw the normal from this line to the center of the circle. Call the
    *  center of the circle C, the Pivot position P and the Target position
    *  T. Call the point at which the normal intersects the fibre line N.
    *  So the angle CNP (which is also CNT, of course) is a right angle.
    *  The angle CPN (also CPT) is the maximum bend angle for the fibre.
    *  Now take the other pivot point, call it Q and draw the line from it
    *  to the target at T. You can now see the cone (actually, its a sector)
    *  defined by this target position. It is the area between the lines PT 
    *  and QT and the pivot circle. Any target within this cone is a target
    *  we want to include in our ConeTargets array.
    *
    *  Draw the line from the center of the pivot circle (C) through the 
    *  target position (T) and extend it until it intersects the pivot
    *  circle at a point we'll call R (midway between Q and P, of course).
    *
    *  If you've actually drawn this, you should see that the vertex angle
    *  of the cone (between TQ and TP) is equal to twice the angle CTN.
    *  The code works out this angle (CTN) and calls it Phi. CT is the distance
    *  from the center of the circle to the target, and is held already in
    *  the target information matrix as TargetR. The distance CN is calculated
    *  by this code as 'Stm'. I'm not completely happy with this calculation:
    *  CPN is also a right angled triangle with a hypotenuse given by the
    *  radius of the pivot circle and we already know that the angle
    *  CPN is the maximum pivot bend angle. The code seems to assume that
    *  the radius of the pivot circle (CP) is the same as the maximum fibre
    *  extension, which is close but not quite correct. Anyway, the code
    *  calculates CN ('Stm' here) on that basis. You can draw a circle
    *  of radius CN around the center of the circle and any target in that
    *  circle (ie any target for which TargetR > Stm) is given special
    *  treatment by the code - and at the moment I'm not quite sure why.
    *
    *  Having done that, we know CT (it's TargetR) and CN, so we can work
    *  out the angle CTN (here 'Phi'). We now know the cone's vertex angle.
    *
    *  Now, take another target at a point we can call U. I've picked one
    *  within the cone in the upper half of the cone, in the sector between
    *  TR and TQ. Draw the x-axis (through C) and the normal from it to U. 
    *  Call the point where this normal intersects the x-axis W. Draw the 
    *  line through T that runs parallel to the x-axis, and let the point 
    *  where this intersects the line UW be Z.  Draw the line TU. Now we've 
    *  got all the diagram we need.
    *
    *  Now look at the triangle TUZ. The angle TZU is a right angle. The
    *  angle ZTU can be calculated from the sides ZU (which is the difference
    *  in y-coordinates of the two targets) and ZT (which is the difference
    *  in x-coordinates of the two targets). What we really want is the
    *  angle RTU. If this is less than Phi, then U is within the cone.
    *  We're almost there. We know the angle ZTR, because this is the same
    *  as the angle WCT and is the basic target angle for the target T
    *  and is held as TTarget. So the code calls RTU 'DeltaT' and calculates
    *  it as atan2(DeltaY,DeltaX) - TTarget. Then if this is less than Phi,
    *  we add the target at U to the list of cone targets.
    *
    *  Also note that the code checks to see if the target at U is further
    *  out radially from the center - if it isn't, it can't be in the cone.
    *  This is the test (TargetMatrix[JTarget].TargetR >= TRad)
    *
    *  It took me a while to understand this from the code, and I was damn
    *  well going to write it down once I'd done so! (KS).
    */
    
   Stm = sin(MethodInfo->MaxPivAngle(MethodInfo))*
             MethodInfo->MaxExtension(MethodInfo);
   Count = 0;
   for (ITarget = 0; ITarget < NTargets; ITarget++) {

     /*  This number is the default IBad number for targets that are at the
      *  apex of cones that have no cone targets. The intent is that such
      *  targets have a low difficulty index and so their allocation is given
      *  a low priority by the iterative algorithm. This number needs to be 
      *  lower than the lowest possible value for 'Number of cone targets
      *  minus number of possible pivots' which is used as the difficulty
      *  index for ordinary targets.  So we just make it the negative of
      *  some number that is greater than then total number of pivots.
      */

     IBad[ITarget+1] = -(NPiv + 1);
     
     if (TargetMatrix[ITarget].NConeTargets !=0 ) {
       Count++;
       TargetCount = 0;
       TargetX = TargetInfo[ITarget].X;
       TargetY = TargetInfo[ITarget].Y;
       TargetType = TargetInfo[ITarget].Type;
       TRad = TargetMatrix[ITarget].TargetR;
       
       /*  The (TRad > Stm) test here has the effect that this code is only
        *  executed for targets a certain distance from the center of the 
        *  field.
        */
        
       if (TRad > Stm) {
         TTarget = TargetMatrix[ITarget].TargetT;
         TargetNP = TargetMatrix[ITarget].NumPivots;
         Phi = asin(Stm/TRad);
         for (JTarget = 0; JTarget < NTargets; JTarget++) {
           if (TargetMatrix[JTarget].NConeTargets !=0 ) {
             if (TargetInfo[JTarget].Type == TargetType) {
               if (TargetMatrix[JTarget].TargetR >= TRad) {
                 if (TargetMatrix[JTarget].NumPivots  <= TargetNP) {
                   if (JTarget != ITarget) { 
                     DeltaX = TargetInfo[JTarget].X - TargetX;
                     DeltaY = TargetInfo[JTarget].Y - TargetY;
                     if (DeltaX == 0 && DeltaY == 0) DeltaX = 0.00000001;
                     DeltaT = atan2(DeltaY,DeltaX) - TTarget;
                     if (abs(DeltaT) <= Phi) {
                       TargetMatrix[ITarget].ConeTargets[TargetCount]=JTarget;
                       TargetCount++;
                     }
                   }
                 }
               }
             }
           }
         }
         TargetMatrix[ITarget].ConeTargets[TargetCount] = ITarget;
         TargetCount++;
         TargetMatrix[ITarget].NConeTargets = TargetCount;
         IBad[ITarget+1] = TargetCount -TargetNP;
         
       } else {
       
         /*  NPiv is the IBad number for targets that are close to 
          *  the center of the field. The code wants to assign these a
          *  difficulty index that is close to that of the targets with
          *  no targets included in their cones, but is slightly higher.
          *  Those were set to -(NPiv + 1), so we make these just -NPiv.
          *
          *  Setting the cone as if it contains only one target, namely 
          *  the target at its apex is clearly not correct, but it does
          *  speed up the allocation considerably. These central
          *  cones contain so many pivots and targets that they really
          *  aren't worth handling in the usual way.
          */
          
         IBad[ITarget+1] = -NPiv;
         TargetMatrix[ITarget].NConeTargets = 1;
         TargetMatrix[ITarget].ConeTargets[0] = ITarget;
         
       }
     }
   }
   
   /*  We now set any fiducial targets to a very high difficulty index, 
    *  so that they get given priority treatment in the allocation. The
    *  number chosen needs to be higher than the value of 'Number of cone
    *  targets - number of possible pivots' for any ordinary target, so we
    *  set it to a value just higher than the total number of targets.
    */
    
   for (ITarget=0 ; ITarget < NTargets ; ITarget++) {
     if (TargetInfo[ITarget].Type == 'F') IBad[ITarget+1] = NTargets + 1;
   }
   
   /*  This is the loop where the code flips the sign of the IBad values
    *  to allow it to use indexi() to sort them. See comments about IBad
    *  in the comments for the CONFIG_METHOD_GlobalType definition at the
    *  start of the file. I think the best thing is to try to forget this
    *  as you read the code, and just remember that the effect of this
    *  sign flip and the indexi() call combined is to get the first element
    *  of the Index array indicating the target that up to this point in the
    *  code had the highest difficulty index (ie the IBad value as computed
    *  in the code up to now.)  I'm inclined to think that I'd have done
    *  better to fix up indexi() so it works better in this program instead
    *  of trying to understand and comment the contortions the code goes through
    *  to use it as it stands... (KS).
    */
      
   for (ITarget=0 ; ITarget < NTargets ; ITarget++) {
     IBad[ITarget+1] = -IBad[ITarget+1];
   }
   indexi(NTargets,IBad,Index);
   
   /* Cut down the value of each index by one here to make up for 
      fortran style indexing in indexi */
   for (ITarget=0 ; ITarget < NTargets ; ITarget++) {
     Index[ITarget+1]--;
   }
      
   /* GBD:Taken a decison here NOT to rearrange the target lists as
      we did in the original code, since we can't get at the contents
      of the TargetInfo structure to make everything
      consistent. Instead I will store Index here in the
      CONFIG_METHOD_Global structure and try to keep track of things
      that way. This probably will waste more time in terms of looking
      up pointers repeatedly than it will save by not re-ordering...*/

   /* Should have finished this bit now */

   CanAllocate = CONFIG_TRUE;

   return(CanAllocate);
}

/*  -------------------------------------------------------------------------

                 I  n  d  e  x  i 
                                                                            
 *  This horror of a routine seems to be a C reworking of some Fortran code
 *  that implements the idea of the Numerical Recipies routine of a similar
 *  name. It takes an array (Array) with N entries, and generates an index
 *  array (Index), such that the elements of Index give the numerical order
 *  of the entries in Array. Specifically, Array[Index[1]] is the lowest
 *  valued element of Array, Array[Index[2]] is the next lowest, and so on
 *  up to Array[Index[N+1]] which is the highest valued element of Array.
 *  Note in particular that Index needs to have N+1 elements, but that
 *  Index[0] is unused. This is used by the rest of the code, where this
 *  routine is used to generate the global array CONFIG_METHOD.ConeIndex.
 *  The code contrives to call this routine (indexi) with an int pointer
 *  that has been set to (CONFIG_METHOD.ConeIndex - 1) ie to the int *before*
 *  the ConeIndex array. This makes it work OK, but results in some confused
 *  code in places.  [Note added by KS.]
 *
 */
static void indexi(int N, int *Array, int *Index){

      int ir,l,j,q,indxt,i;
      for (j = 1; j<=N; j++){
        Index[j] = j;
      }
      l=N/2+1;
      ir = N;
Label10:;
      if (l > 1) {
        l--;
        indxt=Index[l];
        q = Array[indxt];
      } else {
        indxt = Index[ir];
        q = Array[indxt];
        Index[ir]=Index[1];
        ir--;
        if (ir == 1) {
          Index[1] = indxt;
          return;
        }
      }
      i=l;
      j=l+l;
Label20:;
      if (j <= ir){
        if (j < ir){
          if (Array[Index[j]] < Array[Index[j+1]]){
            j++;
          }
        }
        if (q < Array[Index[j]]) {
          Index[i] = Index[j];
          i=j;
          j=j+j;
        } else {
          j = ir+1;
        }
        goto Label20;
      }
      Index[i] = indxt;
      goto Label10;
}

/* -------------------------------------------------------------------------- */

/*             C o n f i g  M e t h o d  C o n e  A l l o c a t e
 *
 *  Function:
 *     First Allocation pass using cones ordering - iterative allocation.
 *
 *  Description:
 *     This routine does an initial allocation pass based on the cone
 *     ordering set up by a call to ConfigMethodDefineCones(). This will
 *     have defined cones for all the targets in the current priority
 *     range.
 *
 *     The crucal point is that the Oxford algorithm is optimised for cases
 *     (such as the 2dF redshift survey) where the number of targets in a field
 *     is more or less the same as the number of pivots.
 *
 *     Conceptually, for each target the code examines a 'cone' formed by the 
 *     two extreme positions for fibres that can be allocated to that target.
 *     (The constraint here is just the maximum bend angle for each fibre, so
 *     targets close to the outside have a very narrow cone - very few possible
 *     fibres - and targets closer to the center have a wider cone.) The code
 *     counts the number of targets included in such a cone.  A cone that has
 *     about as many targets as fibres is probably going to be more or less OK
 *     in terms of allocation difficulty is concerned. A cone that has more 
 *     targets than fibres is going to be a problem - some of these will have
 *     to be unallocated. More to the point, if we have such a cone we want 
 *     to try to be sure that as many as possible of the fibres that can be
 *     allocated to targets within that cone are allocated to those targets.
 *     If we were to allocate them to other targets, perhaps to targets outside
 *     the cone in question and closer to the field center, then we're going
 *     to have wasted them. (Unless they were allocated in this way because 
 *     there was a cone with an apex closer to the field center that had an
 *     even greater mismatch between pivots and targets. Remember that most
 *     pivots and targets will be in a number of different cones.)
 *
 *     So the idea of the Oxford algorithm is to work out which cones are the
 *     most important to deal with first - the ones that have the greatest
 *     excess of targets over pivots. This routine assumes that a previous
 *     call to ConfigMethodDefineCones() has set up the index array 
 *     CONFIG_METHOD_Global.ConeIndex so that it lists the targets under
 *     consideration in order of the importance of their cones - it will
 *     deal with the cones in the order they appear in this array. See the
 *     comments for ConfigMethodDefineCones() for more details as to how this
 *     is set up. The other important arrays set up by ConfigMethodDefineCones()
 *     are the set of arrays, one for each cone, held in the global structure
 *     CONFIG_METHOD_Global.TargetMatrix that list the targets included in
 *     each cone. This is the array ConeTargets[] associated with each target
 *     detailed in that global structure. So the code takes each cone in turn,
 *     starting with the most important, and then looks at each target in that
 *     cone, working through this ConeTargets[] array.
 *
 *     Once the code has settled on a target to allocate, it looks at the
 *     list of fibres that can reach this target. This is held in the Pivot[]
 *     array associated with each target in that global .TargetMatrix structure.
 *     It assigns the to the target the first pivot that it finds in that
 *     array that can be assigned (not all fibres that reach can be assigned,
 *     because of collisions with previously assigned fibres).
 *
 *     The allocation that results from this routine can then be refined
 *     in the swap phase of the iterative algorithm, in the routine
 *     ConfigMethodBlockSwapAllocate().
 *
 *  Prerequisites:
 *     This routine needs to be called immediately after a call to
 *     ConfigMethodDefineCones().
 */

static void ConfigMethodConeAllocate(
   const CONFIG_MethodInfo *MethodInfo,
   CONFIG_ReportFunction
            ReportFunction,          /* (>) Feedback routine to call */
   void *FunctionArg)                /* (>) Caller-supplied argument */
{

   /*  Local variables  */

   CONFIG_Boolean CanAllocate;   /* True if there are possible allocations */
   CONFIG_Boolean CancelFlag;    /* True if GUI requests a cancel */  
   int IPiv;                     /* Loop index through possible pivots */
   int JPiv;                     /* Actual pivot number corresponding to IPiv */
   int ITarget;                  /* Target at apex of current cone */
   int JTarget;                  /* Loop index through targets */
   int KTarget;                  /* Target we are trying to allocate */
   CONFIG_METHOD_TargetElement 
                  *TargetMatrix; /* Local synonym for .TargetMatrix */
   CONFIG_METHOD_MatrixElement 
                       **Matrix; /* Local synonym for .Matrix */
   int NPiv;                     /* Number of pivots */
   int NTargets;                 /* Number of targets */
   int LTarget;                  /* Target to which JPiv is allocated */
   const CONFIG_TargetInfo 
                    *TargetInfo; /* Target info array of structures */
   const CONFIG_PivotInfo 
                     *PivotInfo; /* Pivot info array */
   const CONFIG_AllocInfo 
                     *AllocInfo; /* Allocation table info */
   CONFIG_Boolean Retry;         /* Used to control re-allocation loop */  
   CONFIG_Boolean SkyTest;       /* Global Sky flag value */
   int TargetCount;              /* Acts as loop index through cones */
   int UnallocCount;             /* Number of unallocated fibres ! */
   int Count;                    /* Number of cones of interest */
   int *Index;                   /* Local synonym for .ConeIndex */
   double Theta;                 /* Theta angle for an allocation */
   double PercentDone;           /* Percent complete for report to GUI */
   StatusType LocalStatus;       /* Local inherited status variable */
   
   /*  Pick up local pointers to the various information structures whose
    *  addresses are held in the method-specific global structure. Also
    *  pick up the number of possible targets and pivots.
    */

   Matrix = CONFIG_METHOD_Global.Matrix;
   TargetMatrix = CONFIG_METHOD_Global.TargetMatrix;
   NTargets = CONFIG_METHOD_Global.NTargets;
   NPiv = CONFIG_METHOD_Global.NPiv;
   PivotInfo = MethodInfo->PivotInfo;
   TargetInfo = MethodInfo->TargetInfo;
   AllocInfo = MethodInfo->AllocInfo;
   Index = CONFIG_METHOD_Global.ConeIndex;
   CanAllocate = CONFIG_FALSE;
   CancelFlag = CONFIG_FALSE;
   LocalStatus = STATUS__OK;
   
   /*  Go through Allocation list to see if there are any free fibres. If
    *  not, there's nothing useful to do here. 
    */
   
   UnallocCount=0;
   for (IPiv = 0; IPiv < NPiv; IPiv++){
     if ((!MethodInfo->FibreIsBroken(MethodInfo,PivotInfo[IPiv].FibreType))
        && (PivotInfo[IPiv].AllowAllocChange)
        && (AllocInfo[IPiv].State == 'U')) UnallocCount++;
   }
   if (UnallocCount == 0) return;

   /*  ConfigMethodDefineCones() has set up the TargetMatrix structure so
    *  that the ConeIndex[] array - which we will use in the main loop -
    *  lists cones in the following order: Cones with Fiducials at their 
    *  apex; cones with more targets than fibres that can reach them (in
    *  decreasing order of excess targets); cones with fewer targets than
    *  fibres that can reach them (in increasing order of excess fibres);
    *  cones whose apex targets are close to the center of the field;
    *  and finally, cones where the apex target is outside the field or
    *  outside the current priority range. (Note that Index in this code
    *  points to the global ConeIndex array.)
    *
    *  Of these, all bar the last category - cones outside the field or
    *  the priority range - will be processed by this routine. So first, we
    *  work out how many cones we have to deal with. ConfigMethodDefineCones()
    *  sets the NConeTargets value for cones in the last category to zero, so
    *  this loop works out how many cones we have to deal with by counting
    *  those cones flagged as having a non-zero number of cone targets.
    */

   Count = 0;
   for (TargetCount = 0; TargetCount < NTargets; TargetCount++) {
     if (TargetMatrix[TargetCount].NConeTargets != 0){
       Count++;
     }
   }
   
   SkyTest = CONFIG_METHOD_Global.SkyFlag;
   
   /*  Check to see if we're in the auto-deallocation loop. If we are, the 
    *  code for this routine is made much more complicated. So much so,
    *  that it has been split into two cases. So first we have the simple
    *  allocation loop used when we are not auto-deallocating.
    */
   
   if (CONFIG_METHOD_Global.ParameterValues[CONFIG_METHOD_SELECT_PARM]
                             .IntegerValue != CONFIG_METHOD_SEL_BROKEN) {
      
      /*  This is the simple case, where we don't have to worry about auto-
       *  deallocation. We work through all the cones we are interested in. 
       *  There are 'Count' of these.
       */
    
      for (TargetCount = 0; TargetCount < Count; TargetCount++) {
   
         /*  The Index array tells us which target is at the apex of the cone
          *  we should work on next. So this gives us the target we will try
          *  to allocate next.
          */

         ITarget = Index[TargetCount];

         /*  PercentDone here denotes percentage of attempts, not allocations */

         PercentDone = 100.0 * (TargetCount+1) / (Count+1);
         
         /*  Since we're dealing with this cone, we assume it has a +ve
          *  number of targets in it. (If it doesn't, the loop just isn't
          *  executed, but it would indicate a problem with the cone
          *  definition code.) We work through the list of targets in the
          *  cone, attempting to allocate each in turn.
          */
          
         for (JTarget = 0; JTarget < TargetMatrix[ITarget].NConeTargets; 
                                                                  JTarget++) {
            KTarget = TargetMatrix[ITarget].ConeTargets[JTarget];
            
            /*  KTarget is now the next target we want to try and allocate. We
             *  stop allocating if we've reached the number of fibres we were
             *  supposed to allocate for sky. (If we're in a sky allocation
             *  phase, we continue, of course.)
             */
            
            if ((SkyTest && TargetInfo[KTarget].Type == 'S') || 
                  (UnallocCount > CONFIG_METHOD_Global.FibresForSky)) {
                  
               /*  We work through the possible pivots that can reach targets
                *  in the cone. Note that ConfigMethodSortPivots() sorts these
                *  so that the pivots are listed in order of distance from
                *  the center of the cone. It may be that we should try to
                *  sort the targets in the same way? At any rate, these are all
                *  pivots that can reach any target in the cone, but some will
                *  fail the allocation test because they will collide with 
                *  previously allocated fibres. We keep trying until we find
                *  one that can be allocated.
                */
               
               for (IPiv = 0 ; IPiv < TargetMatrix[KTarget].NumPivots; IPiv++) {
                  JPiv = TargetMatrix[KTarget].Pivot[IPiv];
                  if (AllocInfo[JPiv].State == 'U') {
                     if (Matrix[JPiv][KTarget].PathStatus == 
                                                   CONFIG_METHOD_POSSIBLE){
                        Theta = Matrix[JPiv][KTarget].Theta;
                        
                        /*  This allocation looks possible. It is flagged in 
                         *  the main matrix as OK, and the fibre is unallocated.
                         *  So we attempt an allocation using a call to 
                         *  ConfigMethodSetAlloc(). If that fails, it means
                         *  it must collide with an already allocated fibre,
                         */
                         
                        LocalStatus = STATUS__OK;
                        ConfigMethodSetAlloc(MethodInfo,JPiv,KTarget,
                                           Theta,&LocalStatus);
                        if (LocalStatus == STATUS__OK) {
                        
                           /*  Succeeded. Report the allocation, and break out
                            *  of the loop for this target.
                            */
                            
                           if (ReportFunction != NULL) {
                              if ((*ReportFunction)(FunctionArg,'A',
                                    "Allocating fibres",JPiv,
                                    (TargetInfo[KTarget].Type == 'F'),
                                    TargetInfo[KTarget].OriginalIndex,
                                    TargetInfo[KTarget].X,
                                    TargetInfo[KTarget].Y,
                                    Theta,PercentDone)) {
                                 CancelFlag = CONFIG_TRUE;
                                 return;
                              } 
                           } 
                           UnallocCount--;
                           break;
                        }
                     }
                  }
               }
            }
         }
      }

   } else {
   
      /*  This is the case where we're in the auto-deallocation loop. The
       *  code is more or less the same as the simpler case above, but
       *  is complicated by the possibility that we can deallocate some
       *  less important fibres (sky, or low priority targets) in order to
       *  allocate something more important.
       */
   
      /*  Now we work through all the cones we are interested in. There are
       *  Count of these.
       */

      for (TargetCount = 0; TargetCount < Count; TargetCount++) {

         /*  The Index array tells us which target is at the apex of the cone
          *  we should work on next. So this gives us the target we will try
          *  to allocate next.
          */

         ITarget = Index[TargetCount]; 
         PercentDone = 100.0 * (TargetCount+1) / (Count+1);
         
         /*  We may go through this loop again, if we end up deallocating a
          *  low-priority target to see if it can help. Normally, we
          *  only go through this once, hence the clearing of the Retry flag
          *  at the start of the loop. Deallocating a low-priority target 
          *  will set the Retry flag true again.
          */
          
         Retry = CONFIG_TRUE;

         while (Retry) {
            Retry = CONFIG_FALSE;
            for (JTarget = 0; JTarget < TargetMatrix[ITarget].NConeTargets ; 
                                                                   JTarget++) {
               KTarget = TargetMatrix[ITarget].ConeTargets[JTarget];
               
               /*  KTarget is now the next target we want to try and allocate.
                *  The -2 code is normally set to indicate that a target has 
                *  been deallocated because its allocation was found to be 
                *  invalid - this is used by ConfigMethodAutoDeallocate() to
                *  flag the deallocations it makes. In this auto-recover
                *  mode, these are the only targets we are interested in.
                */
                
               if (TargetMatrix[KTarget].TargetPivot == -2) {
               
                  /*  KTarget is now the next target we want to try and 
                   *  allocate. We stop allocating if we've reached the number
                   *  of fibres we were supposed to allocate for sky. (If 
                   *  we're in a sky allocation phase, we continue, of course.)
                   */
            
                  if ((SkyTest && TargetInfo[KTarget].Type == 'S') || 
                       (UnallocCount>CONFIG_METHOD_Global.FibresForSky)){
                       
                     /*  We work through the possible pivots that can reach
                      *  targets in the cone. See simpler loop coments above
                      *  for more details.
                      */
                       
                     for (IPiv = 0; IPiv < TargetMatrix[KTarget].NumPivots; 
                                                                     IPiv++) {
                        JPiv = TargetMatrix[KTarget].Pivot[IPiv];
                        if (AllocInfo[JPiv].State == 'U') {
                           if (Matrix[JPiv][KTarget].PathStatus == 
                                              CONFIG_METHOD_POSSIBLE) {
                                              
                              /*  This allocation looks possible. It is 
                               *  flagged in the main matrix as OK, and the 
                               *  fibre is unallocated. So we attempt an 
                               *  allocation using a call to 
                               *  ConfigMethodSetAlloc(). If that fails, 
                               *  it means it must collide with an already 
                               *  allocated fibre,
                               */
                               
                              Theta = Matrix[JPiv][KTarget].Theta;
                              LocalStatus = STATUS__OK;
                              ConfigMethodSetAlloc(MethodInfo,JPiv,KTarget,
                                           Theta,&LocalStatus);
                              if (LocalStatus == STATUS__OK) {
                                 if (ReportFunction != NULL) {
                                    if ((*ReportFunction)(FunctionArg,'A',
                                          "Allocating fibres",JPiv,
                                          (TargetInfo[KTarget].Type == 'F'),
                                          TargetInfo[KTarget].OriginalIndex,
                                          TargetInfo[KTarget].X,
                                          TargetInfo[KTarget].Y,
                                          Theta,PercentDone)) {
                                       CancelFlag = CONFIG_TRUE;
                                       return;
                                    } 
                                 } 
                                 UnallocCount--;
                                 
                                 /*  We've allocated a fibre to KTarget. We can
                                  *  break out of the IPiv loop (and the Retry
                                  *  loop as well).
                                  */
                                  
                                 break;
                              }
                           }
                        } else {
                        
                           /*  At this point, we've found that JPiv is an
                            *  allocated fibre. If it's a low-priority or sky
                            *  fibre, we consider deallocating it to see if that
                            *  helps.
                            */
                            
                           LTarget = AllocInfo[JPiv].TargetIndex;
                           if (TargetInfo[LTarget].Type == 'S' ||
                                    TargetInfo[LTarget].Priority 
                                        <= CONFIG_METHOD_Global.BonusPrio) {
                              if (Matrix[JPiv][KTarget].PathStatus ==
                                                CONFIG_METHOD_POSSIBLE) {
                                 
                                 /*  We have checked that JPiv can be allocated
                                  *  to KTarget, so we deallocate it from its
                                  *  current low-priority target.
                                  */
                                   
                                 LocalStatus = STATUS__OK;
                                 /* DEBUG(
                                    "Deallocated a Low Priority Fibre %d %c\n",
                                               JPiv,TargetInfo[LTarget].Type);*/
                                 ConfigMethodDealloc(MethodInfo,JPiv,LTarget,
                                                          NULL,0,&LocalStatus);
                                 if ((*ReportFunction)(FunctionArg,'D',
                                         "Swapping Fibres...",JPiv,
                                         (TargetInfo[LTarget].Type == 'F'),
                                         TargetInfo[LTarget].OriginalIndex,
                                         TargetInfo[LTarget].X,
                                         TargetInfo[LTarget].Y,
                                         AllocInfo[JPiv].Theta,PercentDone)) {
                                    CancelFlag = CONFIG_TRUE;
                                    return;
                                 }
                                 
                                 /*  Sky is sky is sky, but if this was 
                                  *  a low-priority target, we flag it as 
                                  *  having been auto-deallocated.
                                  */
/*
 * J.Pritchard, 2016-10-05
 * Following line corrected on the basis of compiler warning
 * 
 * config_method.c: In function ‘ConfigMethodConeAllocate’:
 * config_method.c:5885:65: warning: comparison of constant ‘83’ with boolean expression is always false [-Wbool-compare]
 *                                   if (! TargetInfo[LTarget].Type == 'S'){
 *                                                                  ^
 * 
                                 if (!TargetInfo[LTarget].Type = 'S'){
 */
                                 if (TargetInfo[LTarget].Type != 'S'){
                                    CONFIG_METHOD_Global.
                                         TargetMatrix[LTarget].TargetPivot=-2;
                                 }
                                 UnallocCount++;
                                 Theta = Matrix[JPiv][KTarget].Theta;
                                 
                                 /*  Now we see if we can actually re-allocate
                                  *  KTarget to the newly-freed JPiv.
                                  */
                                  
                                 LocalStatus = STATUS__OK;
                                 ConfigMethodSetAlloc(MethodInfo,JPiv,KTarget,
                                                          Theta,&LocalStatus);
                                 if (LocalStatus == STATUS__OK) {
                                    if (ReportFunction != NULL) {
                                       if ((*ReportFunction)(FunctionArg,'A',
                                              "Allocating fibres",JPiv,
                                              (TargetInfo[KTarget].Type == 'F'),
                                              TargetInfo[KTarget].OriginalIndex,
                                              TargetInfo[KTarget].X,
                                              TargetInfo[KTarget].Y,
                                              Theta,PercentDone)) {
                                          CancelFlag = CONFIG_TRUE;
                                          return;
                                       } 
                                    } 
                                    /* DEBUG(
                                       "Reallocated %d to former target: %d\n",
                                                               JPiv,KTarget);*/
                                    /*  We've now allocated KTarget, so we can
                                     *  break out of the IPiv loop and the
                                     *  Retry loop.
                                     */
                                      
                                    UnallocCount--;
                                    break;
                                    
                                 } else {
                                 
                                    /*  We failed to re-allocate JPiv to
                                     *  KTarget, presumably because of a
                                     *  collision with an already-allocated
                                     *  fibre. I (KS) don't see why at this
                                     *  point we flag KTarget as having been
                                     *  auto-deallocated. Trying again makes
                                     *  sense, so we go through the Retry loop
                                     *  again.
                                     */
                                     
                                    TargetMatrix[KTarget].TargetPivot = -2;
                                    Retry = CONFIG_TRUE;
                                    break;
                                 }
                              }
                           }
                        }
                     }
                  }
               }
            }
         }
      }
   }
   
   /* Should have finished this bit now */

   CanAllocate = CONFIG_TRUE;

   return ;

}

/* -------------------------------------------------------------------------- */

/*      C o n f i g  M e t h o d  B l o c k  S w a p  A l l o c a t e
 *
 *  Function:
 *     Iterative Blitz re-arrangement to optimise configuration.
 *
 *  Description:
 *     This routine attempts to improve on the current allocation. It
 *     assumes that there are no 'easy' allocations left to be made - ie
 *     there are no more unallocated targets that can be allocated to
 *     unallocated fibres. If there are any gains left to be made, they
 *     must come from de-allocating fibres in the hope that this will
 *     clear the way for additional allocations - if we can allocate 
 *     more fibres than we deallocate, we have made a gain.
 *
 *     The code goes through all the targets, in order of importance,
 *     as determined by the global .ConeIndex array as set up by
 *     ConfigMethodDefineCones(). In each case, it looks to see if the
 *     target at the apex of the cone is unallocated. If so, it
 *     calls ConfigMethodDoSwap(). For more details, see the comments to
 *     ConfigMethodDoSwap(), but in essence it has been given an unallocated
 *     target. It tries to deallocate a fibre that can reach this unallocated
 *     target and allocates that fibre to the previously unallocated target.
 *     It now has another deallocated target. It looks to see if
 *     it can find a fibre that was either allocated to a lower priority
 *     target or was parked that it can allocate to this newly deallocated
 *     target. If it can, this is a gain. If it can't, it (ConfigMethodDoSwap())
 *     calls itself recursively to see if it can generate such a gain by 
 *     another swap. Eventually, it either achieves a gain or it gives up.
 *     Either way, it ends up returning to this routine (ConfigMethod-
 *     BlockSwapAllocate()), which goes on to the next cone in order of
 *     importance and repeats the procedure.
 *
 *  Prerequisites:
 *     ConfigMethodDefineCones() should just have been called in order
 *     to determine the order in which cones should be considered.
 */
 
static void ConfigMethodBlockSwapAllocate(
   const CONFIG_MethodInfo *MethodInfo,
   CONFIG_ReportFunction
            RF,                   /* (>) Feedback routine to call */
   void *FA)                      /* (>) Caller-supplied argument */

{

   /*  Local variables  */

   CONFIG_Boolean CanAllocate;   /* True if there are possible allocations */
   int IPiv;                     /* Loop index through pivots */
   int ITarget;                  /* Target at apex of current cone */
   int JTarget;                  /* Target in cone to be worked on */
   CONFIG_METHOD_TargetElement
                  *TargetMatrix; /* Local synonym for .TargetMatrix */
   CONFIG_METHOD_MatrixElement
                  **Matrix;      /* Local synonym for .Matrix */
   int NPiv;                     /* Number of pivots */
   int Nav;                      /* Number of unallocated pivots */
   int NTargets;                 /* Number of targets */
   int Nreq;                     /* Number of possible target gains */
   const CONFIG_TargetInfo 
                   *TargetInfo;  /* Target info array of structures */
   const CONFIG_PivotInfo
                    *PivotInfo;  /* Pivot info array */
   const CONFIG_AllocInfo 
                    *AllocInfo;  /* Allocation table info */
   long ReportTime;              /* Time of last report in whole seconds */
   int TargetCount;              /* Index through targets */
   int ILev;                     /* Level at which a gain was made */
   int Level;                    /* Loop index through levels for reporting */
   int *Index;                   /* local synonym for .ConeIndex */
   StatusType LocalStatus;       /* Local inherited status variable */
   clock_t cpu_start;            /* CPU time when we started */
   clock_t cpu_time;             /* CPU time now */
   
   /*  Pick up local pointers to the various information structures whose
    *  addresses are held in the method-specific global structure. Also
    *  pick up the number of possible targets and pivots.
    */
    
   Matrix = CONFIG_METHOD_Global.Matrix;
   TargetMatrix = CONFIG_METHOD_Global.TargetMatrix;
   NTargets = CONFIG_METHOD_Global.NTargets;
   NPiv = CONFIG_METHOD_Global.NPiv;
   PivotInfo = MethodInfo->PivotInfo;
   TargetInfo = MethodInfo->TargetInfo;
   AllocInfo = MethodInfo->AllocInfo;
   Index = CONFIG_METHOD_Global.ConeIndex;
   CanAllocate = CONFIG_FALSE;
   LocalStatus = STATUS__OK;
   
   /*  We go through Allocation list to see if there are any free fibres.
    *  Clearly we don't want to go through this process if there aren't.
    *  Moreover, if we have fewer free fibres than we need to keep
    *  free for sky, we don't want to go through it either.
    */

   TargetCount=0;
   for (IPiv = 0; IPiv < NPiv; IPiv++){
      if ((!MethodInfo->FibreIsGuide(MethodInfo,PivotInfo[IPiv].FibreType))
        && (!MethodInfo->FibreIsBroken(MethodInfo,PivotInfo[IPiv].FibreType))
        && (AllocInfo[IPiv].State == 'U')) TargetCount++;
   }
   if (TargetCount <= CONFIG_METHOD_Global.FibresForSky) return;
   
   /*  This loop calculates the number of targets we might be able to
    *  allocate fibres to. The result 'Nreq' used to be used to calculate
    *  the 'percentage done' figure, but now is not used for this - we
    *  report the percentage of targets we've tried instead. So Nreq
    *  isn't used outside this loop, which ends up merely being used to
    *  make sure we don't try to allocate invalid targets by messing
    *  with the Tried counts for them.
    */
   
   Nreq = 0;
   for (TargetCount = 0; TargetCount < NTargets; TargetCount++) {
      CONFIG_METHOD_Global.Tried[TargetCount] = 0;
      if (TargetInfo[TargetCount].Priority == CONFIG_PRIO_INVALID) {
         CONFIG_METHOD_Global.Tried[TargetCount] = -1;
      }
      if (TargetMatrix[TargetCount].TargetPivot == -1) Nreq++;
   }
   
   cpu_start = clock();
   cpu_time = 0;
   ReportTime = (int) (cpu_start / CLOCKS_PER_SEC);
   
   /*  We also count the number of available fibres. If this later drops below
   *   the number we want to leave available for sky, we'll bail out.
    */
   
   Nav = 0;
   CONFIG_METHOD_Global.SwapAbort = CONFIG_FALSE;
   for (ILev =0 ; ILev < NPiv ; ILev++){
      Nav += ((!MethodInfo->FibreIsBroken(MethodInfo,PivotInfo[ILev].FibreType))
         && (AllocInfo[ILev].State == 'U') 
         && (!MethodInfo->FibreIsGuide(MethodInfo,PivotInfo[ILev].FibreType)));
   }
   
   /*  Now we work through the cones, in order of importance.  There are
    *  NTargets in all, and Index[] - really the global ConeIndex[] array
    *  has NTargets entries, giving the target at the apex of each cone
    *  in order of importance.
    */
   
   for (TargetCount = 0; TargetCount < NTargets; TargetCount++) {
      ITarget = Index[TargetCount];
     
      /*  ITarget should now be the next target in order of importance. If
       *  this cone has no other targets of interest, we ignore it. Otherwise,
       *  we set JTarget to the index of the target we want to work on.
       *  ConfigMethodDefineCones() sets up the arrays so that the last entry
       *  in the ConeTargets array (ie with index NConeTargets -1) is the
       *  target at the apex. So in practice, JTarget should be the same as
       *  ITarget and the distinction seems to be purely historical.
       */
      
      if (TargetMatrix[ITarget].NConeTargets > 0){
         JTarget = TargetMatrix[ITarget].
                          ConeTargets[TargetMatrix[ITarget].NConeTargets-1];
                          
         /*  If this target is unallocated, this is a cone we want to 
          *  work on. We leave it be if it is a fiducial target, and clearly
          *  we don't do anything if it isn't a valid target.
          */
          
         if ((TargetMatrix[JTarget].TargetPivot == -1) &&
                 (TargetInfo[JTarget].Type != 'F') &&
                     (TargetInfo[JTarget].Priority != CONFIG_PRIO_INVALID)) {
                     
            CONFIG_METHOD_Global.PcDone = 100.0*TargetCount/NTargets;
                     
            /*  If we're short of fibres to be saved for sky allocations,
             *  bail out now.
             */
             
            if (Nav <= CONFIG_METHOD_Global.FibresForSky) goto RunOut;
         
            /*  This is the important part of the code. We call the routine
             *  ConfigMethodDoSwap(), passing it the unallocated target at the
             *  cone apex, to see if it can get an improvement by allocating a
             *  different fibre to it. ConfigMethodDoSwap() returns the level
             *  at which it made a gain. If there was no gain made, it returns
             *  zero.
             */
         
            CONFIG_METHOD_Global.SwapLevel = 0;
            CONFIG_METHOD_Global.Moves = 0;
            
            /*  Clearing the merit value for the configuration, means that any
             *  changes in merit value made by the call to ConfigMethodDoSwap()
             *  can be judged simply - if the resulting merit value is negative,
             *  it means that although there has been a gain, it isn't one we're
             *  interested in, since it's been the result of deallocating a high
             *  priority target to allocate two lower priority ones. (It seemed
             *  this might be happening in 6dF testing - it looks to me as if
             *  the code would allow it, although I never actually caught it - KS).
             */
             
            CONFIG_METHOD_Global.MeritValue = 0;            
            
            ILev = ConfigMethodDoSwap(MethodInfo,JTarget,0,&ReportTime,RF,FA);
            
            if ((ILev > 0) && (CONFIG_METHOD_Global.MeritValue < 0)) {
               DEBUG ("Gain made at expense of high priority target - ignored\n");
               ILev = 0;
            }
            
            if (ILev != 0){
            
               /*  There was a gain made. Because of the large number of 
                *  deallocations and allocations made in the swapping process,
                *  they are not reported in the usual way to the higher levels
                *  for display in the GUI. Instead, the (de)allocations are
                *  tentative, and logged through ConfigMethodSwapDeallocate()
                *  and ConfigMethodSwapAllocate(). However, now that we have
                *  an actual gain made, we have to record the series of swaps
                *  required to produce it. These were recorded in global
                *  variables that record the allocations and deallocations at
                *  each level. We now have to play back these so that all these
                *  changes are made permanent and reported properly. The
                *  routines that do this for each level involved are 
                *  ConfigMethodSwapDeallocateR() and ConfigMethodSwapAllocateR()
                *  (the 'R' standing for 'Report').
                */
               
               Nav--;
               CONFIG_METHOD_Global.SwapLevel = 0;
               
               /*  There was a comment here to the effect that not resetting
                *  the swap level here may will screw us up in what follows.
                *  I'm not quite sure why that's true (KS).
                *  We loop through each level to that where the gain was made.
                *  Note that the last level has no deallocation - that's what
                *  made this a gain!
                */
                
               for (Level = 1 ; Level < ILev ; Level++){
                  ConfigMethodSwapDeallocateR(MethodInfo,RF,FA,Level);
                  ConfigMethodSwapAllocateR(MethodInfo,RF,FA,Level);
               }
               ConfigMethodSwapAllocateR(MethodInfo,RF,FA,ILev);
            }
            cpu_time = clock() - cpu_start;
            if ((*RF)(FA,'P',"Swapping Fibres...",
                   0,0,0,0,0,0.0,CONFIG_METHOD_Global.PcDone) != 0) {
               return;
            }
         }
      }
   }
   
RunOut:;
   cpu_time = clock() - cpu_start;
   printf("Finished Swapping... took %d cpu seconds\n",
                                          (int) (cpu_time/CLOCKS_PER_SEC)); 
   return;

}

/* -------------------------------------------------------------------------- */

/*               C o n f i g  M e t h o d  D o  S w a p                      
 *
 *  Function:
 *     Sswaps fibres starting from a specified target.
 *
 *  Description:
 *     This is the routine that does most of the work in the 'iterative'
 *     ('Oxford') algorithm's swap phase. It is passed the index number of
 *     a target that is now unallocated (either because it was unallocated
 *     after the initial allocation pass - the case when this routine is
 *     called directly from CongigMethodBlockSwapAllocate() - or because
 *     it has just been tentatively deallocated as part of the swapping
 *     process - the case when this routine calls itself recursively.)
 *     It looks at the various fibres that can be allocated to this target.
 *     If there is an unallocated fibre, or one at such a low priority that
 *     having it allocated is merely regarded as a bonus, then it makes
 *     this allocation and returns immediately, signalling that it has
 *     made a gain. If there is no such immediate gain to be made, it
 *     tries in turn all the currently allocated fibres that could instead
 *     be allocated to the unallocated target. It takes one of these
 *     currently allocated fibres and does a swap, deallocating it from its
 *     current target and allocating it to the unallocated target it was
 *     passed. It now has a new unallocated target - the one the swapped
 *     fibre was previously allocated to.  It may be that there is an 
 *     immediate gain to be made by allocating a parked or bonus fibre to
 *     this. So it calls itself recursively, so see if it can make a gain
 *     following the swap it just made. 
 *
 *     And this carries on until it makes a gain, or has made more than
 *     a limiting number of swaps, or until the user cancels it using the
 *     GUI, or until it finds itself trying the same fibre again. Lpivot 
 *     lists the last pivot to be moved, since we're in a recursive loop 
 *     and we don't want to just shuffle the same fibre backwards and 
 *     forwards while not getting anywhere.
 *
 *  Prerequisites:
 *     This routine is normally called from ConfigMethodBlockSwapAllocate(),
 *     or recursively by itself. It is also called as part of the auto-
 *     reallocation code. It needs the information in the global .TargetMatrix
 *     to have been set up - it needs to know which fibres can reach which
 *     targets, and which have been allocated already.
 */

static int ConfigMethodDoSwap(
   const CONFIG_MethodInfo *MethodInfo,
   int  ITarget,                     /* (>) Target to work on */
   int  LPivot,                      /* (>) Last Pivot moved */
   long *ReportTime,                 /* (>) Time in secs count at last report */
   CONFIG_ReportFunction RF,         /* (>) Feedback routine to call */
   void *FA)                         /* (>) Caller-supplied argument */
{

   /*  Local variables  */

   clock_t cpu_secs;                   /* Current clock time in whole seconds */
   int IPiv;                           /* Loop index through pivots */
   int JPiv;                           /* Pivot number to allocate to ITarget */
   int KTarget;                        /* Target JPiv was allocated to */
   CONFIG_METHOD_TargetElement 
                       *TargetMatrix;  /* Local synonym for .TargetMatrix */
   CONFIG_METHOD_MatrixElement
                            **Matrix;  /* Local synonym for .Matrix */
   int NPiv;                           /* Number of pivots */
   int NTargets;                       /* Number of targets */
   const CONFIG_TargetInfo *TargetInfo;/* Target info array of structures */
   const CONFIG_PivotInfo *PivotInfo;  /* Pivot info array */
   const CONFIG_AllocInfo *AllocInfo;  /* Allocation table info */
   char KType;                         /* Type of original JPiv target */
   int BaseLevel;                      /* Level at which we were called */
   int *Tried;                         /* Local synonym for tried */
   int Test;                           /* Result of recursive call */
   int ThisPri;                        /* Priority of original JPiv target */
   CONFIG_Boolean AllocatedFibre;      /* True if JPiv was allocated */
   CONFIG_Boolean RecFlag;             /* True if in auto-reallocation mode */
   
   /*  Pick up local pointers to the various information structures whose
    *  addresses are held in the method-specific global structure. Also
    *  pick up the number of possible targets and pivots.
    */
   
   RecFlag = CONFIG_METHOD_Global.RecoverFlag;
   Matrix = CONFIG_METHOD_Global.Matrix;
   TargetMatrix = CONFIG_METHOD_Global.TargetMatrix;
   NTargets = CONFIG_METHOD_Global.NTargets;
   NPiv = CONFIG_METHOD_Global.NPiv;
   PivotInfo = MethodInfo->PivotInfo;
   TargetInfo = MethodInfo->TargetInfo;
   AllocInfo = MethodInfo->AllocInfo;
   Tried = CONFIG_METHOD_Global.Tried;

   /*  RecFlag is set in the case where we're being called as part of the
    *  auto-reallocation code.
    */
    
   RecFlag = CONFIG_METHOD_Global.RecoverFlag;
      
   CONFIG_METHOD_Global.SwapAbort = CONFIG_FALSE;
   CONFIG_METHOD_Global.SwapLevel++;
   
   /*  This is the bail-out clause. Don't get rid of this! We bail out after
    *  getting to a swap depth given by the global .MaxSwapLevel.
    */
    
   if ((CONFIG_METHOD_Global.SwapLevel == CONFIG_METHOD_Global.MaxSwapLevel) ||
       (TargetInfo[ITarget].Priority == CONFIG_PRIO_INVALID)) {
     return (0);
   }
   
   BaseLevel = CONFIG_METHOD_Global.SwapLevel;
   ThisPri = TargetInfo[ITarget].Priority;
   
   /*  We see if this target has been tried already at this swap level. If so,
    *  don't try it again.
    */
    
   if (Tried[ITarget] == 0 || Tried[ITarget] > CONFIG_METHOD_Global.SwapLevel) {
   
      /*  Now we work through all the pivots that are able to reach this
       *  target.
       */
       
      for (IPiv = 0 ; IPiv < TargetMatrix[ITarget].NumPivots ;IPiv ++) {
     
         /*  This routine can take a very long time, so we need to make sure it
          *  reports regularly so that it can be cancelled by the user. We
          *  make sure there is a report every second. The percentage complete
          *  needs be worked out rather better, but this is a difficult thing
          *  to calculate.
          */
       
         cpu_secs = (int) (clock()/CLOCKS_PER_SEC);
         if (cpu_secs > *ReportTime) {
            *ReportTime = cpu_secs;
            if ((*RF)(FA,'P',"Swapping Fibres...",
                   0,0,0,0,0,0.0,CONFIG_METHOD_Global.PcDone) != 0) {
               CONFIG_METHOD_Global.SwapAbort = CONFIG_TRUE;
               return (0);
            }
         }
       
         CONFIG_METHOD_Global.SwapLevel = BaseLevel;
         JPiv = TargetMatrix[ITarget].Pivot[IPiv];
         
         /*  JPiv is one of the pivots that can reach the target in question.
          *  We make sure it isn't a broken pivot. If it is, shouldn't we
          *  go on to the next pivot, rather than just returning? Ie shouldn't
          *  this return be a continue, without messing with the move count?
          */
          
         if (MethodInfo->FibreIsBroken(MethodInfo,PivotInfo[JPiv].FibreType)) {
            CONFIG_METHOD_Global.Moves = CONFIG_METHOD_Global.MaxMoves;
            return(0);
         }
         
         /*  We also check to see if this is a pivot we've moved before. 
          *  Strictly, since the configuration has changed, we might try 
          *  this again, but on balance we don't seem to gain.
          */
          
         if (JPiv != LPivot) { 
         
            /*  OK, JPiv is a fibre that we can in principle allocate to the
             *  currently unalocated target ITarget. It may be already 
             *  unallocated, in which case we've made a clear gain. It may 
             *  have been allocated but only to a bonus target, in which
             *  case this is also a gain if we're in auto-revovery mode. Or 
             *  it may have been allocated to a target we'd like to keep 
             *  allocated, in which case we've merely made a change that may
             *  allow us to make a real gain at a lower level.
             */
             
            AllocatedFibre = (AllocInfo[JPiv].State == 'A');
            if (AllocatedFibre) {
               KTarget = AllocInfo[JPiv].TargetIndex;
               KType = TargetInfo[KTarget].Type;
               
               /*  JPiv was in fact allocated, to KTarget. So we de-allocate
                *  it and see where that gets us.
                */
                
               ConfigMethodSwapDeallocate(MethodInfo,JPiv,KTarget);
            } else {
               KTarget = 0;
               KType = ' ';
            }
            
            /*  JPiv is now parked. Either it was originally, or we've just
             *  parked it. Now, can we actually assign it to ITarget, which is
             *  what we want to do. We may not be able to, because of
             *  collisions with other allocated fibres. We test this using
             *  ConfigMethodSwapAllocate(), which tests to see if we can do
             *  the allocation, and makes the tentative allocation if possible.
             */
             
            if (ConfigMethodSwapAllocate(MethodInfo,JPiv,ITarget)) {
            
               /*  Yes, we could do the allocation. Now, does this represent a
                *  gain, in which case we can quit now, or not?  Note that
                *  KType is only set if AllocatedFibre is set, but that's OK.
                */
                
               CONFIG_METHOD_Global.Moves++;
               if (!AllocatedFibre || 
                    (RecFlag && AllocatedFibre && KType == 'S')
                    || (RecFlag && ThisPri <= CONFIG_METHOD_Global.BonusPrio)) {
                    
                  /*  This is a gain. For debugging purposes, we see what sort
                   *  of a gain it was. If it was not previously allocated,
                   *  (AllocatedFibre not set) then this is an allocation of
                   *  a parked fibre. Otherwise, in auto-recover mode, we
                   *  may have re-allocated a fibre allocated to a sky target
                   *  or to a 'bonus' target.
                   */
                  
                  if (AllocatedFibre){
                     CONFIG_METHOD_Global.SwapPromote = CONFIG_TRUE;
                     /*
                     if (KType == 'S'){
                        DEBUG("Level %d: Swap released a Sky Fibre: %d\n",
                                                               BaseLevel,JPiv);
                     } else {
                        DEBUG(
                        "Level %d: Swap promoted a low priority fibre: %d %d\n",
                                                      BaseLevel,JPiv,KTarget);
                     }
                     */
                  } else {
                     /*
                     DEBUG("Level %d: Swap Allocated %d from park\n",
                                                       BaseLevel,JPiv);
                     */
                  }
                  
                  /*  Since we made a gain, return the level at which we made
                   *  the gain. The higher level code will now have to change
                   *  the tentative changes we made into permanent ones, which
                   *  is why it will need to know what level we got to.
                   */
                         
                  return(CONFIG_METHOD_Global.SwapLevel);
                  
               } else {
               
                  /*  At this point, we've simply made a swap. We've allocated
                   *  JPiv, which used to be allocated to KTarget, to the
                   *  original target, ITarget. So we now have KTarget 
                   *  unallocated instead of ITarget. But we may be able to
                   *  make a gain having made this swap. So now we call
                   *  ourselves again, and see if we get any further.
                   *
                   *  We see if we've made too many moves already, in which
                   *  case we don't go any deeper.
                   */
           
                  if (CONFIG_METHOD_Global.Moves < 
                                           CONFIG_METHOD_Global.MaxMoves) {
                                           
                     /*  Here is the recursive call to DoSwap(). If a gain
                      *  is made at a lower level, this returns a +ve level #.
                      */
                      
                     Test = ConfigMethodDoSwap(MethodInfo,KTarget,JPiv,
                                                         ReportTime,RF,FA);
                     if (CONFIG_METHOD_Global.SwapAbort) return (0);
                     if (Test > 0) {
                     
                        /*  We made a gain at a lower level, so climb back out
                         *  of the recursive call stack.
                         */
                         
                        return(Test);
                        
                     } else {
                     
                        /*  No, we didn't make a gain at a lower level, so
                         *  put back what we changed and keep trying.
                         */
                        Tried[ITarget] = CONFIG_METHOD_Global.SwapLevel;
                        ConfigMethodSwapDeallocate(MethodInfo,JPiv,ITarget);
                        ConfigMethodSwapAllocate(MethodInfo,JPiv,KTarget);
                 
                     }
                     
                  } else {
                  
                     /*  Maximum number of moves exceeded, so put things back */
                     
                     ConfigMethodSwapDeallocate(MethodInfo,JPiv,ITarget);
                     ConfigMethodSwapAllocate(MethodInfo,JPiv,KTarget);
                     return (0);
                  }
               }
               
               /*  This is the end of the code where we were able to allocate
                *  JPiv to the original target KTarget.
                */
                
            } else {
            
               /*  This is the case where ConfigMethodSwapAllocate() reported
                *  that it wasn't able to allocate JPiv to the original
                *  ITarget. In that case, we may have unallocated JPiv 
                *  unnecessarily, and should put things back.
                */
                
               if (AllocatedFibre) {
                  ConfigMethodSwapAllocate(MethodInfo,JPiv,KTarget);
               }

            }
            
         }  /*(JPiv != LPivot) */
         
      } /* Close Loop Over Pivots */
      
   } /* Tried it before and it doesn't work!*/
   
   return(0);
  
}

/* -------------------------------------------------------------------------- */

/*           C o n f i g  M e t h o d  R a n d o m  S o r t
 *
 *  Function:
 *     Randomly re-arranges the contents of an array of integers.
 *
 *  Description:
 *     This routine is passed an array of integers and returns it randomly
 *     re-arranged.
 */

static void ConfigMethodRandomSort (
   int Array[],                        /* (!) Array to be jumbled */
   int Entries)                        /* (>) Number of entries in array */
{
   /*  Local variables */
   
   int   Entry;                        /* Index in original array to use */
   int*  FlagsArray;                   /* Flags indicating element used */
   short Found;                        /* Used to control loop for element */
   int   I;                            /* Loop index */
   int*  JumbledArray;                 /* Work array for rearranged data */
   float RandomValue;                  /* A random number between 0 and 1.0 */
   int   Repeat;                       /* Controls times main loop repeated */
   
   /*  Initial values */
   
   JumbledArray = (int*) 0;
   FlagsArray = (int*) 0;
   
   /*  Allocate memory for work arrays */
   
   FlagsArray = (int*) malloc (Entries * sizeof(int));
   if (FlagsArray) {
      JumbledArray = (int*) malloc (Entries * sizeof(int));
      if (JumbledArray) {
      
         /*  If we got here, we allocated work arrays OK. Now we do the
          *  real work.  The algorithm is crude and was coded without any
          *  serious attempt to generate anything efficient or with the
          *  best random characteristics. For each element in the array, we
          *  generate a random number between 0 and 1.0 and hence an index
          *  into the array between 0 and Entries-1. We put the entry at
          *  that index into the jumbled array at index I. The problem is
          *  that we might generate the same index more than once as we
          *  pass through the entries. So we use the flag array to make sure
          *  we only use entries that haven't been used yet. But what do we
          *  do if we find we've got a clash? We could keep trying until we
          *  don't get a clash, but that could take a while. Instead, we
          *  just use the next element that hasn't been used yet. This leads
          *  to a rather non-random distribution towards the top end of the
          *  array. So we repeat the whole thing twice, and that seems to 
          *  give a reasonable result. Good enough for the likes of us..
          */
          
         for (Repeat = 0; Repeat < 2; Repeat++) {
            for (I = 0; I < Entries; I++) FlagsArray[I] = 0;     
            for (I = 0; I < Entries; I++) {
            
               /*  Random() returns a value between 0 and 0xFFFFFFFF. We
                *  don't need that much randomness, so we use the bottom
                *  16 bits and scale it to a value between 0 and 1.
                */
                
               RandomValue = (float)(random() & 0xFFFF) / 65535.0;
               Entry = (int) (RandomValue * (float)(Entries - 1));
               if (Entry < 0) Entry = 0;
               if (Entry >= Entries) Entry = Entries - 1;
               
               /*  Entry is now the index for the element we're going to
                *  use for our new index I value. We search up from Entry
                *  (wrapping round if necessary) until we find an unused
                *  element. We will always find such an element.
                */
                
               Found = 0;
               while (!Found) {
                  if (FlagsArray[Entry] == 0) {
                     FlagsArray[Entry] = 1;
                     JumbledArray[I] = Array[Entry];
                     Found = 1;
                  } else {
                     Entry++;
                     if (Entry >= Entries) Entry = 0;
                  }
               }
            }
            
            /*  We now have our jumbled set of values in JumbledArray, so
             *  we copy them back into Array.
             */
             
            for (I = 0; I < Entries; I++) Array[I] = JumbledArray[I];
         }
      }
   }
   
   /*  Release any allocated memory */
   
   if (JumbledArray) free (JumbledArray);
   if (FlagsArray) free (FlagsArray);
}

/* -------------------------------------------------------------------------- */

/*           C o n f i g  M e t h o d  N t h  L o w e s t
 *
 *  Function:
 *     Picks the nth lowest value from an array of integers.
 *
 *  Description:
 *     This routine is passed an array of integers and an n value. It 
 *     returns the index (from zero) of the nth lowest entry.
 */

static int ConfigMethodNthLowest (
   int Array[],                        /* (>) Array in question */
   int Entries,                        /* (>) Number of entries in array */
   int N)                              /* (>) The value of N (from 1 up) */
{
   /*  Local variables */
   
   int*  ValueArray;                   /* Copy of the values */
   int   I;                            /* Loop index */
   int*  IndexArray;                   /* Work array for rearranged data */
   int   ITemp;                        /* Used to exchange values */
   int   J;                            /* Inner loop index */
   int   ReturnIndex;                  /* The required index */
   
   /*  Initial values */
   
   IndexArray = (int*) 0;
   ValueArray = (int*) 0;
   ReturnIndex = 0;
   
   /*  Sanity check */
   
   if ((N > 0) && (N <= Entries)) {
   
      /*  Allocate memory for work arrays */

      ValueArray = (int*) malloc (Entries * sizeof(int));
      if (ValueArray) {
         IndexArray = (int*) malloc (Entries * sizeof(int));
         if (IndexArray) {

            /*  If we got here, we allocated work arrays OK. Now we do the
             *  real work.  The algorithm is crude and was coded without any
             *  serious attempt to generate anything efficient. 
             */

            for (I = 0; I < Entries; I++) {
               ValueArray[I] = Array[I];
               IndexArray[I] = I;
            }

            /*  This is just a bubble sort of the two arrays, based on the
             *  values in the main array.
             */

            for (I = 0; I < Entries; I++) {
               for (J = I + 1; J < Entries; J++) {
                  if (ValueArray[I] > ValueArray[J]) {
                     ITemp = ValueArray[J];
                     ValueArray[J] = ValueArray[I];
                     ValueArray[I] = ITemp;
                     ITemp = IndexArray[J];
                     IndexArray[J] = IndexArray[I];
                     IndexArray[I] = ITemp;
                  }
               }
            }

            ReturnIndex = IndexArray[N - 1];
         }
      }
   }
            
   /*  Release any allocated memory */
   
   if (ValueArray) free (ValueArray);
   if (IndexArray) free (IndexArray);
   
   return ReturnIndex;
}

/* -------------------------------------------------------------------------- */

/*            C o n f i g  M e t h o d  S w a p  D e a l l o c a t e
 *
 *  Function:
 *     Makes a tentative deallocation of a fibre from a pivot.
 *
 *  Description:
 *     This routine makes an easily reversed deallocation of a pivot from
 *     a target. It is used during the swap phase of the 'Oxford' algorithm.
 *     It does the deallocation, but does not report it to the GUI level
 *     and records it in the SwapMatrix array for the current swap level.
 *     It clears the TargetPivot value for the target in the global structure
 *     .TargetMatrix.
 */
 
static void ConfigMethodSwapDeallocate(
   const CONFIG_MethodInfo *MethodInfo,
   int  IPiv,                        /* (>) Pivot to Deallocate */
   int  ITarget)                     /* (>) Target to Deallocate */
{

   /*  Local variables  */

   const CONFIG_TargetInfo *TargetInfo;/* Target info array of structures */
   StatusType LocalStatus;
   int SwapLevel;
   
   /*  Pick up local pointers to the various information structures whose
    *  addresses are held in the method-specific global structure. Also
    *  pick up the number of possible targets and pivots.
    */

   TargetInfo = MethodInfo->TargetInfo;
   SwapLevel = CONFIG_METHOD_Global.SwapLevel;
   LocalStatus = STATUS__OK;
   if (MethodInfo->AllocInfo[IPiv].State != 'A') {
      printf("  ERROR... Fibre was not allocated\n");
   }
   ConfigMethodDealloc (MethodInfo,IPiv,ITarget,NULL,0,&LocalStatus);
   CONFIG_METHOD_Global.TargetMatrix[ITarget].TargetPivot = -1;
   CONFIG_METHOD_Global.SwapMatrix[0][SwapLevel] = IPiv;
   CONFIG_METHOD_Global.SwapMatrix[1][SwapLevel] = ITarget;
   CONFIG_METHOD_Global.MeritValue -= (MethodInfo->TargetInfo[ITarget].Priority
                                    * MethodInfo->TargetInfo[ITarget].Priority);

}

/* -------------------------------------------------------------------------- */

/*            C o n f i g  M e t h o d  S w a p  A l l o c a t e 
 *
 *  Function:
 *     Makes a tentative allocation of a fibre to a pivot.
 *
 *  Description:
 *     This routine makes an easily reversed allocation of a pivot to
 *     a target. It is used during the swap phase of the 'Oxford' algorithm.
 *     It does the allocation, but does not report it to the GUI level
 *     and records it in the SwapMatrix array for the current swap level.
 *     It sets the TargetPivot value for the target in the global structure
 *     .TargetMatrix. This routine is also responsible for checking that the
 *     allocation is in fact possible. Because the Oxford algorithm does not
 *     update the global allocation matrix each time it does a tentative
 *     allocation, the caller of this routine knows that the suggested
 *     allocation is OK on fixed physical grounds (bend angle etc), but
 *     does not know if the allocation will be prevented by a recently
 *     tentatively allocated fibre. If the allocation is not in fact possible,
 *     this routine returns false. If it returns true, the allocation was
 *     possible and has now been recorded.
 */
 
static CONFIG_Boolean ConfigMethodSwapAllocate(
   const CONFIG_MethodInfo *MethodInfo,
   int  IPiv,                        /* (>) Pivot to Allocate */
   int  ITarget)                     /* (>) Target to Allocate */
{

   /*  Local variables  */

   const CONFIG_TargetInfo *TargetInfo;/* Target info array of structures */
   double Theta;
   int SwapLevel;
   StatusType LocalStatus;
   
   /*  Pick up local pointers to the various information structures whose
    *  addresses are held in the method-specific global structure. Also
    *  pick up the number of possible targets and pivots.
    */

   if (CONFIG_METHOD_Global.Matrix[IPiv][ITarget].PathStatus == 
                                                      CONFIG_METHOD_POSSIBLE){
     TargetInfo = MethodInfo->TargetInfo;
     SwapLevel=CONFIG_METHOD_Global.SwapLevel;
     Theta = CONFIG_METHOD_Global.Matrix[IPiv][ITarget].Theta;
     LocalStatus = STATUS__OK;
     ConfigMethodSetAlloc(MethodInfo,IPiv,ITarget,Theta,&LocalStatus);
     if (LocalStatus == STATUS__OK) {
       CONFIG_METHOD_Global.TargetMatrix[ITarget].TargetPivot = IPiv;
       CONFIG_METHOD_Global.SwapMatrix[2][SwapLevel] = IPiv;
       CONFIG_METHOD_Global.SwapMatrix[3][SwapLevel] = ITarget;
       CONFIG_METHOD_Global.MeritValue += (MethodInfo->TargetInfo[ITarget].Priority
                                    * MethodInfo->TargetInfo[ITarget].Priority);
       return(CONFIG_TRUE);
     }
   }
   return(CONFIG_FALSE);
}

/* -------------------------------------------------------------------------- */

/*            C o n f i g  M e t h o d  S w a p  A l l o c a t e  R           */

static void ConfigMethodSwapAllocateR(
   const CONFIG_MethodInfo *MethodInfo,
   CONFIG_ReportFunction
            ReportFunction,          /* (>) Feedback routine to call */
   void *FunctionArg,                /* (>) Caller-supplied argument */
   int  ILev)                        /* (>) Pivot to Allocate */
{

   /*  Local variables  */

   const CONFIG_TargetInfo *TargetInfo;/* Target info array of structures */
   double Theta,PercentDone;
   int IPiv,ITarget;
   /*  Pick up local pointers to the various information structures whose
    *  addresses are held in the method-specific global structure. Also
    *  pick up the number of possible targets and pivots.
    */

   TargetInfo = MethodInfo->TargetInfo;
   IPiv = CONFIG_METHOD_Global.SwapMatrix[2][ILev];
   ITarget = CONFIG_METHOD_Global.SwapMatrix[3][ILev];
   CONFIG_METHOD_Global.MeritValue += (MethodInfo->TargetInfo[ITarget].Priority
                                    * MethodInfo->TargetInfo[ITarget].Priority);
   if (ReportFunction != NULL) {
     Theta = CONFIG_METHOD_Global.Matrix[IPiv][ITarget].Theta;
     PercentDone = CONFIG_METHOD_Global.PcDone;
     if ((*ReportFunction)(FunctionArg,'A',
            "Swapping Fibres...",IPiv,(TargetInfo[ITarget].Type == 'F'),
                           TargetInfo[ITarget].OriginalIndex,
                           TargetInfo[ITarget].X,TargetInfo[ITarget].Y,
                           Theta,PercentDone)) {
       return;  /* Exit on Cancel ...may have to cancel a few!*/
     } 
   }

}

/* -------------------------------------------------------------------------- */

/*            C o n f i g  M e t h o d  S w a p  D e a l l o c a t e  R     */

static void ConfigMethodSwapDeallocateR(
   const CONFIG_MethodInfo *MethodInfo,
   CONFIG_ReportFunction
            ReportFunction,          /* (>) Feedback routine to call */
   void *FunctionArg,                /* (>) Caller-supplied argument */
   int  ILev)                        /* (>) Pivot to Allocate */
{

   /*  Local variables  */

   const CONFIG_TargetInfo *TargetInfo;/* Target info array of structures */
   double PercentDone;
   int ITarget,IPiv;
    /*  Pick up local pointers to the various information structures whose
    *  addresses are held in the method-specific global structure. Also
    *  pick up the number of possible targets and pivots.
    */

   TargetInfo = MethodInfo->TargetInfo;
   IPiv = CONFIG_METHOD_Global.SwapMatrix[0][ILev];
   ITarget = CONFIG_METHOD_Global.SwapMatrix[1][ILev];
   if (ReportFunction != NULL) {
     PercentDone = CONFIG_METHOD_Global.PcDone;
     if ((*ReportFunction)(FunctionArg,'D',
             "Swapping Fibres...",IPiv,(TargetInfo[ITarget].Type == 'F'),
              TargetInfo[ITarget].OriginalIndex,
              TargetInfo[ITarget].X,TargetInfo[ITarget].Y,
              MethodInfo->AllocInfo[IPiv].Theta,PercentDone)) {
       return;  /* Exit on Cancel ...may have to cancel a few!*/
     } 
   } 
   CONFIG_METHOD_Global.TargetMatrix[ITarget].TargetPivot = -1;
   CONFIG_METHOD_Global.MeritValue -= (MethodInfo->TargetInfo[ITarget].Priority
                                    * MethodInfo->TargetInfo[ITarget].Priority);

}

/* -------------------------------------------------------------------------- */

/*                C o n f i g  M e t h o d  A l l o c  C o u n t
 *
 *  Function:
 *     Returns the number of pivots currently allocated.
 *
 *  Description:
 *     This is really a debugging utility that can be useful at times
 *     when tracing how well an allocation algorithm is doing. It simply
 *     returns the number of pivots currently allocated. Note that this
 *     includes both tentative and non-tentative allocations.
 */
 
static int ConfigMethodAllocCount (
   const CONFIG_MethodInfo *MethodInfo)
{  
   const CONFIG_AllocInfo *AllocInfo = MethodInfo->AllocInfo;
   int AllocCount = 0;
   int NPiv = MethodInfo->NumberPivots;
   int IPiv;
   for (IPiv = 0; IPiv < NPiv; IPiv++) {
      if (AllocInfo[IPiv].State == 'A') AllocCount++;
   }
   return (AllocCount);
}


/* -------------------------------------------------------------------------- */

/*              C o n f i g  M e t h o d  C r a s h  T e s t
 *    GBD 31/10/96
 *  Function:
 *     Checks to see whether the pivot in question acutally hit anything
 *     given the current allocation table...
 *
 *  Description:
 *     Rather than keep the matrix up to date, it's more useful to 
 *     keep a list of what was originally possible in the matrix and just
 *     test for hits each time we try to move a specific fibre... or
 *     at least that was the case for GBD's code. i.e. we're only going
 *     to worry about the 400 allocations that exist at the moment, 
 *     and not about the 400^2 allocations that are in principle possible.
 *     -This sounds like it's nitpicking a little bit, if you make the
 *     assumption that most of those don't change most of the time, but
 *     that's not true if your juggling fibres left right and centre!
 *
 */

static CONFIG_Boolean ConfigMethodCrashTest (
   const CONFIG_MethodInfo *MethodInfo,
   int Pivot,                     /* (>) The number of the pivot just moved*/
   int Target,                    /* (>) The number of the pivot just moved*/
   double Theta)                  /* (>) The number of the pivot just moved*/
{

   /*  A note on variable names.  A new allocation of a fibre to a pivot has 
    *  been made.  So there is now a new, unalterable, position for a button
    *  and there is a button running from the newly allocated pivot point to
    *  the virtual pivot point of that fibre. All the variables describing
    *  this new allocation contain 'New'. All the variables describing 
    *  the potential allocations represented by the matrix do not have 'New'
    *  but are otherwise the same names. In many cases, these qualtities are
    *  coordinates in microns, and as such are held as integers. However, we
    *  sometimes need these values in floating point as well, and in these
    *  cases we prefix the name with a 'D'.
    */ 
 
   /*  Local variables  */

   long ButtonX1;              /* X-start of safe rectangle around old target */
   long ButtonX2;              /* X-end of safe rectangle around old target */
   long ButtonY1;              /* Y-start of safe rectangle around old target */
   long ButtonY2;              /* Y-end of safe rectangle around old target */
   double DFibrePivotX;        /* Virtual pivot position in X for new fibre */
   double DFibrePivotY;        /* Virtual pivot position in Y for new fibre */
   double DNewPivotX;          /* Floating point value of NewPivotX */
   double DNewPivotY;          /* Floating point value of NewPivotY */
   double DNewTargetX;         /* Floating point value of NewTargetX */
   double DNewTargetY;         /* Floating point value of NewTargetY */
   double DPivotX;             /* Floating point value of PivotX */
   double DPivotY;             /* Floating point value of PivotY */
   double DTargetX;            /* Floating point value of TargetX */
   double DTargetY;            /* Floating point value of TargetY */
   long FibreX1;               /* X-start of safe rectangle around old fibre */
   long FibreX2;               /* X-end of safe rectangle around old fibre */
   long FibreY1;               /* Y-start of safe rectangle around old fibre */
   long FibreY2;               /* Y-end of safe rectangle around old fibre */
   double DNewFibrePivotX;     /* Virtual pivot position in X for fibre */
   double DNewFibrePivotY;     /* Virtual pivot position in Y for fibre */
   int IPiv;                   /* Loop index through pivots */
   int ITarget;                /* Loop index through targets */
   CONFIG_METHOD_MatrixElement **Matrix;
                               /* Local synonym for .Matrix in globals */
   CONFIG_METHOD_TargetElement *TargetMatrix;
                               /* Local synonym for .Matrix in globals */
   int NPiv;                   /* Number of pivots */
   long NewButtonX1;           /* X-start of safe rectangle around new target */
   long NewButtonX2;           /* X-end of safe rectangle around new target */
   long NewButtonY1;           /* Y-start of safe rectangle around new target */
   long NewButtonY2;           /* Y-end of safe rectangle around new target */
   long NewFibreX1;            /* X-start of safe rectangle around fibre posn */
   long NewFibreX2;            /* X-end of safe rectangle around fibre posn */
   long NewFibreY1;            /* Y-start of safe rectangle around fibre posn */
   long NewFibreY2;            /* Y-end of safe rectangle around fibre posn */
   long NewTargetX;            /* X-coordinate of newly allocated target */
   long NewTargetY;            /* Y-coordinate of newly allocated target */
   long NewPivotX;             /* X-coordinate of newly allocated pivot */
   long NewPivotY;             /* Y-coordinate of newly allocated pivot */
   int NTargets;               /* Number of possible targets */
   const CONFIG_PivotInfo *PivotInfo;/* Pivot info array of structures */
   const CONFIG_AllocInfo *AllocInfo;/* Pivot info array of structures */
   long PivotX;                /* X-coordinate of previously allocated pivot */
   long PivotY;                /* Y-coordinate of previously allocated pivot */
   CONFIG_Boolean Safe;        /* Indicates rects have no overlap regions */
   const CONFIG_TargetInfo *TargetInfo; /* Target info array of structures */
   long TargetX;               /* X-coordinate of previously allocated target */

   long TargetY;               /* Y-coordinate of previously allocated target */
   double TargetTheta;         /* Theta of previously allocated target */
   long SafeClearance;         /* Safe clearance (microns) about button */
   int  CheckCollision;
   int Collision;   /* True if there is a collision */
   int ParkMayCollide;         /* Indicates if park positions may collide */

   /*  Note that this code could be simplified enormously if we were game
    *  just to make one call to tdFcollisionButBut() and two calls to
    *  tdFcollisionButFib() for each of the possible allocations. However,
    *  these are very precise, careful, slow routines, and we want to avoid
    *  using them if possible, simply because we expect this routine to be
    *  called a great many times. Hence the business with 'safe' rectangles,
    *  and the huge number of variables needed to describe them.
    */

   /*  Useful constants. We need to know the maximum offset of a button
    *  from the target point, so we know, crudely, how much clearance we
    *  need to be really safe when looking at overlapping rectangles later
    *  in the code. We get this from the housekeeping routines rather than
    *  from the dimensions .h file, to allow larger clearances to be 
    *  experimented with easily.
    */

   SafeClearance = MethodInfo->SafeClearance(MethodInfo);
   /*  Pick up local pointers to the various information structures whose
    *  addresses are held in the method-specific global structure. Also
    *  pick up the number of possible targets and pivots.
    */

   Matrix = CONFIG_METHOD_Global.Matrix;
   NTargets = CONFIG_METHOD_Global.NTargets;
   NPiv = CONFIG_METHOD_Global.NPiv;
   AllocInfo = MethodInfo->AllocInfo;
   PivotInfo = MethodInfo->PivotInfo;
   TargetInfo = MethodInfo->TargetInfo;
   TargetMatrix = CONFIG_METHOD_Global.TargetMatrix;

   /*  Calculate the rectangles surrounding the newly allocated fibre and
    *  its button. The NewButtonxxx values give the rectangle that safely
    *  clears the newly allocated button position.  The NewFibrexxx values
    *  give the rectangle that safely clears the whole new allocated fibre.
    */

   NewTargetX = TargetInfo[Target].X;
   NewTargetY = TargetInfo[Target].Y;
   DNewTargetX = (double) NewTargetX;
   DNewTargetY = (double) NewTargetY;
   NewButtonX1 = NewTargetX - SafeClearance;
   NewButtonX2 = NewTargetX + SafeClearance;
   NewButtonY1 = NewTargetY - SafeClearance;
   NewButtonY2 = NewTargetY + SafeClearance;
   NewPivotX = PivotInfo[Pivot].X;
   NewPivotY = PivotInfo[Pivot].Y;
   DNewPivotX = (double) NewPivotX;
   DNewPivotY = (double) NewPivotY;
   if (NewPivotX > NewTargetX) {
      NewFibreX1 = NewTargetX - SafeClearance;
      NewFibreX2 = NewPivotX + SafeClearance;
   } else {
      NewFibreX1 = NewPivotX - SafeClearance;
      NewFibreX2 = NewTargetX + SafeClearance;
   }
   if (NewPivotY > NewTargetY) {
      NewFibreY1 = NewTargetY - SafeClearance;
      NewFibreY2 = NewPivotY + SafeClearance;
   } else {
      NewFibreY1 = NewPivotY - SafeClearance;
      NewFibreY2 = NewTargetY + SafeClearance;
   }



   /*  We we also need to know the position of the virtual pivot point
    *  of the newly allocated button, since that is where the fibre is
    *  regarded as joining the button.
    */
    
   /* Have to calculate this here, as this is a new allocation */
   MethodInfo->GetVirtPiv (MethodInfo,NewTargetX,NewTargetY,Theta,
                           &DNewFibrePivotX,
                           &DNewFibrePivotY);

   CheckCollision = CONFIG_FALSE;
   Collision = CONFIG_FALSE;

   /*
    * We need to determine if we have to check for collisions against
    * parked fibres.
    */
   ParkMayCollide = MethodInfo->ParkMayCollide(MethodInfo);


   /*  We need to pass through every allocated pivot to see if we hit.
    *  We can't assume unallocated/broken pivots are safe, because for some
    *  instruments an allocated fibre can collide with a parked pivot.
    *  (It seems to me [KS] that this shouldn't be an issue here, since
    *  this code shouldn't be being asked to check on an allocation that 
    *  can collide with a parked pivot, since that should be ruled out by
    *  the initial state of the matrix. At least, that's true now that the
    *  matrix initialisation code checks for that case. But it's as well to
    *  be sure.)
    */

   for (IPiv = 0; IPiv < NPiv; IPiv++) {
       if (IPiv != Pivot) {
           PivotX = PivotInfo[IPiv].X;
           PivotY = PivotInfo[IPiv].Y;
           DPivotX = (double) PivotX;
           DPivotY = (double) PivotY;

           if (AllocInfo[IPiv].State == 'A') {
               /* Allocated fibre, get details from target position */
               ITarget     = AllocInfo[IPiv].TargetIndex;
               TargetX     = TargetInfo[ITarget].X;
               TargetY     = TargetInfo[ITarget].Y;
               TargetTheta =  AllocInfo[IPiv].Theta;
           } else if (!ParkMayCollide) {
               /* If park won't collide, can skip this one */
               continue;
           } else if (!MethodInfo->FibreReallyExists(MethodInfo,IPiv)) {
               /*  If the parked fibre doesn't really exist, we can skip
                *  this one. */
               continue;
           } else {
               /* Unallocated, get details from park position */
               TargetX     = PivotInfo[IPiv].ParkX;
               TargetY     = PivotInfo[IPiv].ParkY;
               TargetTheta =  PivotInfo[IPiv].Theta;
           }

            /*  We need to check
             *  1) whether there is a collision between the newly allocated
             *  button and the existing button position, 2) whether there is
             *  a collision between the newly allocated button and the 
             *  the existing fibre path, 3) whether there is a collision
             *  between the newly allocated fibre path and the existing
             *  button position. The standard routines for doing the formal
             *  checks for these are quite time-consuming, so we try to do 
             *  some quick and dirty checks first to see if we need to call
             *  them.  We crudely define 'rectangles of interest' around
             *  the two button positions and the two fibres - these are just
             *  the smallest rectangles in the standard coordinate system 
             *  that fully enclose (and allow for clearance) each button and
             *  each fibre. Only if these overlap do we need to call the
             *  more detailed check routines.
             */

           DTargetX = (double) TargetX;
           DTargetY = (double) TargetY;
           
           ButtonX1 = TargetX - SafeClearance;
           ButtonX2 = TargetX + SafeClearance;
           ButtonY1 = TargetY - SafeClearance;
           ButtonY2 = TargetY + SafeClearance;
           
            /* So, do we need to call the button-button collision routine?  */

           Safe = CONFIG_FALSE;
           if (ButtonX2 < NewButtonX1) {
               Safe = CONFIG_TRUE;
           } else if (NewButtonX2 < ButtonX1) {
               Safe = CONFIG_TRUE;
           } else if (ButtonY2 < NewButtonY1) {
               Safe = CONFIG_TRUE;
           } else if (NewButtonY2 < ButtonY1) {
               Safe = CONFIG_TRUE;
           }

           if (!Safe) {
               if (MethodInfo->ColButBut (MethodInfo,DTargetX, DTargetY, 
                                          TargetTheta, 
                                          DNewTargetX, DNewTargetY,
                                          Theta)) {
                   return(CONFIG_TRUE);
               }
           }

           /*  Now, what about the possibility of the newly allocated fibre
            *  colliding with the potentially allocated button? The potentially
            *  allocated button is at (TargetX,TargetY), and its 'safe'
            *  rectangle is delimited by (ButtonX1,ButtonX2,ButtonY1,ButtonY2).
            *  The fibre leading to the newly allocated button goes
            *  from its pivot (NewPivotX,NewPivotY) to the virtual pivot
            *  point of its button (NewFibrePivotX,NewFibrePivotY). Its 'safe'
            *  rectangle is given by (NewFibreX1,NewFibreX2,NewFibreY1,
            *  NewFibreY2).
            */
           
           Safe = CONFIG_FALSE;
           if (ButtonX2 < NewFibreX1) {
               Safe = CONFIG_TRUE;
           } else if (NewFibreX2 < ButtonX1) {
               Safe = CONFIG_TRUE;
           } else if (ButtonY2 < NewFibreY1) {
               Safe = CONFIG_TRUE;
           } else if (NewFibreY2 < ButtonY1) {
               Safe = CONFIG_TRUE;
           }
           
           if (!Safe) {
               if (MethodInfo->ColButFib (MethodInfo,DTargetX, DTargetY, 
                                          TargetTheta,
                                          DNewFibrePivotX,DNewFibrePivotY,
                                          DNewPivotX,DNewPivotY)) {
                   return(CONFIG_TRUE);
               }
           }
           
           /*  And, finally, what about the possibility of the potentially
            *  allocated fibre colliding with the newly allocated button?
            *  The newly allocated button is at (NewTargetX,NewTargetY), and
            *  its 'safe' rectangle is given by (NewButtonX1,NewButtonX2,
            *  NewButtonY1,NewButtonY2).  The fibre leading to the potentially
            *  allocated button goes from its pivot (PivotX,PivotY) to the
            *  virtual pivot point of its button, which we still have to
            *  calculate as (FibrePivotX,FibrePivotY). The 'safe' rectangle
            *  for the fibre is (FibreX1,FibreX2,FibreY1,FibreY2), which we
            *  also still have to calculate.
            */


           if (PivotX > TargetX) {
               FibreX1 = TargetX - SafeClearance;
               FibreX2 = PivotX + SafeClearance;
           } else {
               FibreX1 = PivotX - SafeClearance;
               FibreX2 = TargetX + SafeClearance;
           }
           if (PivotY > TargetY) {
               FibreY1 = TargetY - SafeClearance;
               FibreY2 = PivotY + SafeClearance;
           } else {
               FibreY1 = PivotY - SafeClearance;
               FibreY2 = TargetY + SafeClearance;
           }
           
           Safe = CONFIG_FALSE;
           if (NewButtonX2 < FibreX1) {
               Safe = CONFIG_TRUE;
           } else if (FibreX2 < NewButtonX1) {
               Safe = CONFIG_TRUE;
           } else if (NewButtonY2 < FibreY1) {
               Safe = CONFIG_TRUE;
           } else if (FibreY2 < NewButtonY1) {
               Safe = CONFIG_TRUE;
           }
               
           if (!Safe) {
               /* For allocated targets, use the value we caluclated
                  when we allocated this target.  For parked fibres, 
                  we must calucate it
               */
               if (AllocInfo[IPiv].State == 'A') {
                   DFibrePivotX = CONFIG_METHOD_Global.VPivX[IPiv];
                   DFibrePivotY = CONFIG_METHOD_Global.VPivY[IPiv];
               } else {
                   MethodInfo->GetVirtPiv (MethodInfo,
                                           TargetX,TargetY,Theta,
                                           &DFibrePivotX,
                                           &DFibrePivotY);
               }
                   

               if (MethodInfo->ColButFib (MethodInfo, DNewTargetX, 
                                          DNewTargetY, Theta, 
                                          DFibrePivotX,DFibrePivotY,
                                          DPivotX,DPivotY)) {
                   return(CONFIG_TRUE);
               }
           } 
       }
   }
   /* If we got to here then we're OK!*/
   CONFIG_METHOD_Global.VPivX[Pivot] = DNewFibrePivotX;
   CONFIG_METHOD_Global.VPivY[Pivot] = DNewFibrePivotY;
   return(CONFIG_FALSE);

}

/*             C o n f i g  M e t h o d  F i b r e  C r o s s               */
static CONFIG_Boolean ConfigMethodFibreCross(
     const CONFIG_MethodInfo *MethodInfo,
     int IPiv,                      /*(>) 1st Fibre */
     int JPiv)                      /*(>  2nd Fibre */
{

  /* This routine calculates whether the two fibre cross */

  /* Local Variables */

  double X1,X2,X3,X4,Y1,Y2,Y3,Y4;
  double M1,M2,M3,C1,C2,XC,YC;
  int ITarg,JTarg;
  double ITheta,JTheta,X,Y;
  const CONFIG_PivotInfo *PivotInfo;
  const CONFIG_AllocInfo *AllocInfo;
  const CONFIG_TargetInfo *TargetInfo;
  CONFIG_METHOD_MatrixElement **Matrix;
  CONFIG_METHOD_TargetElement *TargetMatrix;   

  Matrix = CONFIG_METHOD_Global.Matrix;
  AllocInfo = MethodInfo->AllocInfo;
  PivotInfo = MethodInfo->PivotInfo;
  TargetInfo = MethodInfo->TargetInfo;
  TargetMatrix = CONFIG_METHOD_Global.TargetMatrix;

  if (AllocInfo[IPiv].State != 'A' || AllocInfo[JPiv].State != 'A' ||
      IPiv == JPiv) return(CONFIG_FALSE);

  ITarg = AllocInfo[IPiv].TargetIndex;
  JTarg = AllocInfo[JPiv].TargetIndex;
  ITheta = Matrix[IPiv][ITarg].Theta;
  JTheta = Matrix[JPiv][JTarg].Theta;
  X1 = (double) PivotInfo[IPiv].X;
  Y1 = (double) PivotInfo[IPiv].Y;
  X3 = (double) PivotInfo[JPiv].X;
  Y3 = (double) PivotInfo[JPiv].Y;


  X = (double) TargetInfo[ITarg].X;
  Y = (double) TargetInfo[ITarg].Y;
  X2 = CONFIG_METHOD_Global.VPivX[IPiv];
  Y2 = CONFIG_METHOD_Global.VPivY[IPiv];
/*  ConfigGetVirtPiv(X,Y,ITheta,&X2,&Y2);*/
  X = (double) TargetInfo[JTarg].X;
  Y = (double) TargetInfo[JTarg].Y;
  X4 = CONFIG_METHOD_Global.VPivX[JPiv];
  Y4 = CONFIG_METHOD_Global.VPivY[JPiv];
/*  ConfigGetVirtPiv(X,Y,JTheta,&X4,&Y4);*/

  /* Give this a try and see if JOS is faster! */
  return(MethodInfo->ColFibFib(MethodInfo,X1,Y1,X2,Y2,X3,Y3,X4,Y4));

  /* Should probably return(tdFcollisionFibFib(X1,Y1,X2,Y2,X3,Y3,X4,Y4)); here
  but I think this could be faster...*/
  if (X1 != X2) {
    M1 = (Y1-Y2)/(X1-X2);
  } else {
    M1 = (Y1-Y2)/(X1-X2 + 0.0000001);
  }
  if (X3 != X4) {
    M2 = (Y3-Y4)/(X3-X4);
  } else {
    M2 = (Y3-Y4)/(X3-X4 + 0.0000001);
  }
  if (M2 == M1) return(CONFIG_FALSE);
  M3= 1.0/(M2-M1);
  /* Now we can find the intersection of the two lines */
  C1 = Y1-M1*X1;
  C2 = Y3-M2*X3;
  XC = (C1-C2)*M3;
  YC = M1*XC + C1;
  if (X1 > X2){
    if (XC <= X1 && XC >= X2){
      if (Y1 > Y2){
        if (YC > Y1 && YC < Y2) return(CONFIG_FALSE);
      } else {
        if (YC > Y2 && YC < Y1) return(CONFIG_FALSE);
      }
    } else {
      return(CONFIG_FALSE);
    }
  } else {
    if (XC <= X2 && XC >= X1){
      if (Y1 > Y2){
        if (YC > Y1 && YC < Y2) return(CONFIG_FALSE);
      } else {
        if (YC > Y2 && YC < Y1) return(CONFIG_FALSE);
      }
    } else {
      return(CONFIG_FALSE);
    }
  }

  if (X3 > X4) {
    if (XC <= X3 && XC >= X4){
      if (Y3 > Y4){
        if (YC <= Y3 && YC >= Y4){
          return(CONFIG_TRUE);
        } else { 
          return(CONFIG_FALSE);
        }
      } else {
        if (YC <= Y4 && YC >= Y3){
          return(CONFIG_TRUE);
        } else {
          return(CONFIG_FALSE);
        }
      }
    } else {
      return(CONFIG_FALSE);
    }
  } else {
    if (XC <= X4 && XC >= X3){
      if (Y3 > Y4){
        if (YC <= Y3 && YC >= Y4){
          return(CONFIG_TRUE);
        } else {
          return(CONFIG_FALSE);
        }
      } else {
        if (YC <= Y4 && YC >= Y3){
          return(CONFIG_TRUE);
        } else {
          return(CONFIG_FALSE);
        }
      }
    } else {
      return(CONFIG_FALSE);
    }
  }
}

/*        C o n f i g  M e t h o d  R e p o r t  U n c r o s s i n g        */

/* This is just a boring block of code that was making things hard to follow */
static void ConfigMethodReportUncrossing(
   const CONFIG_MethodInfo *MethodInfo,
   int IPiv, int ITarg,
   int JPiv, int JTarg,
   CONFIG_ReportFunction  RF,          /* (>) Feedback routine to call */
   void *FA)                           /* (>) Caller-supplied argument */
{
  CONFIG_METHOD_Global.SwapLevel = 0;
  CONFIG_METHOD_Global.SwapMatrix[0][0] = JPiv;
  CONFIG_METHOD_Global.SwapMatrix[1][0] = JTarg;
  ConfigMethodSwapDeallocateR(MethodInfo,RF,FA,0);
  CONFIG_METHOD_Global.SwapMatrix[0][0] = IPiv;
  CONFIG_METHOD_Global.SwapMatrix[1][0] = ITarg;
  ConfigMethodSwapDeallocateR(MethodInfo,RF,FA,0);
  CONFIG_METHOD_Global.SwapMatrix[2][0] = JPiv;
  CONFIG_METHOD_Global.SwapMatrix[3][0] = ITarg;
  ConfigMethodSwapAllocateR(MethodInfo,RF,FA,0);
  CONFIG_METHOD_Global.SwapMatrix[2][0] = IPiv;
  CONFIG_METHOD_Global.SwapMatrix[3][0] = JTarg;
  ConfigMethodSwapAllocateR(MethodInfo,RF,FA,0);
}

/*     C o n f i g M e t h o d T r y U n c r o s s                     */

/*
 * Try to uncross two fibres.
 *
 * We assume the fibres are currently cross and that they can each reach
 * the target the other is assigned too. 
 *
 * Returns true if successfull.  In this case, will also adjuct the Cross
 * matrix
 */
 
static int ConfigMethodTryUncross(
   const CONFIG_MethodInfo *MethodInfo,
   const int IPiv,               /* First pivot         */
   const int ITarg,              /* Target first pivot is assigned too */
   const int JPiv,               /* Second pivot        */
   const int JTarg)              /* Target second pivot is assigned too */
{
    int IVPivX;        /* Space for saving virtual pivot positions */
    int IVPivY; 
    int JVPivX;
    int JVPivY;
    
    int IPivX;        /* Space for saving current fibre info   */
    int IPivY;
    double IPivT;
    int JPivX;
    int JPivY;
    double JPivT;

    CONFIG_Boolean ResultOK = CONFIG_FALSE;

    /*
     * Point to the target information
     */
    const CONFIG_TargetInfo *TargetInfo = MethodInfo->TargetInfo;
    /*
     * Point to the pivot information
     */
    const CONFIG_PivotInfo *PivotInfo = MethodInfo->PivotInfo;
    /*
     * Point to the allocation information
     */
    const CONFIG_AllocInfo *AllocInfo = MethodInfo->AllocInfo;


    /*
     * Save the current positions of the two pivots.
     */
    IPivX = AllocInfo[IPiv].X;
    IPivY = AllocInfo[IPiv].Y;
    IPivT = AllocInfo[IPiv].Theta;
    JPivX = AllocInfo[JPiv].X;
    JPivY = AllocInfo[JPiv].Y;
    JPivT = AllocInfo[JPiv].Theta;

    /*
     * Save the current virtual pivot positions of each fibre
     * (Overwritten by ConfigMethodCrashTest()
     */
    IVPivX = CONFIG_METHOD_Global.VPivX[IPiv];
    IVPivY = CONFIG_METHOD_Global.VPivY[IPiv];
    JVPivX = CONFIG_METHOD_Global.VPivX[JPiv];
    JVPivY = CONFIG_METHOD_Global.VPivY[JPiv];

    /*
     * Temporaly set these pivots to their parked positions.  We must
     * be dam sure later to restore them.
     *
     * Doing this allows us to run ConfigMethodCrashTest for the 
     * uncrossed allocation.
     */
    MethodInfo->SetFibrePos(MethodInfo, IPiv,
                            PivotInfo[IPiv].ParkX,
                            PivotInfo[IPiv].ParkY,
                            PivotInfo[IPiv].Theta,
                            0, CONFIG_FALSE);

    MethodInfo->SetFibrePos(MethodInfo, JPiv,
                            PivotInfo[JPiv].ParkX,
                            PivotInfo[JPiv].ParkY,
                            PivotInfo[JPiv].Theta,
                            0, CONFIG_FALSE);




    /*
     * Try the allocation of JPiv to I's target
     */
    if (!ConfigMethodCrashTest(MethodInfo, JPiv, ITarg, 
                              CONFIG_METHOD_Global.Matrix[JPiv][ITarg].Theta)){
        /*
         * This one works.  Set the fibre position appropriately before
         * tring the other one
         */
        MethodInfo->SetFibrePos(MethodInfo, JPiv, 
                                TargetInfo[ITarg].X,
                                TargetInfo[ITarg].Y, 
                                CONFIG_METHOD_Global.Matrix[JPiv][ITarg].Theta,
                                ITarg,
                                CONFIG_TRUE);
        if (!ConfigMethodCrashTest(MethodInfo, IPiv, JTarg, 
                              CONFIG_METHOD_Global.Matrix[IPiv][JTarg].Theta)){
            /*
             * Both are ok.  This uncross is successfull.
             */ 
            ResultOK = CONFIG_TRUE;
        }
        
        
    }
    
    /*
     * Restore the original positions and virtual pivot values.  We do
     * this even when the cross was successfull, was the SetFibrePos
     * function above does really do the job and we now have to
     * do it correctly.
     */
    MethodInfo->SetFibrePos(MethodInfo, IPiv, IPivX, IPivY, IPivT, 
                            ITarg, CONFIG_TRUE);
    MethodInfo->SetFibrePos(MethodInfo, JPiv, JPivX, JPivY, JPivT,
                            JTarg, CONFIG_TRUE);

    CONFIG_METHOD_Global.VPivX[IPiv] = IVPivX;
    CONFIG_METHOD_Global.VPivY[IPiv] = IVPivY;
    CONFIG_METHOD_Global.VPivX[JPiv] = JVPivX;
    CONFIG_METHOD_Global.VPivY[JPiv] = JVPivY;


    if (ResultOK) {
        /*
         * Deallocate both fibres
         */
        ConfigMethodSwapDeallocate(MethodInfo,IPiv,ITarg);
        ConfigMethodSwapDeallocate(MethodInfo,JPiv,JTarg);
        /*
         * Try to allocate the I to J's target
         */
        if (ConfigMethodSwapAllocate(MethodInfo,IPiv,JTarg)) {
            /*
             * Try to allocate J to I's target
             */
            if (ConfigMethodSwapAllocate(MethodInfo,JPiv,ITarg)) {
                /*
                 * Success.  We this should have happened due
                 * to the above testing.
                 */
                return CONFIG_TRUE;
            } else {
                /*
                 * Allocation of J to I's target failed.  Unwind.
                 */
                ConfigMethodSwapDeallocate(MethodInfo,IPiv,JTarg);
                ConfigMethodSwapAllocate(MethodInfo,IPiv,ITarg);
                ConfigMethodSwapAllocate(MethodInfo,JPiv,JTarg);
            }
        } else {
            /*
             * Allocation of I to J's target failed.  Unwind.
             */
            ConfigMethodSwapAllocate(MethodInfo,IPiv,ITarg);
            ConfigMethodSwapAllocate(MethodInfo,JPiv,JTarg);
        }
        
        fprintf(stderr,"Warning.  ConfigMethodTryUncross was successfull in crash testing but not in allocation, %d, %d, %d, %d\n",
                IPiv, ITarg, JPiv, JTarg);
        return CONFIG_FALSE;
    }
    /*
     * Did not uncross.
     */
    return CONFIG_FALSE;
}

/*        C o n f i g  M e t h o d  U n c r o s s  F i b r e s          */


/* For this routine to work we need to have the cones matrix defined 
 * for the whole priority range...
 *
 * NOTE - CURRENTLY DOES NOT WORK FOR INSTRUMENTS ON WHICH PARKED FIBRES
 * MAY BE CROSSED BY OTHERS (e.g. 6dF). 
 */

static void ConfigMethodUncrossFibres(
   const CONFIG_MethodInfo *MethodInfo,
   CONFIG_ReportFunction  RF,          /* (>) Feedback routine to call */
   void *FA)                           /* (>) Caller-supplied argument */
{
  int IPiv,JPiv,KPiv,LPiv,ITarg,JTarg;
  int NPiv,NCross,Nloop,Nswap,Nstuck;
  int NTargets;
  CONFIG_METHOD_MatrixElement **Matrix;
  CONFIG_METHOD_TargetElement *TM;
  const CONFIG_AllocInfo *AllocInfo;
  const CONFIG_TargetInfo *TargetInfo;
  const CONFIG_PivotInfo *PivotInfo;
  CONFIG_Boolean **Cross;
  double PcDone;


  /*
   * Once off malloc of space here, as we can no longer always have
   * a fixed number of pivots and must use MethodInfo->NumberPivots.  But 
   * the current structure requires this to be efficent.  
   *
   * THIS MALLOC IS NEVER FREEED (but it is only malloced once)
   *
   * This function does not support error reporting so we must
   * use the assertion style exit if malloc fails.
   */
  static int *ICross = 0;
  if (!ICross) {
      ICross = malloc(sizeof(*ICross) * MethodInfo->NumberPivots);
      if (!ICross) {
          fprintf(stderr,"%s:%d Assertion failed\n", __FILE__,__LINE__);
          fprintf(stderr,"Unsupported failure of malloc\n");
          fprintf(stderr,"Talk to tjf@aaoepp.aao.gov.au\n");
          exit(-1);
      }
  }

  /*
   * Set up pointers
   */
  PivotInfo = MethodInfo->PivotInfo;
  TargetInfo = MethodInfo->TargetInfo;
  AllocInfo = MethodInfo->AllocInfo;
  Matrix = CONFIG_METHOD_Global.Matrix;
  TM = CONFIG_METHOD_Global.TargetMatrix;
  NPiv = CONFIG_METHOD_Global.NPiv;
  NTargets = CONFIG_METHOD_Global.NTargets;
  Cross = CONFIG_METHOD_Global.Cross;
  /*
   * Initialise array of number of crosses
   */
  NCross = 0;
  Nstuck = 0;
  for (IPiv = 0; IPiv < NPiv; IPiv++) {
    ICross[IPiv]=0;
  }
  /*
   * Determine which fibres cross.  Note ConfigMethodFibreCross will
   * not bother about unallocated fibres.
   *
   * The result of this is the array Cross[i][j] indicating if 
   * fibres I and J cross.
   */
  for (IPiv = 0; IPiv < NPiv-1; IPiv++) {
    if (AllocInfo[IPiv].State == 'A'){
      for (JPiv = IPiv; JPiv < NPiv; JPiv++){
        if (AllocInfo[JPiv].State != 0){
          Cross[IPiv][JPiv] = ConfigMethodFibreCross(MethodInfo,IPiv,JPiv);
          Cross[JPiv][IPiv] = Cross[IPiv][JPiv];
          if (Cross[IPiv][JPiv]) {
            NCross++;
            ICross[IPiv]++;
            ICross[JPiv]++;
          }
        }
      }
    }
  }
  /* DEBUG("Found %d fibre crossings\n",NCross); */
  fflush(stdout);
  Nswap = 1;
  Nloop = 0;
  if (NCross == 0) 
      return;
  
  /* 
   * Loop until there's nothing left to do.. 
   */
  while (Nswap != 0){
    Nstuck = NCross;
    Nloop++;
    Nswap = 0;
    /* 
     *  For each fibre IPiv.
     */
    for (IPiv=0; IPiv<NPiv-1; IPiv++){
      /*
       * Report progress.
       */
      if (RF != NULL){
        if ((IPiv % 40) == 0){
          PcDone = (100.*IPiv)/NPiv;
          CONFIG_METHOD_Global.PcDone=PcDone;
          if ((*RF)(FA,'P',"Uncrossing Fibres...",
                    0,0,0,0,0,0.0,PcDone) != 0) {
            return;
          }
        }
      }
      /*
       * If IPiv has crosses (Should only happen if I is allocated)
       */
      if (ICross[IPiv] > 0 /*&& AllocInfo[IPiv].State == 'A'*/){
        /* 
         * Save the target I is allocated to.
         */
        ITarg = AllocInfo[IPiv].TargetIndex;
        /*
         * For every fibre JPiv, from IPiv onwards.
         */
        for (JPiv=IPiv;JPiv<NPiv;JPiv++){
          /*
           * If JPiv has crosses (Should only happend if J is allocated)
           */
          if (ICross[JPiv] > 0 /*&& AllocInfo[JPiv].State == 'A'*/){
            /*
             * If I and J cross
             */
            if (Cross[IPiv][JPiv]){
              /*
               * Save the target J is allocated to
               */
              JTarg = AllocInfo[JPiv].TargetIndex;
              /*
               * Loop through all the pivots which can read target ITarg.
               */
              for (KPiv = 0; KPiv < TM[ITarg].NumPivots; KPiv++){
                /*
                 * Can JPiv reach IPiv's Target
                 */
                if (TM[ITarg].Pivot[KPiv] == JPiv){ 
                  /*
                   * Loop through all the pivots which can read target JTarg
                   */
                  for (LPiv = 0; LPiv < TM[JTarg].NumPivots; LPiv++){
                    /*
                     * Can IPiv reach JPiv's target.
                     */
                    if (TM[JTarg].Pivot[LPiv] == IPiv){
                      /* I and J can both reach each others targets, see
                       * if they can be legimitately be exchanged.
                       */
                      if ((Matrix[IPiv][JTarg].PathStatus 
                                         == CONFIG_METHOD_POSSIBLE) &&
                          (Matrix[JPiv][ITarg].PathStatus 
                                              == CONFIG_METHOD_POSSIBLE)) {
                        /*
                         * I and J can both be allocated to each others targets.
                         * Try to uncross them.
                         */
                        if (ConfigMethodTryUncross(MethodInfo, 
                                                 IPiv, ITarg,
                                                 JPiv, JTarg)) {
                          /*
                           * Success.  Note it and report it.
                           */
                          Cross[IPiv][JPiv] = CONFIG_FALSE;
                          Cross[JPiv][IPiv] = CONFIG_FALSE;
                          Nswap++;
                          ConfigMethodReportUncrossing(MethodInfo,
                                                       IPiv,ITarg,
                                                       JPiv,JTarg,RF,FA);
                          goto Swapped;
                        }
                        break;
                      }
                    }
                  }
                  break;
                }
              }
            }
          }
        }
      }
Swapped:;
    }
    /* DEBUG("Found %d viable swaps\n",Nswap); */
    NCross = 0;
    for (IPiv = 0; IPiv < NPiv; IPiv++) ICross[IPiv] = 0;
    for (IPiv = 0; IPiv < NPiv-1; IPiv++) {
      if (AllocInfo[IPiv].State != 0){
        for (JPiv = IPiv; JPiv < NPiv; JPiv++){
          if (AllocInfo[JPiv].State != 0){
            Cross[IPiv][JPiv] = ConfigMethodFibreCross(MethodInfo,IPiv,JPiv);
            Cross[JPiv][IPiv] = Cross[IPiv][JPiv];
            if (Cross[IPiv][JPiv]) {
              NCross++;
              ICross[IPiv]++;
              ICross[JPiv]++;
            }
          }
        }
      }
    }
    /* DEBUG("Found %d fibre crossings\n",NCross); */
    if (NCross == 0)
        return;
    if (NCross >= Nstuck) 
        return;
      
  }
}

/*    C o n f i g  M e t h o d  A u t o  D e a l l o c a t e  
 *
 *
 *  This is invoked when we have an existing allocation but it is possible the
 *  constants file (broken fibres etc) has changed.  It removes all allocations
 *  which don't make sense, such as broken fibres, colliding fibres etc.
 *  (Routine by GBD, comment by TJF).
 */
 

static void ConfigMethodAutoDeallocate(
   const CONFIG_MethodInfo *MethodInfo,
   CONFIG_ReportFunction
            ReportFunction,         /* (>) Feedback routine to call */
   void *FunctionArg,               /* (>) Caller-supplied argument */
   StatusType *Status)              /* (!) Inherited status */
{
   /*  Local variables */
  int AllocCount;                   /* Number of pivots allocated so far */
  int IPiv,JPiv,ILev;               /* Pivot index number */
  int NPiv,Field,IRep;              /* Total number of pivots */
  int ITarget,JTarget,KTarget;      /* Target index number */ 
  int NTargets;                     /* Number of targets */
  double PercentDone;               /* Percentage of current task completed */
  long ReportTime;                  /* Time of last report in whole seconds */
  double Theta;                     /* Theta value for possible allocation */
  const CONFIG_AllocInfo *AllocInfo;      /* Address of allocation structure */
  const CONFIG_TargetInfo *TargetInfo;    /* Address of target information */
  const CONFIG_PivotInfo *PivotInfo;      /* Address of pivot information */
  int IPrio,JPrio;                  /* Priority level to work on */
  int FreeGuide,FreeObject;         /*Number of free fibres */
  CONFIG_METHOD_MatrixElement **Matrix;
  CONFIG_METHOD_TargetElement *TargetMatrix;
  StatusType LocalStatus;
  int Test;
  int NZap;
  
  /*
   * Once off malloc of space here, as we can no longer always have
   * a fixed number of pivots and must use MethodInfo->NumberPivots.  But 
   * the current structure requires this to be efficent.  
   *
   * THIS MALLOC IS NEVER FREEED (but it is only malloced once)
   *
   * This function does not support error reporting so we must
   * use the assertion style exit if malloc fails.
   */
  static int *Zapped = 0;
  if (!Zapped) {
      Zapped = malloc(sizeof(*Zapped) * MethodInfo->NumberPivots);
      if (!Zapped) {
          fprintf(stderr,"%s:%d Assertion failed\n", __FILE__,__LINE__);
          fprintf(stderr,"Unsupported failure of malloc\n");
          fprintf(stderr,"Talk to tjf@aaoepp.aao.gov.au\n");
          exit(-1);
      }
  }

  AllocInfo = MethodInfo->AllocInfo;
  TargetInfo = MethodInfo->TargetInfo;
  PivotInfo = MethodInfo->PivotInfo;
  NPiv = CONFIG_METHOD_Global.NPiv;
  NTargets = CONFIG_METHOD_Global.NTargets;
  Matrix = CONFIG_METHOD_Global.Matrix;
  TargetMatrix = CONFIG_METHOD_Global.TargetMatrix;
  Field = CONFIG_METHOD_Global.Field;

  AllocCount = 0;
  NZap = 0;
  for(IPiv = 0; IPiv < NPiv ; IPiv++){
    if ((MethodInfo->FibreIsBroken(MethodInfo,PivotInfo[IPiv].FibreType)) 
          && (AllocInfo[IPiv].State == 'A')){
      AllocCount++;
      ITarget = AllocInfo[IPiv].TargetIndex;
      Theta = Matrix[IPiv][ITarget].Theta;
      LocalStatus = STATUS__OK;
      MethodInfo->NoteDeallocationBroken(MethodInfo,IPiv,&LocalStatus);
      /* Set this to something unique here to see if we can recover later */
      TargetMatrix[ITarget].TargetPivot = -2;
      /* Unless it's a sky fibre, in which case we can safely ignore it */
      if (TargetInfo[ITarget].Type == 'S'){
        TargetMatrix[ITarget].TargetPivot = -1;
      } else {
        Zapped[NZap] = ITarget;
        NZap++;
      }
      if (ReportFunction != NULL) {
        PercentDone = (float) IPiv / (float) NPiv * 100.0; 
        if ((*ReportFunction)(FunctionArg,'D',
            "Deallocating broken fibres",IPiv,(TargetInfo[ITarget].Type
             == 'F'),TargetInfo[ITarget].OriginalIndex,
            TargetInfo[ITarget].X,TargetInfo[ITarget].Y,Theta,PercentDone)) {
          return;
        }
        /*        DEBUG("Deallocated broken fibre %d from %d\n",IPiv,ITarget);*/
      }
    }
  }
  /* DEBUG ("Deallocated %d broken fibres\n",AllocCount); */
  ILev = AllocCount;
  AllocCount = 0;
  for(IPiv = 0; IPiv < NPiv ; IPiv++){
    if (AllocInfo[IPiv].State == 'A'){
      ITarget = AllocInfo[IPiv].TargetIndex;
      if (Matrix[IPiv][ITarget].PathStatus == CONFIG_METHOD_BAD_ALPHA){
        printf("Illegal Alpha Angle found for %d %d\n",IPiv,ITarget);
        AllocCount++;
        Theta = Matrix[IPiv][ITarget].Theta;
        LocalStatus = STATUS__OK;
        MethodInfo->NoteDeallocation(MethodInfo,IPiv,&LocalStatus);
        TargetMatrix[ITarget].TargetPivot = -2;
        if (TargetInfo[ITarget].Type == 'S'){
          TargetMatrix[ITarget].TargetPivot = -1;
        } else {
          Zapped[NZap]=ITarget;
          NZap++;
        }
        if (ReportFunction != NULL) {
          PercentDone = (float) IPiv / (float) NPiv * 100.0; 
          if ((*ReportFunction)(FunctionArg,'D',
              "Deallocating illegal fibres",IPiv,(TargetInfo[ITarget].Type
               == 'F'),TargetInfo[ITarget].OriginalIndex,
              TargetInfo[ITarget].X,TargetInfo[ITarget].Y,Theta,PercentDone)) {
            return;
          }
          /* DEBUG("Deallocated illegal fibre %d %d\n",IPiv,ITarget); */
        }
      }
    }
  }
  /* DEBUG ("Deallocated %d illegal fibres\n",AllocCount); */
  /* Any Screw Collisions? */
  ILev += AllocCount;
  AllocCount = 0;
  for(IPiv = 0; IPiv < NPiv ; IPiv++){
    if (AllocInfo[IPiv].State == 'A'){
      ITarget = AllocInfo[IPiv].TargetIndex;
      if (Matrix[IPiv][ITarget].PathStatus == CONFIG_METHOD_SCREWED){
        AllocCount++;
        Theta = Matrix[IPiv][ITarget].Theta;
        LocalStatus = STATUS__OK;
        MethodInfo->NoteDeallocation(MethodInfo,IPiv,&LocalStatus);
        TargetMatrix[ITarget].TargetPivot = -2;
        if (TargetInfo[ITarget].Type == 'S'){
          TargetMatrix[ITarget].TargetPivot = -1;
        } else {
          Zapped[NZap] = ITarget;
          NZap++;
        }
        if (ReportFunction != NULL) {
          PercentDone = (float) IPiv / (float) NPiv * 100.0; 
          if ((*ReportFunction)(FunctionArg,'D',
              "Deallocating screwy fibres",IPiv,(TargetInfo[ITarget].Type
               == 'F'),TargetInfo[ITarget].OriginalIndex,
              TargetInfo[ITarget].X,TargetInfo[ITarget].Y,Theta,PercentDone)) {
            return;
          }
          /* DEBUG("Deallocated screwy fibre %d from %d\n",IPiv,ITarget); */
        }
      }
    }
  }
  /* DEBUG ("Deallocated %d screwy fibres\n",AllocCount); */
  ILev += AllocCount;
  /* OK, so we've cleared up everything that was broken or illegal... now
   * we're left with the possibility of there being collisions in the current
   * allocation that weren't there when it was originally allocated that are
   * due to a change in the relative positions due to a different set of 
   * distortion coefficients or a different plate being requested. 
   *
   * This is actually fairly unpleasant, as there may be more than one
   * collision for a given fibre, so it's not immediately clear which one
   * we should deallocate to keep the required number of parks to a minimum 
   * --For the moment just park based on priorities, and loop to make sure
   * we've got everything */

  AllocCount = 1;
  while (AllocCount != 0){
    AllocCount = 0;
    for (IPiv = 0; IPiv < NPiv ; IPiv++){
      if (AllocInfo[IPiv].State == 'A'){
        ITarget = AllocInfo[IPiv].TargetIndex;
        if (!MethodInfo->CheckAllocation(MethodInfo,IPiv,ITarget,
                                   Matrix[IPiv][ITarget].Theta,&JPiv)) {
          JTarget = AllocInfo[JPiv].TargetIndex;
          AllocCount++;
          IPrio = TargetInfo[ITarget].Priority;
          JPrio = TargetInfo[JTarget].Priority;
          Test = (TargetInfo[ITarget].Type == 'S' || 
                            (IPrio <=CONFIG_METHOD_Global.BonusPrio 
                                     && TargetInfo[JTarget].Type == 'S'));
          if (Test) goto Zap_me_first;
          /*    if ((IPrio >= JPrio) && (TargetInfo[JTarget].Type != 'F')){*/
            Theta = Matrix[JPiv][JTarget].Theta;
            LocalStatus = STATUS__OK;
            MethodInfo->NoteDeallocation(MethodInfo,JPiv,&LocalStatus);
            TargetMatrix[JTarget].TargetPivot = -2;
            if (TargetInfo[ITarget].Type == 'S') {
              TargetMatrix[ITarget].TargetPivot = -1;
            } else {
              Zapped[NZap]=ITarget;
              NZap++;
            }
            if (ReportFunction != NULL) {
              PercentDone = (float) IPiv / (float) NPiv * 100.0; 
              if ((*ReportFunction)(FunctionArg,'D',
                      "Deallocating illegal fibres",JPiv,(TargetInfo[JTarget].
                      Type == 'F'),TargetInfo[JTarget].OriginalIndex,
                      TargetInfo[JTarget].X,TargetInfo[JTarget].Y,Theta,
                                         PercentDone)) {
                return;
              }
              /* DEBUG("Deallocated colliding fibre %d from %d",JPiv,JTarget);
              */
            }
            /*          } else {*/
            /* Now deallocate the other one as well to make sure we
               have fair crack at recovery */
Zap_me_first:;
            Theta = Matrix[IPiv][ITarget].Theta;
            LocalStatus = STATUS__OK;
            MethodInfo->NoteDeallocation(MethodInfo,IPiv,&LocalStatus);
            TargetMatrix[ITarget].TargetPivot = -2;
            if (TargetInfo[ITarget].Type == 'S') {
              TargetMatrix[ITarget].TargetPivot = -1;
            } else {
              Zapped[NZap]=ITarget;
              NZap++;
            }
            if (ReportFunction != NULL) {
              PercentDone = (float) IPiv / (float) NPiv * 100.0; 
              if ((*ReportFunction)(FunctionArg,'D',
                      "Deallocating illegal fibres",IPiv,(TargetInfo[ITarget].
                      Type == 'F'),TargetInfo[ITarget].OriginalIndex,
                      TargetInfo[ITarget].X,TargetInfo[ITarget].Y,Theta,
                                         PercentDone)) {
                return;
              }
              /* DEBUG("Deallocated colliding fibre %d from %d\n",IPiv,ITarget);
              */
            }
            /* }*/
        }
      }
    }
    printf("Deallocated %d colliding fibres\n",AllocCount);
  }
  IPrio = ILev+AllocCount;
  /* DEBUG ("Attempting to Recover from %d Deallocations\n",NZap+1); */
  /* Should really try to allocate free guide fibres at this point, since
   * if we've lost any for the above reasons we should try to get something
  */
  IRep = 0;
Restart:;
  
  if (ConfigMethodDefineCones(MethodInfo,0,9)) {
    ConfigMethodConeAllocate(MethodInfo,ReportFunction,FunctionArg);
    FreeGuide = 0;
    FreeObject= 0;
    AllocCount = 0;
    for (IPiv = 0 ; IPiv < NPiv; IPiv++){
      if (AllocInfo[IPiv].State == 'U'){
        AllocCount++;
        if (MethodInfo->FibreIsGuide(MethodInfo,PivotInfo[IPiv].FibreType)){
          FreeGuide++;
        } else{
          FreeObject++;
        }
      }
    }
    for (KTarget = 0 ; KTarget < NZap ; KTarget++){
      if (TargetMatrix[Zapped[KTarget]].TargetPivot < 0){
        TargetMatrix[Zapped[KTarget]].TargetPivot = -2;
      }
    }
    /*     Try swapping if there are fibres left free... */
    /*
    DEBUG ("Attempting to Recover %d fibres by swapping: %d fibres free\n",
                                                              NZap,FreeObject);
    DEBUG ("Bonus Priority set to %d\n",CONFIG_METHOD_Global.BonusPrio);
    */
    CONFIG_METHOD_Global.RecoverFlag = CONFIG_TRUE;
    if (AllocCount > 0){
      for (KTarget = 0; KTarget < NTargets ; KTarget++){
        JTarget = CONFIG_METHOD_Global.ConeIndex[KTarget];
        ITarget = TargetMatrix[JTarget].
          ConeTargets[TargetMatrix[JTarget].NConeTargets-1];
        if (TargetMatrix[ITarget].TargetPivot == -2  && 
                                          TargetInfo[ITarget].Type != 'S'){
          /* DEBUG("Searching for a fibre to place on %d %d\n",ITarget,
                                          TargetMatrix[ITarget].TargetPivot);*/
          if ((TargetInfo[ITarget].Type == 'F' && FreeGuide > 0) ||
              (TargetInfo[ITarget].Type != 'F' && FreeObject > 0)){
            CONFIG_METHOD_Global.SwapPromote = CONFIG_FALSE;
            CONFIG_METHOD_Global.SwapLevel = 0;
            CONFIG_METHOD_Global.Moves = 0;
            ReportTime = (int) (clock() / CLOCKS_PER_SEC);
            if ((ILev = ConfigMethodDoSwap(MethodInfo,ITarget,0,
                                 &ReportTime,ReportFunction,FunctionArg)) > 0){
              if (TargetInfo[ITarget].Type == 'F') {
                FreeGuide--;
              } else {
                FreeObject--;
              }
              AllocCount++;
              CONFIG_METHOD_Global.PcDone = 100.*KTarget/NTargets;
              /* DEBUG("Made a gain at level %d\n",ILev); */
              CONFIG_METHOD_Global.SwapLevel = 0;
              for (IPiv = 1 ; IPiv < ILev ; IPiv++){
                ConfigMethodSwapDeallocateR(MethodInfo,ReportFunction,
                                                              FunctionArg,IPiv);
                ConfigMethodSwapAllocateR(MethodInfo,ReportFunction,
                                                              FunctionArg,IPiv);
              }
              if (CONFIG_METHOD_Global.SwapPromote){
                ConfigMethodSwapDeallocateR(MethodInfo,ReportFunction,
                                                              FunctionArg,ILev);
                CONFIG_METHOD_Global.SwapPromote = CONFIG_FALSE;
              }
              ConfigMethodSwapAllocateR(MethodInfo,ReportFunction,
                                                              FunctionArg,ILev);
            }
            if ((*ReportFunction)(FunctionArg,'P',"Swapping Fibres...",
                     0,0,0,0,0,0.0,CONFIG_METHOD_Global.PcDone) != 0) {
              return;
            }
          }
        }
      } 
    }
  }
  ITarget = 0;
  for (KTarget = 0 ; KTarget < NZap ; KTarget++){
    if (TargetMatrix[Zapped[KTarget]].TargetPivot < 0){
      TargetMatrix[Zapped[KTarget]].TargetPivot = -2;
      ITarget++;
    }
  }
  if (ITarget > 0) {
    /* DEBUG ("Didn't quite make it... going round again!\n");*/
    IRep++;
    if (IRep < 3) goto Restart;
  }
  CONFIG_METHOD_Global.RecoverFlag = CONFIG_FALSE;
  MethodInfo->FieldCheck(MethodInfo,ReportFunction,FunctionArg,1,Status);
  /* DEBUG ("Check status %ld\n",(long)*Status);*/
  if (*Status != STATUS__OK) return;
  AllocCount = 0;
  for (IPiv = 0; IPiv < NPiv; IPiv++) {
    if (AllocInfo[IPiv].State == 'A') AllocCount++;
  }
  
}

/*-----------------------------------------------------------------*/

static void ConfigMethodSetImportFile(
   const CONFIG_MethodInfo *MethodInfo DUNUSED,
   char name[80])
{
printf("Importfile name is %s\n",name);
strncpy(CONFIG_METHOD_Global.ImportFile,name,80);
}
/*-----------------------------------------------------------------*/
static void ConfigMethodImportAlloc(
   const CONFIG_MethodInfo *MethodInfo,
   CONFIG_ReportFunction RF,
   void *FA)
{

  
  FILE *impfile;
  StatusType status;
  int IPiv,inum,IT,NTarg;
  char dataline[82];
  char objname[80];
  double AllocCount,PercentDone,Theta;
  const CONFIG_AllocInfo *AllocInfo;      /* Address of allocation structure */
  const CONFIG_TargetInfo *TargetInfo;    /* Address of target information */
  const CONFIG_PivotInfo *PivotInfo;      /* Address of pivot information */
  CONFIG_METHOD_MatrixElement **Matrix;
  CONFIG_METHOD_TargetElement *TargetMatrix;

  status = STATUS__OK;
  impfile = fopen(CONFIG_METHOD_Global.ImportFile,"r");
  if (impfile == NULL) {
      char s[100];
      sprintf(s,"Error Opening Import File %s",
              CONFIG_METHOD_Global.ImportFile);
      MethodInfo->ErrRepRoutine(0,s);
      return;
  }
  AllocInfo = MethodInfo->AllocInfo;
  TargetInfo = MethodInfo->TargetInfo;
  PivotInfo = MethodInfo->PivotInfo;
  Matrix = CONFIG_METHOD_Global.Matrix;
  TargetMatrix = CONFIG_METHOD_Global.TargetMatrix;
  NTarg = CONFIG_METHOD_Global.NTargets;
  inum = 1;    
  AllocCount = 0;
  PercentDone = 0;
  while (fgets(dataline,80,impfile) != NULL){
    inum = sscanf(dataline,"%d %s",&IPiv,objname);
    IPiv -= 1;
    IT = 0;
    /*    DEBUG("Attempting Allocation:%d %s\n",IPiv,objname);*/
    while (strcmp(TargetInfo[IT].ObjName,objname) != 0 && IT < NTarg) IT++;
    if (IT == NTarg){
        char s[100];
        sprintf(s,"Error: Target %s not found in TargetInfo structure",
                objname);
        MethodInfo->ErrRepRoutine(0,s);
      return;
    }
    /* If we got here then IT should be the TargetInfo index corresponding
       to objname, so assuming the fibres are oriented correctly we shoudl
       be able to get away with a straight assign here....*/
    if (Matrix[IPiv][IT].PathStatus == CONFIG_METHOD_POSSIBLE) {
      Theta = Matrix[IPiv][IT].Theta;
      ConfigMethodSetAlloc(MethodInfo,IPiv,IT,Theta,&status);
      if (status == STATUS__OK) {
        if (RF != NULL) {
          if ((*RF)(FA,'A',"Allocating fibres",IPiv,(TargetInfo[IT].Type 
                                                     == 'F'),
                    TargetInfo[IT].OriginalIndex,TargetInfo[IT].X,
                    TargetInfo[IT].Y,Theta,PercentDone)) {
            return;
          }
        }
        AllocCount++;
        PercentDone = (AllocCount*100/CONFIG_METHOD_Global.NPiv);
      } else {
        /* DEBUG("Failed on:%d %s\n",IPiv,objname); */
        status = STATUS__OK;
      }
    } else {
      /*
      DEBUG("Failed Here:%d %s %d\n",IPiv,objname,Matrix[IPiv][IT].PathStatus);
      */
    }
  }
  fclose(impfile);
  return;
}

/*-----------------------------------------------------------------*/

/*
 *  The lone external routine in this module.  Allows the Config library
 *  to featch details of the method.
 */

extern void ConfigMethodDetails(CONFIG_MethodDetails *MethodDetails)
{
    MethodDetails->Version = CONFIG_VERSION;
    MethodDetails->RoutineCount = CONFIG_MTH_N_ROUTINES;
    MethodDetails->clientData = 0;  /* Not currently used */

    strncpy(MethodDetails->Name, "AAO Default Scheme", CONFIG_MTH_NAME_LEN-1);
    MethodDetails->Name[CONFIG_MTH_NAME_LEN-1] = '\0';

    MethodDetails->GlobalInit    = ConfigMethodGlobalInit;
    MethodDetails->Init          = ConfigMethodInit;
    MethodDetails->ReleaseMem    = ConfigMethodReleaseMem;
    MethodDetails->SetSky        = ConfigMethodSetSky;
    MethodDetails->Allocate      = ConfigMethodAllocate;
    MethodDetails->NumParams     = ConfigMethodNumberParams;
    MethodDetails->ParamDetails  = ConfigMethodParamDetails;
    MethodDetails->SetParam      = ConfigMethodSetParam;
    MethodDetails->GetParam      = ConfigMethodGetParam;
    MethodDetails->SetImportFile = ConfigMethodSetImportFile;
}

/*
         1         2         3         4         5         6         7         8
12345678901234567890123456789012345678901234567890123456789012345678901234567890
*/
