#!/bin/bash
#*******************************************************************************
# E.S.O. - VLT project
#
# "@(#) $Id: fposs_install.sh 290531 2016-11-24 19:26:36Z pbaksai $"
#
# who       when        what
# --------  ----------  --------------------------------------------------------
# pbaksai   2014/09/22  CCB-000579: Support for 64 bits and portability.
#                       - Corrected TCLTK_ROOT.
# pbaksai   2014/09/21  CCB-000579: Support for 64 bits and portability.
#                       - Corrected path handling.
# pbaksai   2014/07/28  CCB-000755: Support for 64 bits and portability.
#                       - Fixed advice for 32 bits libraries support.
#                       - Fixed message indentation level.
# pbaksai   2014/07/28  CCB-000755: Support for 64 bits and portability.
#                       - Fixed boolean expression.
# pbaksai   2014/07/24  CCB-000755: Support for 64 bits and portability.
#                       - Check errors with CheckStatus.
#                       - Check support for 32 bits libraries.
# pbaksai   2014/07/22  CCB-000755: Support for 64 bits and portability.
#                       - Minor corrections.
# pbaksai   2014/07/15  CCB-000755: Support for 64 bits and portability.
#                       - Added missing tcl includes.
# pbaksai   2014/07/14  CCB-000755: Support for 64 bits and portability.
#                       - Check status of all compilations.
#                       - Check and search for required libraries.
# pbaksai   2014/07/10  CCB-000755: Support for 64 bits and portability.
#                       - Created.
#
#*******************************************************************************
#   NAME
#       fposs_install.sh - Package installation script for FPOSS.
#
#   SYNOPSIS
#       fposs_install.sh [[-c|--compile]
#                         [-r|--required]
#                         [-u|--uninstall]
#                         [-l|--log[=<logpath>]]
#                         [-d|--debug]
#                         [--help]
#                         [--Usage]
#                         [--License]
#                         [--Version]
#                         --]
#                        [<path>]
#
#
#   DESCRIPTION
#       This script installs FPOSS package distribution in user's environment.
#
#   PARAMETERS
#        -c|--compile     : Recompiles all package components in user's
#                           environment.
#        -r|--required    : Installs pre-compiled required packages:
#                           Tcl/Tk
#        -u|--uninstall   : Removes FPOSS package installation.
#        -l|--log         : Enables logging mode.
#        -d|--debug       : Enables debugging mode.
#       --help            : Displays a help screen explaining the syntax.
#       --Usage           : Displays a help screen explaining the interface.
#       --License         : Displays copyright and license information.
#       --Version         : Displays program version.
#
#   ARGUMENTS
#       ?<logpath>?       : Path to user defined log file.
#                           Default: ""
#       ?<path>?          : Path to the installation directory.
#                           Default: current directory.
#
#   INPUTS
#
#   OUTPUTS
#       <stdout>
#           <exec log>    : Log of execution.
#       <file>
#           <logfile>     : <$PWD>/fposs_install.log | <logpath>
#           <script>      : <path>/fposs<OS><bits>.uninstall
#           <package>     : <path>/fposs<OS><bits>.tar.gz
#
#   FILES
#       bash_library.sh
#
#   ENVIRONMENT
#       USER
#       HOST
#       HOME
#       SHELL_LOG
#       SHELL_DEBUG
#       VLTROOT
#       INTROOT
#
#   RETURN VALUES
#       0: All OK.
#       1: Cancelled due to insufficient conditions.
#       2: Aborted due to errors.
#       3: Rejected due to incorrect arguments.
#       4: Ended with information display only (help, usage, license, version).
#
#   CAUTIONS
#       None
#
#   EXAMPLES
#       None
#
#   SEE ALSO
#       None
#
#   BUGS
#       None
#
#-------------------------------------------------------------------------------
#        1         2         3         4         5         6         7         8
#23456789 123456789 123456789 123456789 123456789 123456789 123456789 1234567890
#
# Include BASH library
#
bashlib=`dirname $0`/bash_library.sh
if [ -f "$bashlib" ]; then
    source $bashlib 2>&-
    status=$?
else
    echo "ERROR: $bashlib not found."
    exit 1
fi
if [ $status -ne 0 ]; then
    echo "ERROR: Code $status sourcing $bashlib."
    exit 1
fi

#################################
# Standard Library Overload BEGIN
#
#
# Standard Library Overload END
###############################

###############################
# Standard Initialization BEGIN
#
#
# Standard Initialization END
#############################

#####################
# Local Library BEGIN
#
#************#
# Parameters #
#************#
#

#*************#
# Environment #
#*************#
#

#
# SafeLink: Makes a symbolic link from source to destination with backup.
#
SafeLink() {
    # Parameters:
    # 1: Source path
    # 2: Destination path
    # Globals:
    # - PACK_NAME: Base name of the package
    # - upath: Path to uninstaller script
    # Assigns:    nothing
    # Returns:
    # 0: All OK
    # 1: Error linking source to destination
    local _source=$1
    local _dest=$2
    local _suffix=fposs
    Debug "Source     : >$_source<"
    Debug "Destination: >$_dest<"
    if [ -z "$_source" -o -z "$_dest" ]; then
        Warning "could not link, source and/or destination are/is blank"
        Explain "Source     : >$_source<"
        Explain "Destination: >$_dest<"
        return 1
    fi
    # Remove existing destination link to package item
    # so that it does not get backed up again.
    if [ -h $_dest ]; then
        if [ "`ls -laA$ls_nobackup $_dest 2>&- | grep $PACK_NAME`" != "" ]; then
            rm -f $_dest 2>&-
        fi
    fi
    # If a backup exists, move it to original destination 
    if [ -h ${_dest}.${_suffix} ]; then
        mv -f ${_dest}.${_suffix} $_dest 2>&-
    fi
    # If destination exists, make a backup with ._suffix suffix
    if [ -h $_dest -o -f $_dest -o -d $_dest ]; then
        mv -f $_dest ${_dest}.${_suffix} 2>&-
    fi
    # Link destination if exists
    Debug "ln -f -s $_source $_dest"
    # ln --suffix=.${_suffix} -b -f -s $_source $_dest 2>&-
    ln -f -s $_source $_dest 2>&-
    if [ $? -ne 0 ]; then
        Warning "Error: ln -f -s $_source $_dest failed"
        Explain "Symbolic link from $_source to $_dest was not established."
    fi
    # Register in uninstaller macro
    if [ -f $upath ]; then
        # Remove link installed by package
        Debug "echo rm -f $_dest 2>&- >> $upath"
        echo "rm -f $_dest 2>&-" >> $upath
        # Restore backup made by ln -b above
        Debug "echo mv -f ${_dest}.${_suffix} $_dest 2>&- >> $upath"
        echo "mv -f ${_dest}.${_suffix} $_dest 2>&-" >> $upath
    fi
    return 0
}
#
# Local Library END
###################

############################
# Local Initialization BEGIN
#
# Local constants
#
HELP_LINE="\
 [[-c|--compile]$LIFS\
  [-r|--required]$LIFS\
  [-u|--uninstall]$LIFS\
  [-l|--log[=<logpath>]] [-d|--debug]$LIFS\
  [--help] [--Usage]$LIFS\
  [--License] [--Version]$LIFS\
  --]$LIFS\
 [<path>] $LIFS\
          "

USAGE_LINE="\
 [[-c|--compile] $LIFS\
  [-r|--required] $LIFS\
  [-u|--uninstall] $LIFS\
  [-l|--log[=<logpath>]] $LIFS\
  [-d|--debug] $LIFS\
  [--help] $LIFS\
  [--Usage] $LIFS\
  [--License] $LIFS\
  [--Version] $LIFS\
  --] $LIFS\
 [<path>] $LIFS\
 where: $LIFS\
   -c or --compile: recompiles FPOSS executables $LIFS\
   -r or --required: installs pre-compiled packages: Tcl/Tk $LIFS\
   -u or --uninstall: removes FPOSS package $LIFS\
   -l or --log: enables log to file $LIFS\
   ?logpath?: path to user specified log file $LIFS\
              Default is $PWD/$progname $LIFS\
   -d or --debug: enables debugging mode $LIFS\
  --help: displays command line syntax $LIFS\
  --Usage: displays this Usage Screen $LIFS\
  --License: displays copyright and license $LIFS\
  --Version: displays the program version $LIFS\
  --: marks end of options, starts positional arguments $LIFS\
   ?path?: path to the installation directory $LIFS\
           Default is current directory $LIFS\
          "

LICENSE_LINE="\
  Copyright (C) $(ExpandRange 2014..$yearstamp)$LIFS\
  Pedro Baksai (pbaksai@eso.org)$LIFS\
  $LIFS\
  This program is free software: you can redistribute$LIFS\
  it and/or modify it under the terms of the$LIFS\
  GNU General Public License as published by$LIFS\
  the Free Software Foundation, either version 3 of$LIFS\
  the License, or (at your option) any later version.$LIFS\
  $LIFS\
  This program is distributed in the hope that it$LIFS\
  will be useful, but WITHOUT ANY WARRANTY,$LIFS\
  without even the implied warranty of$LIFS\
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.$LIFS\
  See the GNU General Public License for more details.$LIFS\
  $LIFS\
  You should have received a copy of the$LIFS\
  GNU General Public License along with this program.$LIFS\
  If not, see <http://www.gnu.org/licenses/gpl.html>.$LIFS\
             "

#
# Local variables
#
PERIOD=99
INSTRUMENT=FLAMES
PACK_NAME=fpossLIN64
INSUSER=.fposs
INSROOT=$HOME
PREP_DIR=$INSROOT/$INSUSER/PREP
LOG_DIR=$INSROOT/$INSUSER/LOG
SET_DIR=$INSROOT/$INSUSER/SET
UNPACK_DIR=`pwd`
INSTALL_DIR=""
cflag=0
rflag=0
uflag=0
ask=0

#
# Local Initialization END
##########################

################
# MAIN PROGRAM #
################

#########################
# Standard Prologue BEGIN
#
StartLog

# Receive options and parameters
#
options=$(GetOpt -q -o crul::d? -a --longoptions compile,\
                                   --longoptions required,\
                                   --longoptions uninstall,\
                                   --longoptions log::,\
                                   --longoptions debug,\
                                   --longoptions help,\
                                   --longoptions Usage,\
                                   --longoptions License,\
                                   --longoptions Version \
                                   -n $progname -- "$@")
status=$?

# Processes valid options
eval set -- "$options"
while [ $# -gt 0 ]; do
    case $1 in
        -c | --compile)
            Debug "Processing option $1 without argument."
            cflag=1
            ;;
        -r | --required)
            Debug "Processing option $1 without argument."
            rflag=1
            ;;
        -u | --uninstall)
            Debug "Processing option $1 without argument."
            uflag=1
            ;;
        -l | -log | --log)
            Debug "Processing option $1 with optional argument >$2<"
            lflag=1
            StartLog "$2"
            shift
            ;;
        -d | -debug | --debug)
            dflag=1
            Debug "Turned ON."
            Debug "Options received: >$options<"
            ;;
        --help | -\?)
            Debug "Processing option $1 without argument."
            Help Syntax "$HELP_LINE"
            Exit $EC_INFO
            ;;
        --Usage)
            Debug "Processing option $1 without argument."
            Help Usage "$USAGE_LINE"
            Exit $EC_INFO
            ;;
        --License)
            Debug "Processing option $1 without argument."
            Help License "$LICENSE_LINE"
            Exit $EC_INFO
            ;;
        --Version)
            Debug "Processing option $1 without argument."
            Help Version "$(Version)"
            Exit $EC_INFO
            ;;
        (--)
            Debug "End of options detected."
            shift
            break
            ;;
        (--*)
            Error "Invalid long option $1"
            eflag=$EC_REJECT
            ;;
        (-?)
            Error "Invalid short option $1"
            eflag=$EC_REJECT
            ;;
        (*)
            Debug "Argument $1 detected."
            break
            ;;
    esac
    shift
done

# Catches invalid options
if [ $status -ne 0 ]; then
    Warning "Code $status: Unrecognized or invalid option(s) ignored"
fi

# Processes valid non-option arguments
shift $((OPTIND-1))
case $# in
    (0)
        # Use default INSTALL_DIR
        INSTALL_DIR=$UNPACK_DIR
        ask=1
        ;;
    (*)
        INSTALL_DIR=$1
        GREP=`echo $INSTALL_DIR | grep "^/"`
        if [ $? -ne 0 ]; then
            INSTALL_DIR=${PWD}/${INSTALL_DIR}
        fi
        if [ ! -d $INSTALL_DIR ]; then
            mkdir -p $INSTALL_DIR 2>/dev/null
            if [ $? -ne 0 ]; then
                Tell "Cannot create this directory!"
                Tell "Please choose another one..."
                INSTALL_DIR=$UNPACK_DIR
                ask=1
            else
                Tell "Created directory $INSTALL_DIR\n"
            fi
        elif [ ! -w $INSTALL_DIR ]; then
            Tell "You do not appear to have write access to"
            Tell "$INSTALL_DIR."
            Tell "Please choose another directory for installation."
            INSTALL_DIR=$UNPACK_DIR
            ask=1
        fi
        shift
        ;;
esac

# Catches remaining non-option arguments
while [ $# -gt 0 ]; do
    case $1 in
        (*)
            Warning "invalid argument $1 ignored"
            ;;
    esac
    shift
done

#
# Validate argument values
#

#
# Validate environment variables
#

#
# Exit if error
#
if [ $eflag -ne 0 ]; then
    HelpAd
    Exit $eflag
fi

#
# Set debug level
#
dflag=$(( dflag || dbg ))

#
# Set log level
#
lflag=$(( lflag || log ))

#
# Standard Prologue END
#######################

######################
# Main Algorithm BEGIN
#
Frame
Banner "BEGIN  $progname by ${user}@${ws} on ${datestamp}T${timestamp}"
Frame

# Check working directory
if [ ! -f "$progname" ]; then
    Error   "Please run this script from inside the package directory."
    Explain "cd /some/path/$PACK_NAME/"
    Exit $EC_CANCEL
fi

# Choice of installation directory
if [ $ask -ne 0 ]; then
    # If it is asking then INSTALL_DIR=UNPACK_DIR
    Doing "Choose Installation Directory:"
    Tell  "The default is the current directory."
fi
while [ $ask -eq 1 ]; do
    Tell "Please enter the installation directory now,"
    Tell "or hit ENTER to accept default"
    echo
     if [ $uflag -ne 0 ]; then
        answer="$(GetAnswer 'Un-Install from here? ' \
                            $INSTALL_DIR \
                            'Type the full path to the installation directory' \
                            0\
                 )"
    else
        answer="$(GetAnswer 'Install here? ' \
                            $INSTALL_DIR \
                            'Type the full path to the installation directory' \
                            0\
                 )"
    fi
    if [ -z "$answer" ]; then
        ask=2
    else
        absolut=`echo $answer | grep "^/"`
        if [ $? -ne 0 ]; then
            answer=${PWD}/$answer
        fi
        if [ ! -d "$answer" ]; then
            if [ $uflag -eq 0 ]; then
                makedir="$(GetAnswer 'Do you want to create '$answer' ?' \
                                     'Y' \
                                     'Answer Y(es) or N(o)' \
                                     0 \
                          )"
                if [ -z "$makedir" ]; then
                    makedir=Y
                fi
                if [ "$makedir" = "Y" ]; then
                    mkdir -p "$answer" 2>/dev/null
                    if [ $? -ne 0 ]; then
                        Tell "Cannot create this directory! Please choose another one..."
                    else
                        INSTALL_DIR="$answer"
                        ask=3
                        Tell "Created directory $INSTALL_DIR"
                    fi
                fi
            else
                Tell "$PACK_NAME package was not installed there! Please choose another one..."
            fi
        elif [ ! -w "$answer" ]; then
            Tell "You do not appear to have write access to"
            Tell "$answer"
            Tell "Please choose another installation directory."
        else
            INSTALL_DIR="$answer"
            ask=4
            Tell "OK with $INSTALL_DIR"
        fi
    fi
done
if [ $ask -ne 0 ]; then
    Done "Choosing Installation Directory."
    echo " "
fi

if [ $uflag -ne 0 ]; then
    Doing "Un-Installing FPOSS package from $INSTALL_DIR"
    if [ -d ${INSTALL_DIR}/${PACK_NAME} ]; then
        cd ${INSTALL_DIR}/${PACK_NAME}
    else
        Error   "$PACK_NAME package was not installed there."
        Explain "${INSTALL_DIR}/${PACK_NAME} not found."
        Exit $EC_CANCEL
    fi
    if [ -f ${PACK_NAME}.uninstall ]; then
        if [ $dflag -ne 0 ]; then
            cat ${PACK_NAME}.uninstall
        fi
        # Execute uninstaller macro
        # Remove package directory
        # Remove package compressed files and installer
        # Leave only files distributed to users, if any
        cmd="
            chmod 755 ${PACK_NAME}.uninstall 2>&- ;
            ./${PACK_NAME}.uninstall 2>&- ;
            mv -f ${PACK_NAME}.uninstall ${PACK_NAME}.uninstall.done 2>&- ;
            cd ../. 2>&- ;
            rm -rf ${INSTALL_DIR}/${PACK_NAME} 2>&- ;
            rm -f ${INSTALL_DIR}/${PACK_NAME}.tar.gz 2>&- ;
            rm -f ${UNPACK_DIR}/${progname} 2>&- ;
            rm -f ${UNPACK_DIR}/bash_library.sh 2>&- ;
            rm -f ${UNPACK_DIR}/${PACK_NAME}.tar.gz 2>&- ;
            rm -f /tmp/*_compilation_warnings.log 2>&- ;
        "
        Waitfor "uninstalling ${PACK_NAME}..." "${cmd}" 1
        status=$?
        CheckStatus $status $status $LINENO $EC_END X "" 1
    else
        Error   "package is just unpacked, not installed"
        Explain "${PACK_NAME}.uninstall not found in"
        Explain "${INSTALL_DIR}/${PACK_NAME}"
        Explain "First install the package with:"
        Explain "~$> $progname $INSTALL_DIR"
        Exit $EC_CANCEL
    fi
    Done "Un-Installing FPOSS package."
    Exit $EC_END
fi

Frame
Banner "       ** Welcome to the FPOSS installation script **"
Frame
Banner "PLEASE NOTE: This package is to be used for period $PERIOD"
Banner "              of observations for the ESO telescopes."
Banner "              Current date $datestamp belongs to period $(Period)."
Filler
Banner "The FPOSS tool in this package is specific to programs for"
Banner "$INSTRUMENT."
Frame

# Check package components
Doing "Checking package is present..."
if [ ! -f "${PACK_NAME}.tar.gz" ]; then
    Error   "Package ${PACK_NAME}.tar.gz not found"
    Explain "in $UNPACK_DIR"
    Explain "Make sure you have unpacked FPOSS"
    Explain "distribution in current directory."
    Explain "Please execute gzip -d -c ${PACK_NAME}*.tar.gz | tar xf - "
    Explain "and then ./$progname"
    Exit $EC_CANCEL
else
    Tell "Package found here." 1
    os=`echo "${PACK_NAME}" | grep "${OSTAG}"`
    if [ $? -ne 0 ]; then
            Warning "Package ${PACK_NAME}.tar.gz does not seem to support this operating system"
            Explain "This is a $UNAME ($OSTAG) system."
    fi
fi
Done "Checking package is present."
echo " "

if [ $cflag -ne 0 ]; then
    Doing "Checking if C and C++ compilers are installed..."
    CC_EXEC=gcc
    CPP_EXEC=g++
    CCv=`$CC_EXEC -v  2>&1 | grep "$CC_EXEC version" | awk '{printf("%s",$3)}'`
    if [ -z "$CCv" ]; then
        Error "C compiler not found."
        Explain "Cannot re-compile modules."
        Exit $EC_ABORT
    else
        CC_PATH=`wich $CC_EXEC 2>/dev/null`
        Tell "C compiler version $CCv found in:"
        Tell "$CC_PATH" 1
        CPPv=`$CPP_EXEC -v  2>&1 | grep "$CC_EXEC version" | awk '{printf("%s",$3)}'`
        if [ -z "$CPPv" ]; then
            Error "C++ compiler not found."
            Explain "Cannot re-compile modules."
            Exit $EC_ABORT
        else
            CPP_PATH=`wich $CPP_EXEC 2>/dev/null`
            Tell "C++ compiler version $CPPv found in:"
            Tell "$CPP_PATH" 1
        fi
    fi
    Done "Checking C and C++ compilers."
    echo " "
fi

if [ $rflag -eq 0 ]; then
    Doing "Checking TCL interpreter is available..."
    TCL_EXEC=tclsh
    TCLv=$(echo "puts [info tclversion]" | $TCL_EXEC 2>/dev/null)
    if [ -z "$TCLv" ]; then
        Error "Tcl interpreter not found."
        Explain "Please run the installation again with -r option"
        Explain "to install Tcl/Tk from this package."
        eflag=$EC_ABORT
    else
        TCL_PATH=`which $TCL_EXEC 2>/dev/null`
        TCLTK_ROOT=$(dirname $(which $TCL_EXEC 2>/dev/null) 2>/dev/null)/..
        tclroot=${TCLTK_ROOT}/lib/tcl[0-9].*
        Tell "Tcl interpreter version $TCLv found in:"
        Tell "$TCL_PATH" 1
    fi
    Done "Checking TCL interpreter."
    echo " " 
else
    TCLTK_ROOT=${INSTALL_DIR}/${PACK_NAME}/${BITS}b/ext
    tclroot=${TCLTK_ROOT}/lib/tcl
fi
test $eflag -eq 0 || Exit $eflag

Doing "Installing FPOSS package in $INSTALL_DIR"
if [ -f "${UNPACK_DIR}/${PACK_NAME}.tar.gz" ]; then
    Doing "Unpacking ${UNPACK_DIR}/${PACK_NAME}.tar.gz" 1
    Tell  "to ${INSTALL_DIR}" 1
    cmd="
         rm -rf ${INSTALL_DIR}/${PACK_NAME} ;
         rm -rf ${INSTALL_DIR}/${PACK_NAME}.uninstall.done ;
         cd ${INSTALL_DIR} ;
         gzip -d -c ${UNPACK_DIR}/${PACK_NAME}.tar.gz | $tar xf - > /dev/null ;
    "
    Waitfor "uncompressing ${PACK_NAME}..." "${cmd}" 1
    status=$?
    CheckStatus $status $status $LINENO $EC_ABORT X "" 1
    Done "Unpacking." 1
    echo " "
    test $eflag -eq 0 || Exit $eflag
    Doing "Checking that package supports host architecture..." 1
    if [ ! -d ${INSTALL_DIR}/${PACK_NAME}/${BITS}b ]; then
        Error   "Package ${PACK_NAME}.tar.gz does not support this architecture."
        Explain "This is a ${BITS} bits system."
        Exit $EC_ABORT
    fi
    Done "Checking architecture." 1
    echo " "
else
    Error   "Package ${PACK_NAME}.tar.gz not found"
    Explain "in $UNPACK_DIR"
    Exit $EC_ABORT
fi

if [ "X$OSTAG" == "XLIN" -a "X$BITS" == "X64" ]; then
    Doing "Checking support for 32 bits libraries..." 1
    LD_LIB=/lib/ld-linux.so
    if [ -f ${LD_LIB}* ]; then 
        Tell "32 bits libraries are supported." 1
    else 
        bits=$(GetELF "${INSTALL_DIR}/${PACK_NAME}/${BITS}b/lib/configure")
        if [ X"$bits" = "X32" ]; then
            Error "There is no support for 32 bits libraries and this package needs it."
            Explain "You need to install glibc.i686 package in your system."
            Exit $EC_ABORT
        fi
    fi
    Done "Checking support for 32 bits libraries." 1
    echo " "
fi

# Create uninstaller macro
upath=${INSTALL_DIR}/${PACK_NAME}/${PACK_NAME}.uninstall
echo "# FPOSS Un-Installer created ${datestamp}T${timestamp}" > $upath
echo "cd ${INSTALL_DIR}/${PACK_NAME}" >> $upath

# Compile FPOSS tool
if [ $cflag -ne 0 ]; then
    cd ${INSTALL_DIR}/${PACK_NAME}/src
    Doing "Compiling fposs module..." 1
    Doing "Uncompressing fposs source code..." 2
    cmd="gzip -d -c fposs-*.tar.gz | $tar xf - > /dev/null 2>&1 ;"
    Waitfor "uncompressing fposs source..." "${cmd}" 2
    status=$?
    CheckStatus $status $status $LINENO $EC_ABORT X "" 2
    Done "Uncompressing fposs source code." 2
    echo " "
    Doing "Uncompressing pafchk source code..." 2
    cmd="gzip -d -c pafchk-*.tar.gz | $tar xf - > /dev/null 2>&1 ;"
    Waitfor "uncompressing pafchk source..." "${cmd}" 2
    status=$?
    CheckStatus $status $status $LINENO $EC_ABORT X "" 2
    Done "Uncompressing pafchk source code." 2
    echo " "
    Doing "Compiling fposs source code..." 2
    if [ -n "${VLTROOT}" ]; then
        makefile=Makefile
    else
        makefile=Makefile
    fi
    clf="/tmp/fposs_compilation_warnings.log "
    csf="/tmp/fposs_make_status.txt"
    cmd="
        cd ${INSTALL_DIR}/${PACK_NAME}/src/fposs/src ;
        rm -f ../tcltk/[il]* 2>&- ;
        ln -s ${INSTALL_DIR}/${PACK_NAME}/include ../tcltk/include ;
        ln -s ${INSTALL_DIR}/${PACK_NAME}/${BITS}b/ext/lib ../tcltk/lib ;
        ln -s /usr/lib/libX11.so.? ../tcltk/lib/libX11.so ;
        export TCLTK_LIB=${INSTALL_DIR}/${PACK_NAME}/${BITS}b/ext/lib ;
        export TCLTK_VERSION=`cat ${INSTALL_DIR}/${PACK_NAME}/${BITS}b/ext/lib/tcl/TCL_VERSION_*`
        make -f $makefile all > $clf ;
        echo \$? > $csf ;
    "
    Waitfor "fposs compilation..." "${cmd}" 2 "$clf"
    status1=$?
    status2=`cat $csf`
    status=$((status1 || status2))
    CheckStatus $status ${status1}.$status2 $LINENO $EC_ABORT I "Look at $clf for causes." 2
    Done "Compiling fposs source code." 2
    echo " "
    Doing "Installing fposs binaries, configurations, and libraries..." 2
    # Copy binaries to bin
    for b in `ls -1A$ls_nobackup $ls_ignore ../bin`; do
        rm -f ${INSTALL_DIR}/${PACK_NAME}/${BITS}b/bin/$b
        cp -pf ${INSTALL_DIR}/${PACK_NAME}/src/fposs/bin/$b ${INSTALL_DIR}/${PACK_NAME}/${BITS}b/bin/$b > /dev/null 2>&1
        status=$?
        if [ $status -ne 0 ]; then
                Warning "Could not cp -pf ${INSTALL_DIR}/${PACK_NAME}/src/fposs/bin/$b ${INSTALL_DIR}/${PACK_NAME}/${BITS}b/bin/$b"
        fi
    done
    # Copy library to lib
    for l in configure configure.tcl mimic.tcl procs.tcl util.tcl; do
        rm -rf ${INSTALL_DIR}/${PACK_NAME}/${BITS}b/lib/$l
        cp -rp ${INSTALL_DIR}/${PACK_NAME}/src/fposs/lib/$l ${INSTALL_DIR}/${PACK_NAME}/${BITS}b/lib/. > /dev/null 2>&1
        status=$?
        if [ $status -ne 0 ]; then
                Warning "Could not cp -rp ${INSTALL_DIR}/${PACK_NAME}/src/fposs/lib/$l ${INSTALL_DIR}/${PACK_NAME}/${BITS}b/lib/."
        fi
    done
    # Copy configuration files to config
    for c in `ls -1A$ls_nobackup $ls_ignore ../config`; do
        rm -f ${INSTALL_DIR}/${PACK_NAME}/config/$c
        cp -pf ${INSTALL_DIR}/${PACK_NAME}/src/fposs/config/$c ${INSTALL_DIR}/${PACK_NAME}/config/. > /dev/null 2>&1
        status=$?
        if [ $status -ne 0 ]; then
                Warning "Could not cp -pf ${INSTALL_DIR}/${PACK_NAME}/src/fposs/config/$c ${INSTALL_DIR}/${PACK_NAME}/config/."
        fi
    done
    Done "Installing fposs binaries, configurations, and libraries." 2
    echo " "
    # Clean up
    Doing "Cleaning up fposs..." 2
    make -f $makefile -s clean > /dev/null 2>&1
    Done "Cleaning up fposs." 2
    cd ${INSTALL_DIR}/${PACK_NAME}/src
    Done "Compiling fposs module." 1
    echo
fi

if [ $eflag -ne 0 ]; then
    Exit $eflag
fi

Doing "Installing scripts, libraries and binaries..." 1
if [ -n "${VLTROOT}" ]; then
    vltroot=${VLTROOT}
else
    vltroot=${INSTALL_DIR}/${PACK_NAME}
fi
if [ -n "${INTROOT}" ]; then
    # If it is installed in a VLTSW directory structure, replace some libraries
    # and binaries in INTROOT. Tcl/Tk should be installed.
    if [ -d ${INTROOT}/bin -a -w ${INTROOT}/bin ]; then
        binpath=${INTROOT}/bin
    fi
    if [ -d ${INTROOT}/lib -a -w ${INTROOT}/lib ]; then
        libpath=${INTROOT}/lib
    fi
fi
# Create local links to libraries, executables and scripts
if [ -z "$binpath" ]; then
    mkdir -p ${HOME}/binfposs
    binpath=${HOME}/binfposs
fi
if [ -z "$libpath" ]; then
    mkdir -p ${HOME}/libfposs
    libpath=${HOME}/libfposs
fi
cd ${INSTALL_DIR}/${PACK_NAME}
if [ -n "$binpath" -a -w $binpath ]; then
    Doing "Linking bin from $binpath..." 2
    if [ $rflag -ne 0 ]; then
        Tell "Adding links to external pre-compiled binaries..." 2
        for eb in `ls -1A$ls_nobackup ${PWD}/${BITS}b/ext/bin`; do
            cp -rfp ${PWD}/${BITS}b/ext/bin/$eb ${PWD}/${BITS}b/bin/. > /dev/null 2>&1
        done
    fi
    for b in `ls -1A$ls_nobackup ${PWD}/${BITS}b/bin`; do
        SafeLink ${PWD}/${BITS}b/bin/$b ${binpath}/$b
    done
    Done "Linking bin." 2
else
    Error   "could not define links to executables"
    Explain "Cannot complete install."
    Exit $EC_ABORT
fi
    
if [ -n "$libpath" -a -w $libpath ]; then
    Doing "Linking lib from $libpath..." 2
    if [ $rflag -ne 0 ]; then
        Tell "Adding links to external pre-compiled libraries..." 2
        for el in `ls -1A$ls_nobackup ${PWD}/${BITS}b/ext/lib`; do
            cp -rfp ${PWD}/${BITS}b/ext/lib/$el ${PWD}/${BITS}b/lib/. > /dev/null 2>&1
        done
        SafeLink ${PWD}/${BITS}b/lib/libtcl8*.so ${PWD}/${BITS}b/lib/libtcl.so
        SafeLink ${PWD}/${BITS}b/lib/libtk8*.so ${PWD}/${BITS}b/lib/libtk.so
    fi
    for l in `ls -1A$ls_nobackup ${PWD}/${BITS}b/lib`; do
        SafeLink ${PWD}/${BITS}b/lib/$l ${libpath}/$l
    done
    if [ ! -f ${PWD}/${BITS}b/ext/lib/libstdc* ]; then
        # Link all the libstdc++ libraries we can find in this system
        lpath_list=`whereis libstdc++.so.* | awk -F: '{printf("%s\n",$2)}'`
        for lp in $lpath_list; do
            ln=`basename $lp`
            SafeLink $lp ${libpath}/$ln
        done
        if [ -z "$lpath_list" ]; then
            Warning "Could not find any libstdc++ to link into your environment."
        fi
    fi
    SafeLink ${PWD}/${BITS}b/lib/configure ${PWD}/${BITS}b/lib/configure_expert
    Done "Linking lib." 2
else
    Error   "could not define links to libraries"
    Explain "Cannot complete install."
    Exit $EC_ABORT
fi

mkdir -p ${HOME}/bin
cp -pf ${UNPACK_DIR}/bash_library.sh ${PWD}/${BITS}b/bin/. > /dev/null 2>&1
SafeLink ${PWD}/${BITS}b/bin/bash_library.sh ${HOME}/bin/bash_library.sh
SafeLink ${PWD}/${BITS}b/bin/fposs.sh ${HOME}/bin/fposs.sh

# Add extra cleaning of bin and lib links to the uninstaller macro
if [ -n "$binpath" ]; then
    echo "rm -f \`ls -laA$ls_nobackup $binpath 2>&- | grep $PACK_NAME | awk '{printf(\"$binpath/%s \",\$9)}'\`" >> $upath
fi
if [ -n "$libpath" ]; then
    echo "rm -f \`ls -laA$ls_nobackup $libpath 2>&- | grep $PACK_NAME | awk '{printf(\"$libpath/%s \",\$9)}'\`" >> $upath
fi
Done "Installing scripts, libraries and binaries." 1

Doing "Updating launch script..." 1
# Jejeje, 27 backslashes to replace one / by \\\/ 
# (3 backslashes followed by a slash)
# giving a substitution like /my/full/path/to/fposs/ for
# \\\/my\\\/full\\\/path\\\/to\\\/fposs
# which is needed to pass this path string through sed again without the / being
# mistaken as pattern delimiters.
FPOSS_PATH=`echo ${INSTALL_DIR}/${PACK_NAME}/${BITS}b | sed s/[/]/\\\\\\\\\\\\\\\\\\\\\\\\\\\//g`
INS_PATH=`echo ${INSROOT} | sed s/[/]/\\\\\\\\\\\\\\\\\\\\\\\\\\\//g`
VLT_PATH=`echo ${vltroot} | sed s/[/]/\\\\\\\\\\\\\\\\\\\\\\\\\\\//g`
TCL_PATH=`echo ${tclroot} | sed s/[/]/\\\\\\\\\\\\\\\\\\\\\\\\\\\//g`
BIN_PATH=`echo ${binpath} | sed s/[/]/\\\\\\\\\\\\\\\\\\\\\\\\\\\//g`
LIB_PATH=`echo ${libpath} | sed s/[/]/\\\\\\\\\\\\\\\\\\\\\\\\\\\//g`
Debug "FPOSS_ROOT  = >$FPOSS_PATH<"
Debug "INSROOT     = >$INS_PATH<"
Debug "VLTROOT     = >$VLT_PATH<"
Debug "TCLPATH     = >$TCL_PATH<"
Debug "BINPATH     = >$BIN_PATH<"
Debug "LIBPATH     = >$LIB_PATH<"
Debug "INSUSER     = >$INSUSER<"
sed -e s/@@FPOSSBASEPATH@@/$FPOSS_PATH/ \
    -e s/@@INSPATH@@/$INS_PATH/ \
    -e s/@@VLTPATH@@/$VLT_PATH/ \
    -e s/@@TCLPATH@@/$TCL_PATH/ \
    -e s/@@BINPATH@@/$BIN_PATH/ \
    -e s/@@LIBPATH@@/$LIB_PATH/ \
    -e s/@@USER@@/$INSUSER/ \
    ${INSTALL_DIR}/${PACK_NAME}/src/fposs.sh.tpl > ${INSTALL_DIR}/${PACK_NAME}/${BITS}b/bin/fposs.sh

chmod 555 ${INSTALL_DIR}/${PACK_NAME}/${BITS}b/bin/fposs.sh
Done "Updating launch scripts." 1

Doing "Copying configuration files..." 1
# Create FPOSS user directories
if [ ! -d "${INSROOT}/${INSUSER}" ]; then
    mkdir -p ${INSROOT}/${INSUSER} > /dev/null 2>&1
fi
if [ ! -d "$PREP_DIR" ]; then
    mkdir -p $PREP_DIR > /dev/null 2>&1
    chmod -f 666 $PREP_DIR/* > /dev/null 2>&1
fi
if [ ! -d "$LOG_DIR" ]; then
    mkdir -p $LOG_DIR > /dev/null 2>&1
    chmod -f 666 $LOG_DIR/* > /dev/null 2>&1
fi
if [ ! -d "$SET_DIR" ]; then
    mkdir -p $SET_DIR > /dev/null 2>&1
    chmod -f 666 $SET_DIR/* > /dev/null 2>&1
fi

# Copy configuration and test files
cp -pf ${INSTALL_DIR}/${PACK_NAME}/config/fpossInsStatus.sds ${PREP_DIR}/. > /dev/null 2>&1
cp -pf ${INSTALL_DIR}/${PACK_NAME}/test/* ${PREP_DIR}/. > /dev/null 2>&1
cp -rpf ${INSTALL_DIR}/${PACK_NAME}/config ${PWD}/${BITS}b/. > /dev/null 2>&1

Done "Copying configuration." 1
Done "Installing FPOSS package."

Frame
Explain " "
Explain "Execute FPOSS tool with:"
Explain "~$> ~/bin/fposs.sh "
Explain "Have Fun With FPOSS! "
Explain " "
Frame
rm -f ${UNPACK_DIR}/${PACK_NAME}.tar.gz > /dev/null 2>&1
#
# Main Algorithm END
####################

#########################
# Standard Epilogue BEGIN
#
Exit $EC_END
#
# Standard Epilogue END
#######################
#
#___FPOSS___
