import XCTest

@testable import MiddleDrag

@unsafe final class DeviceMonitorTests: XCTestCase {

    // Note: DeviceMonitor uses a global variable (gDeviceMonitor) for C callback compatibility
    // This limits testing options since only one instance can be active at a time
    // Tests are run serially to avoid race conditions with the global state

    /// Instance under test - created fresh for each test
    private var monitor: DeviceMonitor!

    override func setUp() {
        super.setUp()
        // Create a fresh monitor for each test
        unsafe monitor = unsafe DeviceMonitor()
    }

    override func tearDown() {
        // Ensure monitor is stopped and cleaned up after each test
        unsafe monitor?.stop()
        unsafe monitor = nil
        // Small delay to ensure cleanup completes before next test
        // This helps prevent race conditions with the global gDeviceMonitor variable
        RunLoop.current.run(until: Date(timeIntervalSinceNow: 0.01))
        super.tearDown()
    }

    // MARK: - Start/Stop Tests

    func testStartDoesNotCrash() {
        // Note: In test environment without a trackpad, this may log warnings
        unsafe XCTAssertNoThrow(monitor.start())
        // stop() is called in tearDown
    }

    func testStopWithoutStartDoesNotCrash() {
        // Calling stop on a monitor that was never started should not crash
        // The monitor is created in setUp, so we just call stop directly
        unsafe XCTAssertNoThrow(monitor.stop())
    }

    func testDoubleStopDoesNotCrash() {
        // Should handle calling stop multiple times gracefully
        unsafe monitor.start()
        unsafe XCTAssertNoThrow(monitor.stop())
        unsafe XCTAssertNoThrow(monitor.stop())
    }

    func testStartStopStartDoesNotCrash() {
        // Should be able to restart the monitor
        unsafe XCTAssertNoThrow(monitor.start())
        unsafe XCTAssertNoThrow(monitor.stop())
        unsafe XCTAssertNoThrow(monitor.start())
    }

    // MARK: - Delegate Tests

    func testDelegateIsSetCorrectly() {
        let delegate = MockDeviceMonitorDelegate()
        unsafe monitor.delegate = delegate

        unsafe XCTAssertNotNil(monitor.delegate)
        unsafe XCTAssertTrue(monitor.delegate === delegate)
    }

    func testDelegateCanBeCleared() {
        let delegate = MockDeviceMonitorDelegate()
        unsafe monitor.delegate = delegate
        unsafe XCTAssertNotNil(monitor.delegate)

        unsafe monitor.delegate = nil
        unsafe XCTAssertNil(monitor.delegate)
    }

    func testDelegateIsWeakReference() {
        var delegate: MockDeviceMonitorDelegate? = MockDeviceMonitorDelegate()
        unsafe monitor.delegate = delegate

        unsafe XCTAssertNotNil(monitor.delegate)

        // Release the delegate
        delegate = nil

        // Delegate should be nil since it's a weak reference
        unsafe XCTAssertNil(monitor.delegate)
    }

    // MARK: - Multiple Instance Tests

    func testMultipleInstancesDoNotCrash() {
        // Create multiple monitors - only first should own global reference
        let monitor2 = unsafe DeviceMonitor()
        let monitor3 = unsafe DeviceMonitor()

        // Starting/stopping multiple instances should not crash
        unsafe XCTAssertNoThrow(monitor.start())
        unsafe XCTAssertNoThrow(monitor2.start())

        unsafe XCTAssertNoThrow(monitor.stop())
        unsafe XCTAssertNoThrow(monitor2.stop())
        // monitor3 never started, just deallocates
        _ = unsafe monitor3
    }

    func testSecondInstanceCanStartAfterFirstStops() {
        unsafe monitor.start()
        unsafe monitor.stop()

        let monitor2 = unsafe DeviceMonitor()
        unsafe XCTAssertNoThrow(monitor2.start())
        unsafe monitor2.stop()
    }

    // MARK: - Deinit/Cleanup Tests
    // Note: Tests that call start() inside autoreleasepool hang in CI (headless environment)
    // Testing deinit without start() is safe since it doesn't touch multitouch framework

    func testDeinitCleansUpGlobalReferenceWithoutStart() {
        // Test deinit cleanup WITHOUT calling start() - safe for CI
        // This exercises lines 74-75 in deinit that clear gDeviceMonitor
        weak var weakRef: DeviceMonitor?
        autoreleasepool {
            let localMonitor = unsafe DeviceMonitor()
            unsafe weakRef = unsafe localMonitor
            // Don't call start() - just let it deallocate
        }
        // The monitor should have been deallocated and global reference cleared
        unsafe XCTAssertNil(weakRef, "Monitor should be deallocated")
    }

    func testStopUnregistersAllDevices() {
        // This test ensures the new loop in stop() executes
        // In CI without devices, registeredDevices is empty but the code path is still exercised
        unsafe monitor.start()
        // Stop should unregister ALL registered devices (the new fix)
        unsafe XCTAssertNoThrow(monitor.stop())
    }

    func testMultipleStartStopCyclesDoNotLeak() {
        // Exercise multiple start/stop cycles to test the registeredDevices cleanup
        // Works in CI because start/stop are no-ops when no devices are found
        for _ in 0..<3 {
            unsafe monitor.start()
            unsafe monitor.stop()
        }
        // If no crashes/hangs, the device registration/unregistration is balanced
        XCTAssertTrue(true)
    }

    // MARK: - Race Condition Prevention Tests
    // These tests verify the fix for the CFRelease(NULL) crash caused by
    // concurrent device stop/release operations between main thread and
    // the MultitouchSupport framework's internal thread (mt_ThreadedMTEntry).

    func testStopIncludesDelayForFrameworkCleanup() {
        // This test verifies that stop() includes a delay to prevent race conditions.
        // The delay allows the framework's internal thread to complete cleanup.
        guard unsafe monitor.start() else {
            print("Skipping testStopIncludesDelayForFrameworkCleanup: No multitouch device found")
            return
        }

        let startTime = CACurrentMediaTime()
        unsafe monitor.stop()
        let elapsed = CACurrentMediaTime() - startTime

        // Use a conservative 10ms threshold to avoid flakiness while still
        // catching regressions where the delay is removed entirely.
        // The actual delay is DeviceMonitor.frameworkCleanupDelay (50ms).
        XCTAssertGreaterThanOrEqual(
            elapsed, 0.01,
            "stop() should include safety delay for framework cleanup")
    }

    func testRapidStartStopCyclesDoNotCrash() {
        // Simulates the race condition scenario where rapid restart cycles
        // could cause the framework's internal thread to access deallocated resources.
        // The fix adds delays to prevent this, so rapid cycles should be safe.
        for _ in 0..<5 {
            unsafe monitor.start()
            // Minimal delay to allow some internal state to build up
            Thread.sleep(forTimeInterval: 0.01)
            unsafe monitor.stop()
        }
        // Test passes if no crash occurred
    }

    func testStopSeparatesCallbackUnregistrationFromDeviceStop() {
        // This test exercises the code path where:
        // 1. Callbacks are unregistered first (MTUnregisterContactFrameCallback)
        // 2. A delay occurs (Thread.sleep)
        // 3. Devices are stopped (MTDeviceStop)
        // The separation prevents the framework's internal thread from calling
        // into our code while we're stopping devices.
        unsafe monitor.start()

        // Calling stop should complete without crash
        unsafe XCTAssertNoThrow(monitor.stop())

        // Verify monitor is properly stopped
        // (Start again should work cleanly if stop completed properly)
        unsafe XCTAssertNoThrow(monitor.start())
        unsafe monitor.stop()
    }

    func testConcurrentStopDoesNotCrash() {
        // Test that even if something tries to access the monitor during stop,
        // it doesn't crash. This simulates what happens when the framework's
        // internal thread is still processing while we stop.
        unsafe monitor.start()

        // Start stop on background thread
        let expectation = XCTestExpectation(description: "Concurrent operations complete")
        DispatchQueue.global().async {
            // Simulate framework internal thread activity
            Thread.sleep(forTimeInterval: 0.02)
            expectation.fulfill()
        }

        // Stop on main thread
        unsafe monitor.stop()

        unsafe wait(for: [expectation], timeout: 1.0)
    }

    func testFrameworkCleanupDelayConstantExists() {
        // Verify the delay constant is defined and has a reasonable value
        unsafe XCTAssertGreaterThan(
            DeviceMonitor.frameworkCleanupDelay, 0,
            "frameworkCleanupDelay should be positive")
        unsafe XCTAssertLessThanOrEqual(
            DeviceMonitor.frameworkCleanupDelay, 0.5,
            "frameworkCleanupDelay should not be excessive")
    }

    // MARK: - Delegate Callback Tests

    func testHandleContactCallsDelegate() {
        let delegate = MockDeviceMonitorDelegate()
        unsafe monitor.delegate = delegate

        // Create a mock touch data pointer
        // Note: In real usage, this would be a pointer to MTTouch array from the framework
        // For testing, we just need a non-nil pointer to exercise the code path
        let mockTouches = UnsafeMutableRawPointer.allocate(byteCount: 1, alignment: 1)
        defer { unsafe mockTouches.deallocate() }

        // Access handleContact through a workaround since it's fileprivate
        // We'll test it indirectly by verifying delegate is called when monitor receives touches
        // In a real scenario, this would be called from the C callback
        // For now, we verify the delegate is set up correctly
        unsafe XCTAssertNotNil(monitor.delegate)
        unsafe XCTAssertTrue(monitor.delegate === delegate)
    }

    func testStartReturnsFalseWhenNoDeviceFound() {
        // In CI/test environments without a trackpad, start() should return false
        // This tests the guard at line 145-150
        let result = unsafe monitor.start()
        // In test environment, this may return false if no device is found
        // The important thing is it doesn't crash and handles the case gracefully
        _ = result  // Result may be true or false depending on test environment
    }

    func testStartHandlesNilDeviceList() {
        // Test that start() handles the case where MTDeviceCreateList() returns nil
        // This exercises the else branch at line 125-127
        // In test environment, this path may be taken
        unsafe XCTAssertNoThrow(monitor.start())
    }

    func testStartHandlesDefaultDeviceAlreadyRegistered() {
        // Test that start() handles the case where default device is already in registeredDevices
        // This exercises the else branch at line 140-142
        // Start twice to potentially register the same device
        unsafe monitor.start()
        // Second start should handle already-registered devices gracefully
        unsafe XCTAssertNoThrow(monitor.start())
        unsafe monitor.stop()
    }
}

// MARK: - Mock Delegate

/// Mock delegate for testing DeviceMonitor delegate callbacks
class MockDeviceMonitorDelegate: DeviceMonitorDelegate {
    var didReceiveTouchesCalled = false
    var receivedTouchCount: Int32 = 0
    var receivedTimestamp: Double = 0

    func deviceMonitor(
        _ monitor: DeviceMonitor,
        didReceiveTouches touches: UnsafeMutableRawPointer,
        count: Int32,
        timestamp: Double
    ) {
        didReceiveTouchesCalled = true
        receivedTouchCount = count
        receivedTimestamp = timestamp
    }
}
